package org.lsst.ccs.subsystem.shutter.statemachine;

import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.GoToProd;
import org.lsst.ccs.subsystem.shutter.plc.Reset;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The state in which the shutter accepts maintenance commands such as independent blade set
 * motions, other motorplatform commands and calibration. Thread-safe.
 * @author tether
 */
class Maint extends CompositeState<Maint, Enabled> {
    private static final Logger LOG = Logger.getLogger(Maint.class.getName());

    // PROTECTED by instance lock
    private Still stillState;
    private Calibrating calibratingState;
    private Centering1 centering1State;
    private Centering2 centering2State;
    private Moving movingState;
    // END PROTECTED

    /**
     * Saves the actions and the context for this composite state.
     * @param actions The action implementations to pass to substates.
     * @param context The super-context for this context.
     * @throws NullPointerException if either argument is null.
     */
    public Maint(final Actions actions, final Enabled context) {
        super(actions, context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    @Override
    public void init() {
        makeTransition(getStillState(), null);
    }

    /**
     * Gets the {@code Still} state for this context, creating it the first time through.
     * @return The {@code Still} state;
     */
    synchronized Still getStillState() {
        if (stillState == null) {stillState = new Still(this);}
        return stillState;
    }

    /**
     * Gets the {@code Calibrating} state for this context, creating it the first time through.
     * @return The {@code Calibrating} state;
     */
    synchronized Calibrating getCalibratingState() {
        if (calibratingState == null) {calibratingState = new Calibrating(this);}
        return calibratingState;
    }

    /**
     * Gets the {@code Centering1} state for this context, creating it the first time through.
     * @return The {@code Centering1} state;
     */
    synchronized Centering1 getCentering1State() {
        if (centering1State == null) {centering1State = new Centering1(this);}
        return centering1State;
    }

    /**
     * Gets the {@code Centering2} state for this context, creating it the first time through.
     * @return The {@code Centering2} state;
     */
    synchronized Centering2 getCentering2State() {
        if (centering2State == null) {centering2State = new Centering2(this);}
        return centering2State;
    }

    /**
     * Gets the {@code Moving} state for this context, creating it the first time through.
     * @return The {@code Moving} state;
     */
    synchronized Moving getMovingState() {
        if (movingState == null) {movingState = new Moving(this);}
        return movingState;
    }

    @Override
    public void error(final Channel<EventReply> chan, final Error err) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
         // Full state-exit and state-(re)entry semantics.
        getContext().makeTransition(getContext().getMaintState(),
                                    () -> {
                                        getContext().getActions().setPhysicalState(PhysicalState.OTHER);
                                        getContext().getActions().raisePLCAlert(err);
                                    });
    }

    @Override
    public void reset(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
        // Full state-exit and state-(re)entry semantics.
        getContext().makeTransition(getContext().getMaintState(),
                                    () -> {getContext().getActions().setPhysicalState(PhysicalState.OTHER);
                                           getContext().getActions().lowerMotionAlert();
                                           getContext().getActions().lowerPLCAlert();
                                           getContext().getActions().lowerWatchdogAlert();
                                           getContext().getActions().resetPLC();
                                           });
    }

    @Override
    public void gotoProd(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        final Enabled ctx = getContext();
        final String msg = ctx.getActions().shutterIsReady();
        if (msg == null) {
            chan.write(ACCEPTED);
            ctx.makeTransition(ctx.getProdState(), () -> ctx.getActions().relay(new GoToProd()));
        }
        else {
            chan.write(new PromptReply(msg));
        }
    }

}
