package org.lsst.ccs.subsystem.shutter.plc;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.fromDcDuration;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.fromDcInstant;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.getBoolean;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.putBoolean;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.toDcDuration;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.toDcInstant;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Contains a motion-done message sent from the PLC. Contains an internal
 * instance of {@code MotionDone}
 * @see MotionDone
 * @author tether
 */
public class MotionDonePLC extends MsgToCCS {
    
    final MotionDone motion;

    /** Constructs from scratch.
     *  @param sequence A message sequence number.
     *  @param motion A reference of {@code MotionDone} which this object will own.
     */
    public MotionDonePLC(final int sequence, final MotionDone motion) {
        super(sequence);
        this.motion = motion;
    }

    /**
     * Reads and converts the PLC form of the message.
     * @param data The PLC message data.
     */
    public MotionDonePLC(final ByteBuffer data) {
        super(data);
        final MotionDone.Builder builder =
            new MotionDone.Builder()
                .side(ShutterSide.fromAxis(Axis.fromAxisNum(data.getInt())))
                .startTime(fromDcInstant(data.getLong()))
                .startPosition(data.getDouble())
                .targetDuration(fromDcDuration(data.getLong()))
                .targetPosition(data.getDouble())
                .endPosition(data.getDouble())
                .actualDuration(fromDcDuration(data.getLong()))
                .hallTransitions(new ArrayList<>())
                .encoderSamples(new ArrayList<>());
        final int ntrans = data.getInt();
        final int maxtrans = data.getInt();
        final int nsamples = data.getInt();
        final int maxsamples = data.getInt();
        final int tranStart = data.position();

        // Gather the Hall transitions.
        for (int i = 0; i < ntrans; ++i) {
            builder.addHallTransition(decodeHallTransition(data));
        }
        
        // Skip the unused elements of the transition array.
        data.position(tranStart + maxtrans * HALL_TRANSITION_SIZE);
       
        // Gather the encoder samples.
        for (int i = 0; i < nsamples; ++i) {
            builder.addEncoderSample(decodeEncoderSample(data));
        }
        
        // Skip unused sample elements to avoid complaints about unused data
        // from unit tests.
        data.position(data.position() + (maxsamples - nsamples) * ENCODER_SAMPLE_SIZE);
        
        // Finally ...
        this.motion = builder.build();                
    }

    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        final Axis ax = motion.side() == PLUSX ? Axis.getPlusXSide() : Axis.getMinusXSide();
        data.putInt(ax.getPLCAxisNum());
        data.putLong(toDcInstant(motion.startTime()));
        data.putDouble(motion.startPosition());
        data.putLong(toDcDuration(motion.targetDuration()));
        data.putDouble(motion.targetPosition());
        data.putDouble(motion.endPosition());
        data.putLong(toDcDuration(motion.actualDuration()));
        final int ntrans = motion.hallTransitions().size();
        final int unused = 5;
        data.putInt(ntrans);
        data.putInt(ntrans + unused);
        final int nsamples = motion.encoderSamples().size();
        data.putInt(nsamples);
        data.putInt(nsamples + unused);
        for (final HallTransition trans: motion.hallTransitions()) {
            encodeHallTransition(data, trans);
        }
        // Skip space before the encoder samples, simulating unused array entries.
        data.position(data.position() + unused * HALL_TRANSITION_SIZE);

        for (final EncoderSample sample: motion.encoderSamples()) {
            encodeEncoderSample(data, sample);
        }
        
        // Unused encoder sample space.
        data.position(data.position() + unused * ENCODER_SAMPLE_SIZE);
    }
    
    private static HallTransition decodeHallTransition(final ByteBuffer data) {
        final int id = data.getInt();
        final CCSTimeStamp time = fromDcInstant(data.getLong());
        final double pos = data.getDouble();
        final boolean on = getBoolean(data);
        return new HallTransition(time, id, pos, on);
    }
    
    // Encodes and returns size in bytes.
    private static int encodeHallTransition(final ByteBuffer data, final HallTransition hall) {
        final int start = data.position();
        data.putInt(hall.getSensorId());
        data.putLong(toDcInstant(hall.getTime()));
        data.putDouble(hall.getPosition());
        putBoolean(data, hall.isOn());
        return data.position() - start;
    }
    
    private static EncoderSample decodeEncoderSample(final ByteBuffer data) {
        final CCSTimeStamp time = fromDcInstant(data.getLong());
        final double pos = data.getDouble();
        return new EncoderSample(time, pos);
    }
    
    private static int encodeEncoderSample(final ByteBuffer data, final EncoderSample sample) {
        final int start = data.position();
        data.putLong(toDcInstant(sample.getTime()));
        data.putDouble(sample.getPosition());
        return data.position() - start;
    }
    
    private static final int HALL_TRANSITION_SIZE;
    private static final int ENCODER_SAMPLE_SIZE;
    static {
        final ByteBuffer data = ByteBuffer.allocate(1000);
        final HallTransition hall = new HallTransition(CCSTimeStamp.currentTime(), 0, 0.0, false);
        HALL_TRANSITION_SIZE = encodeHallTransition(data, hall);
        final EncoderSample sample = new EncoderSample(CCSTimeStamp.currentTime(), 0.0);
        ENCODER_SAMPLE_SIZE = encodeEncoderSample(data, sample);
    }
    

    /**
     * Creates a string of the form 'MotionDonePLC{MsgToCCS={...} motion=MotionDone{...}}'
     * @return 
     */
    @Override
    public String toString() {
        return "MotionDonePLC{" + super.toString() + " motion=" + motion.toString() + '}';
    }
    
    /**
     * Returns the internal {@code MotionDone} instance.
     * @return The {@code MotionDone} instance.
     */
    public MotionDone getStatusBusMessage() {return motion;}

}
