package org.lsst.ccs.subsystem.shutter.plc;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import org.lsst.ccs.subsystem.shutter.common.PLCError;

/**
 * Holds an Error message sent from the PLC.
 * @author tether
 */
public class Error extends MsgToCCS {

    private final Set<PLCError> errors;

    /** Constructs from scratch.
     *  @param sequence  A message sequence number.
     *  @param errors The set of errors.
     */
    public Error(final int sequence, final Set<PLCError> errors) {
        super(sequence);
        this.errors = PLCError.copyOf(errors);
    }

    /**
     * Reads and converts the PLC form of the message.
     * @param data Holds the PLC message data.
     */
    public Error(final ByteBuffer data) {
        super(data);
        this.errors = PLCError.decodeMask(data.getInt());
    }

    /**
     * Puts the message into a data buffer.
     * @param data The buffer.
     */
    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        data.putInt(PLCError.encodeMask(errors));
    }

    /**
     * Gets the set of errors that this message is reporting.
     * @return The unmodifiable set.
     */
    public Set<PLCError> getErrors() {
        return Collections.unmodifiableSet(errors);
    }

    @Override
    public String toString() {
        final StringBuilder bld = new StringBuilder("Error{");
        bld.append(super.toString());
        bld.append(" errors={");
        // Use the natural ordering of the enumerators for the sake of comparisions.
        final List<String> lerr = new ArrayList<>(PLCError.values().length);
        for (final PLCError err: PLCError.values()) {
            if (errors.contains(err)) {lerr.add(err.toString());}
        }
        bld.append(String.join(", ", lerr));
        bld.append("}");
        bld.append("}");
        return bld.toString();
    }

}
