package org.lsst.ccs.subsystem.shutter;

import java.time.Duration;
import java.util.concurrent.Callable;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.CalibDone;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.Error;
import org.lsst.ccs.subsystem.shutter.plc.Ignored;
import org.lsst.ccs.subsystem.shutter.plc.MotionDonePLC;
import org.lsst.ccs.subsystem.shutter.statemachine.Actions;
import org.lsst.ccs.subsystem.shutter.statemachine.BlackHoleChannel;
import org.lsst.ccs.subsystem.shutter.statemachine.Channel;
import org.lsst.ccs.subsystem.shutter.statemachine.EventReply;
import org.lsst.ccs.subsystem.shutter.statemachine.Events;
import org.lsst.ccs.subsystem.shutter.statemachine.SynchronousChannel;
import org.lsst.ccs.subsystem.shutter.statemachine.TopContext;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.PeriodicTask;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Owns and operates the central shutter-control state machine. Also owns, registers
 * and updates a component state bundle based on classes {@code PhysicalState} and {@code SoftwareState}.
 * <p>
 * Subsystem components wishing to submit events to the state machine do so by calling one of
 * the event methods in this object. Each event is accompanied by a 
 * {@code Channel<EventReply>} which the state machine uses to send event accept/reject notices back to the
 * submitter.
 * @see PhysicalState
 * @see SoftwareState
 * @author tether
 */
public class StateMachine implements HasLifecycle, Events {
    private static final Logger LOG = Logger.getLogger(StateMachine.class.getName());

    /** A reference to the {@code Subsystem} component in the subsystem tree. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Subsystem subsys;

    /** A reference to the {@code Controller} component that communicates with the shutter controller. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Controller controller;

    /** A reference to the {@code Publisher} component that sends status bus messages. */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Publisher publish;

    /** A reference to the {@code Watchdog} component which checks whether we've recently heard from the
     *  shutter controller.
     */
    @LookupField(strategy=LookupField.Strategy.TREE)
    private volatile Watchdog wdog;

    @ConfigurationParameter(description="How long to delay restarting a crashed long-running task.", units="s")
    private volatile Duration taskRestartDelay = Duration.ofMillis(10);

    @ConfigurationParameter(description="Wait this long for the shutter to enter Still or Disabled after a reset.",
            units="s")
    private volatile Duration resetSyncTimeout = Duration.ofSeconds(10);

    // GUARDED by the constructor.
    /** Use to pass events to the internal state internalMachine. */
    private final Channel<Callable<Void>> eventChan;

    /** The scheduler for the tasks used to operate the central state machine. */
    private final Scheduler taskScheduler;
    // END GUARDED

    /** Reads events and feeds them to the internal state machine. */
    private volatile PeriodicTask eventTask;

    /** Implements the sync timeout for the {@code Synchronizing} state. */
    private volatile ScheduledFuture<?> syncTimeoutTask;

    /** The actual state machine implementation. */
    private volatile TopContext internalMachine;

    /** The Actions implementation to be used by the state machine. */
    private volatile Actions actions;

    // Need separate threads for the event task, the watchdog and timer tasks.
    private final static int THREAD_POOL_SIZE = 10;

    /**
     * Creates the state machine, event queue and scheduler. No tasks are started.
     */
    public StateMachine() {
        this.eventChan = new SynchronousChannel<>();
        this.taskScheduler = new Scheduler("StateMachine tasks", THREAD_POOL_SIZE);
        this.taskScheduler.setLogger(LOG);
        this.taskScheduler.setDefaultLogLevel(Level.SEVERE);
    }

    /**
     * Registers the shutter physical-state and software-state bundles and sets their initial values to
     * {@code OTHER} and {@code SYNCHRONIZING}, respectively.
     * @see PhysicalState
     */
    @Override
    public void init() {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.registerState(PhysicalState.class, "The physical state of the shutter", this);
        stateServ.updateAgentComponentState(this, PhysicalState.OTHER);
        stateServ.registerState(SoftwareState.class, "The state of the subsystem's central state machine", this);
        stateServ.updateAgentComponentState(this, SoftwareState.SYNCHRONIZING);
    }

    /**
     * Creates the actual state machine and starts the task that manages event submission. Decides
     * based on the CCS run mode whether the implementation of {@code Actions} used by the state
     * machine is for real or simulated hardware.
     */
    @Override
    public void postStart() {
        // The state machine can't be created in the constructor because it would allow "this"
        // to escape.
        if (RunMode.isSimulation()) {
            actions = new SimulatedActions(controller, this, subsys, publish, wdog);
            internalMachine = new TopContext(actions);
        }
        else {
            actions = new RealActions(controller, this, subsys, publish, wdog);
            internalMachine = new TopContext(actions);
        }
        internalMachine.init();
        eventTask =
            taskScheduler.scheduleWithFixedDelay(this::eventTaskBody,
                                                      0, taskRestartDelay.toMillis(),
                                                      TimeUnit.MILLISECONDS);
        // If we're in simulated mode then put the state machine into the Closed state.
        if (RunMode.isSimulation()) {
            try {
                final Channel<EventReply> replyChan = new SynchronousChannel<>();
                plcIsEnabled(replyChan);
                replyChan.read();
                gotoProd(replyChan);
                replyChan.read();
            } catch (InterruptedException ex) {
                LOG.log(Level.WARNING, null, ex);
            }
        }
    }

    /**
     * Gets the {@code Actions} implementation being used by the state machine.
     * @return The actions.
     */
    public Actions getActions() {return actions;}

    private void eventTaskBody() {
        LOG.info("The event task has started.");
        try {
            while (true) {
                LOG.fine("About to read an event.");
                final Callable<Void> event = eventChan.read();
                LOG.fine("Running an event.");
                event.call();
            }
        }
        catch (InterruptedException exc) {
            LOG.info("Normal stop of the event task.");
        }
        catch (Exception exc) {
            LOG.warning("Exception thrown in the internal state machine.", exc);
        }
    }

    @Override
    public void shutdown() {
        taskScheduler.shutdownNow();
    }


    ////////// Events //////////

    @Override
    public void contactLost(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write(
                () -> {internalMachine.contactLost(replyChan); return null;}
        );
    }

    @Override
    public void plcIsEnabled(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write(
            () -> {internalMachine.plcIsEnabled(replyChan); return null;}
        );
    }

    @Override
    public void plcIsDisabled(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.plcIsDisabled(replyChan); return null;} );
    }

    @Override
    public void resync(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write(() -> {internalMachine.resync(replyChan); return null;});
    }

    @Override
    public void syncTimeout(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.syncTimeout(replyChan); return null;} );
    }

    @Override
    public void enable(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.enable(replyChan); return null;} );
    }

    @Override
    public void disable(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.disable(replyChan); return null;} );
    }

    @Override
    public void motionDone(final Channel<EventReply> replyChan, final MotionDonePLC profileData) throws InterruptedException {
        eventChan.write( () -> {internalMachine.motionDone(replyChan, profileData); return null;} );
    }

    @Override
    public void calibrate(final Channel<EventReply> replyChan, final Calibrate calibParams) throws InterruptedException {
        eventChan.write( () -> {internalMachine.calibrate(replyChan, calibParams); return null;} );
    }

    @Override
    public void toggleSafetyCheck(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.toggleSafetyCheck(replyChan); return null;} );
    }

    @Override
    public void calibDone(final Channel<EventReply> replyChan, final CalibDone calibResults) throws InterruptedException {
        eventChan.write( () -> {internalMachine.calibDone(replyChan, calibResults); return null;} );
    }

    @Override
    public void error(final Channel<EventReply> replyChan, final Error err) throws InterruptedException {
        eventChan.write( () -> {internalMachine.error(replyChan, err); return null;} );
    }

    @Override
    public void reset(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.reset(replyChan); return null;} );
    }

    @Override
    public void takeExposure(final Channel<EventReply> replyChan, final Duration exposureTime) throws InterruptedException {
        eventChan.write( () -> {internalMachine.takeExposure(replyChan, exposureTime); return null;} );
    }

    @Override
    public void openShutter(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.openShutter(replyChan); return null;} );
    }

    @Override
    public void timer(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.timer(replyChan); return null;} );
    }

    @Override
    public void closeShutter(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.closeShutter(replyChan); return null;} );
    }

    @Override
    public void ignored(final Channel<EventReply> replyChan, final Ignored.Reason reason) throws InterruptedException {
        eventChan.write( () -> {internalMachine.ignored(replyChan, reason); return null;} );
    }

    @Override
    public void gotoProd(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.gotoProd(replyChan); return null;} );
    }

    @Override
    public void gotoCenter(final Channel<EventReply> replyChan) throws InterruptedException {
        eventChan.write( () -> {internalMachine.gotoCenter(replyChan); return null;} );
    }

    @Override
    public void moveAxisAbsolute(final Channel<EventReply> replyChan, final MoveAxisAbsolute req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.moveAxisAbsolute(replyChan, req); return null;} );
    }

    @Override
    public void moveAxisRelative(final Channel<EventReply> replyChan, final MoveAxisRelative req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.moveAxisRelative(replyChan, req); return null;} );
    }

    @Override
    public void clearAllFaults(final Channel<EventReply> replyChan, final ClearAllFaults req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.clearAllFaults(replyChan, req); return null;} );
    }

    @Override
    public void changeAxisEnable(final Channel<EventReply> replyChan, final ChangeAxisEnable req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.changeAxisEnable(replyChan, req); return null;} );
    }

    @Override
    public void changeBrakeState(final Channel<EventReply> replyChan, final Axis ax, final ChangeBrakeState.State newState) throws InterruptedException {
        eventChan.write( () -> {internalMachine.changeBrakeState(replyChan, ax, newState); return null;} );
    }

    @Override
    public void clearAxisFaults(final Channel<EventReply> replyChan, final ClearAxisFaults req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.clearAxisFaults(replyChan, req); return null;} );
    }

    @Override
    public void enableAllAxes(final Channel<EventReply> replyChan, final EnableAllAxes req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.enableAllAxes(replyChan, req); return null;} );
    }

    @Override
    public void disableAllAxes(final Channel<EventReply> replyChan, final DisableAllAxes req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.disableAllAxes(replyChan, req); return null;} );
    }

    @Override
    public void homeAxis(final Channel<EventReply> replyChan, final HomeAxis req) throws InterruptedException {
        eventChan.write( () -> {internalMachine.homeAxis(replyChan, req); return null;} );
    }

    ////////// Actions //////////

    void setPhysicalState(final PhysicalState newState) {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.updateAgentComponentState(this, newState);
    }

    void setSoftwareState(final SoftwareState newState) {
        final AgentStateService stateServ = subsys.getAgentService(AgentStateService.class);
        stateServ.updateAgentComponentState(this, newState);
    }

    synchronized void startSyncTimer() {
        // We don't bother testing for rejection of the syncTimeout() event, which would happen
        // if we were late in canceling the timer task.
        final Channel<EventReply> chan = new BlackHoleChannel<>();
        final Callable<Void> snippet = () -> {this.syncTimeout(chan); return null;};
        syncTimeoutTask = this.taskScheduler.schedule(
            snippet,
            resetSyncTimeout.toMillis(), TimeUnit.MILLISECONDS);
    }

    synchronized void cancelSyncTimer() {
        if (syncTimeoutTask != null) { // Might be null if contact was never made with the PLC.
            syncTimeoutTask.cancel(true);
        }
    }

}
