package org.lsst.ccs.subsystems.shutter.ptp;

import java.nio.LongBuffer;
import java.time.Duration;
import java.time.Instant;
import java.util.Arrays;
import java.util.stream.Collectors;
import org.lsst.ccs.drivers.ads.ADSDriver;
import org.lsst.ccs.drivers.ads.Notification;
import org.lsst.ccs.drivers.ads.VariableHandle;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Sets up ADS notification for changes to the variable MAIN.tinfo in the PLC program
 * running on the Beckhoff CX controller. The dat contains five 64-byte quantities:
 * <ol>
 * <li>dctime - The unsigned Distributed Clock (DC) time for the EtherCAT bus, represented as
 * the number of nanoseconds since Jan 1 2000 00:00:00. This is the common time established
 * for all EtherCAT modules which have internal clocks, including the EtherCAT master. The
 * PLC program gets this by calling F_GetActualDcTime64().</li>
 * <li>dctotcoff - The signed offset in nanoseconds of the TwinCAT (TC) time from the 
 * DC time. This is a process data item available from the EtherCAT master, which has to
 * be configured to provide it.</li>
 * <li><dctoextoff - The signed offset in nanoseconds of the PTP (external) clock
 * maintained by the EL6688 module from the DC time. Also a process data item
 * from the EtherCAT master.</li>
 * <li>internalstamp - The unsigned time in nanoseconds of the EL6688 EtherCAT clock. SHould be
 * the same as the DC time.</li>
 * <li><externalstamp - The unsigned time in nanoseconds of the EL6688 PTP clock. Should
 * be the TAI PTP time.</li>
 * </ol>
 * <p>
 * The main program sets up notification for updates to the "tinfo" variable in the MAIN program of
 * the currently running PLC program, which should be The PTP program of the CCS solution
 * in the CCS Electronics GitHub repository. The PLC program will update the structure
 * at tinfo once a second with the above data, which the Java program will write to
 * stdout. The format is CSV with the first line containing the column headings. All
 * the numbers are in nanoseconds. Each non-header line will start with the TAI time
 * of the system running this Java program followed by the PLC time info variables in the
 * order listed above.
 * @author tether
 */
public class PTPInfoReader {
    
    // We set our AMS NetID to this, which is the NetID of the route set up for
    // lsst-lion01 in the CX controller.
    private static final String LOCAL_AMS_NETID = "192.168.10.80.1.1";
    
    // The AMS address of the Beckhoff controller in Tether's office.
    private static final String REMOTE_AMS_NETID = "5.56.86.54.1.1";
    // The IP address of that controller.
    private static final String REMOTE_IP_ADDRESS = "192.168.10.40";
    
    // The name of the variable in the PLC MAIN program.
    private final static String VARIABLE_NAME = "tinfo";
    
    private final static int NUM_SAMPLES = 1000;
    
    private final ADSDriver driver;
    
    private final VariableHandle varHandle;
    
    PTPInfoReader() throws DriverException {
        driver = new ADSDriver(LOCAL_AMS_NETID);
        driver.open(REMOTE_AMS_NETID, REMOTE_IP_ADDRESS);
        varHandle = driver.getVariableHandle(VARIABLE_NAME);
        driver.requestNotifications(varHandle, Duration.ZERO, Duration.ZERO, true);
    }
    
    public void close() {
        driver.close();
    }
    
    public static class TimeInfo {
        public final long[] info = new long[5];

        public TimeInfo(final LongBuffer buf) {
            buf.get(info);
        }
        
        @Override
        public String toString() {
            return Arrays.stream(info).mapToObj(Long::toString).collect(Collectors.joining(","));
        }        
    }
   
    public TimeInfo getInfo() throws InterruptedException {
        final Notification notice = driver.takeNotification();
        return new TimeInfo(notice.getData().asLongBuffer());
   }
    
    public static void main(final String[] args) throws DriverException, InterruptedException  {
        final PTPInfoReader reader = new PTPInfoReader();
        final Instant dcEpochUTC = Instant.parse("2000-01-01T00:00:00Z");
        final CCSTimeStamp dcEpochCCS = CCSTimeStamp.currentTimeFromMillis(dcEpochUTC.toEpochMilli());
        final long dcEpochTAI = dcEpochCCS.getTAIInstant().getEpochSecond() * 1_000_000_000 + dcEpochCCS.getTAIInstant().getNano();
        System.out.println(
            "\"Local TAI time\",\"DC time\",\"DC to TC offset\",\"DC to ext offset\",\"EL6688 internal\",\"EL6688 external\"");
        try {
            reader.getInfo(); // Throw the first notice away, it's sent whhen notifications are first turned on.
            while (true) {
                final TimeInfo info = reader.getInfo();
                final Instant now = CCSTimeStamp.currentTime().getTAIInstant();
                final long taiNanos = now.getNano() + now.getEpochSecond() * 1_000_000_000 - dcEpochTAI;
                System.out.println(taiNanos + "," + info.toString());
            }
        }
        finally {
            reader.close();
        }
        
    }
    
}
