package org.lsst.ccs.subsystem.shutter.statemachine;

/**
 * The top-level context for the state machine. Thread-safe.
 * @author tether
 */
public final class TopContext extends SimpleContext<TopContext> {

    // PROTECTED by the instance lock
    private State<TopContext> synchronizingState;
    private State<TopContext> syncErrorState;
    private State<TopContext> inSyncState;
    private boolean plcEnableFlag;
    // END PROTECTED

    /**
     * Saves the action implementations for this context.
     * @param actions The action implementations.
     * @throws NullPointerException if the actions ref is null.
     */
    public TopContext(final Actions actions) {
        super(actions);
        plcEnableFlag = false;
    }

    /**
     * Sets the flag that the {@code InSync} state will use to determine which of its
     * substates will become its current state when {@code InSync} is entered.
     * @param flag true for {@code Enabled}, false for {@code Disabled}.
     * @see Disabled
     * @see Enabled
     * @see #getPLCEnabled() 
     */
    synchronized void setPLCEnabled(final boolean flag) {plcEnableFlag = flag;}

    /**
     * Gets the flag set by {@code setPLCEnabled}.
     * @return The flag value last set.
     * @see #setPLCEnabled(boolean) 
     */
    synchronized boolean getPLCEnabled() {return plcEnableFlag;}

    /**
     * Instantiates the {@code Synchronizing} state if need be, thereafter returning
     * that instance.
     * @return The {@code Synchronizing} state object.
     */
    synchronized State<TopContext> getSynchronizingState() {
        if (synchronizingState == null) synchronizingState = new Synchronizing(this);
        return synchronizingState;
    }

    /**
     * Instantiates the {@code SyncError} state if need be, thereafter returning
     * that instance.
     * @return The {@code SyncError} state object.
     */
    synchronized State<TopContext> getSyncErrorState() {
        if (syncErrorState == null) syncErrorState = new SyncError(this);
        return syncErrorState;
    }

    /**
     * Instantiates the {@code InSync} state if need be, thereafter returning
     * that instance.
     * @return The {@code InSync} state object.
     */
    synchronized State<TopContext> getInSyncState() {
        if (inSyncState == null) inSyncState = new InSync(getActions(), this);
        return inSyncState;
    }

    /** @implNote Sets the current state to {@code Synchronizing}. */
    @Override
    public void init() {
        makeTransition(getSynchronizingState(), null);
    }

}
