package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Level;

/**
 * Holder for static logging methods.
 * @author tether
 */
class Logging {

    /**
     * Logs a FINE level message containing "{ENTRY state-name OF context-name}".
     * @param state The state that's being entered.
     */
    public static final void logEntry(final State<?> state) {
        state.getLogger().fine(
            () -> String.format("{ENTRY %s OF %s}",
                    state.getName(),
                    state.getContext().getName()) );
    }

    /**
     * Logs a FINE level message containing "{TRANSITION ACTION name}".
     * @param state The state whose logger will be used, usually the state which
     * defines the transition action body.
     * @param name The name of the transition action to appear in the log message.
     */
    public static final void logTransitionAction(final State<?> state, final String name) {
        state.getLogger().fine(
            () -> String.format("{TRANSITION ACTION %s}", name) );
    }

    /**
     * Logs a FINE level message containing "(EXIT state-name OF context-name}".
     * @param state The state that's being exited.
     */
    public static final void logExit(final State<?> state) {
        state.getLogger().fine(
            () -> String.format("{EXIT %s OF %s}",
                    state.getName(),
                    state.getContext().getName()));
    }

    /**
     * Logs a FINE level message containing "(EVENT event-name IN state-name}".
     * Must be called directly from the state's event.
     * @param state The state which is handling the event.
     */
    public static final void logEvent(final State<?> state) {
        // We have to use the isLoggable() test since generating exceptions is expensive
        // but we can't use a lambda expression or anonymous class since then the
        // stack trace would depend on the details of logger implementation.
        if (state.getLogger().isLoggable(Level.FINE)) {
            final String eventName = new Exception().getStackTrace()[1].getMethodName();
            state.getLogger()
                .fine(String.format("{EVENT %s() IN %s}",
                                          eventName,
                                          state.getName()));
        }
    }

}
