package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.Objects;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logExit;

/**
 * Both a context for its substates and a state itself in some other context.
 * Adds a field for the owning super-context.
 * @param <C> The concrete context class inheriting from this one.
 * @param <S> The class of the super-context that will be owning this one as a state.
 * Thread-safe.
 * @author tether
 */
abstract class CompositeState<C extends Context<?>, S extends Context<?>>
    extends SimpleContext<C>
    implements State<S>
{
    private final S context;

    /**
     * Saves the refs to the super-context and the action implementations.
     * @param actions The action implementations.
     * @param context The super-context which owns this one.
     * @throws NullPointerException if either argument is null.
     */
    protected CompositeState(final Actions actions, final S context) {
        super(actions);
        this.context = Objects.requireNonNull(context, "The context argument must not be null.");
    }

    @Override
    public final S getContext() {return context;}

    /**
     * @implSpec All implementations MUST call {@code init()} and should call
     * {@code logEntry()} before that.
     * @implNote This default implementation conforms to the spec but does nothing else.
     * @see #init() 
     */
    @Override
    public void entry() {
        logEntry(this);
        init();
    }
    
    /**
     * @implSpec All implementations must call the exit method of this context's
     * current state, call {@code finish()} on {@code this} and finally call
     * {@code logExit()}.
     * @implNote This default implementation calls conforms to the spec but does nothing else.
     * @see #finish() 
     * @see Logging#logExit(org.lsst.ccs.subsystem.shutter.statemachine.State) 
     */
    @Override
    public void exit() {
        getState().exit();
        finish();
        logExit(this);
    }

}
