package org.lsst.ccs.subsystem.shutter.statemachine;

import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.lsst.ccs.subsystem.motorplatform.bus.ChangeAxisEnable;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAllFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.ClearAxisFaults;
import org.lsst.ccs.subsystem.motorplatform.bus.DisableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.EnableAllAxes;
import org.lsst.ccs.subsystem.motorplatform.bus.HomeAxis;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisAbsolute;
import org.lsst.ccs.subsystem.motorplatform.bus.MoveAxisRelative;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.plc.Calibrate;
import org.lsst.ccs.subsystem.shutter.plc.ChangeAxisEnablePLC;
import org.lsst.ccs.subsystem.shutter.plc.ChangeBrakeState;
import org.lsst.ccs.subsystem.shutter.plc.ClearAllFaultsPLC;
import org.lsst.ccs.subsystem.shutter.plc.ClearAxisFaultsPLC;
import org.lsst.ccs.subsystem.shutter.plc.DisableAllAxesPLC;
import org.lsst.ccs.subsystem.shutter.plc.EnableAllAxesPLC;
import org.lsst.ccs.subsystem.shutter.plc.HomeAxisPLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisAbsolutePLC;
import org.lsst.ccs.subsystem.shutter.plc.MoveAxisRelativePLC;
import org.lsst.ccs.subsystem.shutter.plc.ToggleSafetyCheck;

import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEntry;
import static org.lsst.ccs.subsystem.shutter.statemachine.Logging.logEvent;
import static org.lsst.ccs.subsystem.shutter.statemachine.PromptReply.ACCEPTED;

/**
 * The maint-mode state in which neither blade set is moving. Thread-safe.
 * @author tether
 */
class Still extends SimpleState<Maint> {
    private static final Logger LOG = Logger.getLogger(Still.class.getName());

    /**
     * Saves the context for this state.
     * @param context The context.
     * @throws NullPointerException if the context is null.
     */
    public Still(final Maint context) {
        super(context);
    }

    @Override
    public Logger getLogger() {
        return LOG;
    }

    @Override
    public void entry() {
        logEntry(this);
        getContext().getActions().setSoftwareState(SoftwareState.STILL);
    }

    @Override
    public void enableAllAxes(final Channel<EventReply> chan, final EnableAllAxes req) throws InterruptedException {
        logEvent(this);
        if (req == null) {
            chan.write(new PromptReply("Null argument for enableAllAxes() event."));
        }
        else {
            chan.write(ACCEPTED);
            final Maint ctx = getContext();
            ctx.getActions().relay(new EnableAllAxesPLC(req));
        }
    }

    @Override
    public void disableAllAxes(final Channel<EventReply> chan, final DisableAllAxes req) throws InterruptedException {
        logEvent(this);
        if (req == null) {
            chan.write(new PromptReply("Null argument for disableAllAxes() event."));
        }
        else {
            final Actions acts = getContext().getActions();
            boolean allBrakesOn = true;
            for (final ShutterSide side: ShutterSide.values()) {
                allBrakesOn &= acts.brakeIsEngaged(side);
            }
            if (allBrakesOn) {
                chan.write(ACCEPTED);
                acts.relay(new DisableAllAxesPLC(req));
            }
            else {
                chan.write(new PromptReply("Engage all brakes before disabling all axes."));
            }
        }
    }

    @Override
    public void clearAxisFaults(final Channel<EventReply> chan, final ClearAxisFaults req) throws InterruptedException {
        logEvent(this);
        if (req == null) {
            chan.write(new PromptReply("Null argument for clearAxisFaults() event."));
        }
        else if (Axis.fromName(req.getAxisName()) == null) {
            chan.write(badAxisReply(req.getAxisName()));
        }
        else {
            final Actions acts = getContext().getActions();
            chan.write(ACCEPTED);
            acts.relay(new ClearAxisFaultsPLC(req));
        }
    }

    @Override
    public void changeAxisEnable(final Channel<EventReply> chan, final ChangeAxisEnable req) throws InterruptedException {
        logEvent(this);
        if (req == null) {
            chan.write(new PromptReply("Null argument for changeAxisEnable() event."));
        }
        else if (Axis.fromName(req.getAxisName()) == null) {
            chan.write(badAxisReply(req.getAxisName()));
        }
        else {
            final ShutterSide side = ShutterSide.fromAxis(Axis.fromName(req.getAxisName()));
            final Actions acts = getContext().getActions();
            if (req.isEnabled() || acts.brakeIsEngaged(side)) {
                chan.write(ACCEPTED);
                acts.relay(new ChangeAxisEnablePLC(req));
            }
            else {
                chan.write(new PromptReply("Before disabling an axis you must engage its brake."));
            }
        }
    }

    @Override
    public void changeBrakeState(final Channel<EventReply> chan, Axis ax, ChangeBrakeState.State newState) throws InterruptedException {
        logEvent(this);
        final Actions acts = getContext().getActions();
        if (ax == null || newState == null) {
            chan.write(new PromptReply("The axis or the new brake state is null."));
        }
        else {
            final ShutterSide side = ShutterSide.fromAxis(ax);
            if (newState == ChangeBrakeState.State.RELEASED && !acts.axisIsEnabled(side))
            {
                chan.write(new PromptReply("An axis must be enabled before releasing its brake."));
            }
            else {
                chan.write(ACCEPTED);
                acts.relay(new ChangeBrakeState(ax, newState));
            }
        }
    }

    @Override
    public void clearAllFaults(final Channel<EventReply> chan, final ClearAllFaults req) throws InterruptedException {
        logEvent(this);
        if (req == null) {
            chan.write(new PromptReply("Null argument for clearAllFaults() event."));
        }
        else {
            chan.write(ACCEPTED);
            final Maint ctx = getContext();
            ctx.getActions().relay(new ClearAllFaultsPLC(req));
        }
    }

    @Override
    public void moveAxisRelative(final Channel<EventReply> chan, final MoveAxisRelative req) throws InterruptedException {
        logEvent(this);
        if (req == null) {
            chan.write(new PromptReply("Null argument for moveAxisRelative() event."));
        }
        else if (Axis.fromName(req.getAxisName()) == null) {
            chan.write(badAxisReply(req.getAxisName()));
        }
        else if (req.getTime().isNegative() || req.getTime().isZero()) {
            chan.write(new PromptReply("Move time must be greater than zero."));
        }
        else {
            chan.write(ACCEPTED);
            final Maint ctx = getContext();
            ctx.makeTransition(
                ctx.getMovingState(),
                () -> ctx.getActions().relay(new MoveAxisRelativePLC(req)));
        }
    }

    @Override
    public void moveAxisAbsolute(final Channel<EventReply> chan, final MoveAxisAbsolute req) throws InterruptedException {
        logEvent(this);
        if (req == null) {
            chan.write(new PromptReply("Null argument for moveAxisAbsolute() event."));
        }
        else if (Axis.fromName(req.getAxisName()) == null) {
            chan.write(badAxisReply(req.getAxisName()));
        }
        else if (req.getSpeed() <= 0.0) {
            chan.write(new PromptReply("Move speed must be greater than zero."));
        }
        else {
            chan.write(ACCEPTED);
            final Maint ctx = getContext();
            ctx.makeTransition(
                ctx.getMovingState(),
                () -> ctx.getActions().relay(new MoveAxisAbsolutePLC(req)));
        }
    }

    @Override
    public void homeAxis(final Channel<EventReply> chan, final HomeAxis req) throws InterruptedException {
        logEvent(this);
        if (req == null) {
            chan.write(new PromptReply("Null argument for homeAxis() event."));
        }
        else if (Axis.fromName(req.getAxisName()) == null) {
            chan.write(badAxisReply(req.getAxisName()));
        }
        else {
            chan.write(ACCEPTED);
            final Maint ctx = getContext();
            ctx.makeTransition(
                ctx.getMovingState(),
                () -> ctx.getActions().relay(new HomeAxisPLC(req)));
        }
    }

    @Override
    public void gotoCenter(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
        final Maint ctx = getContext();
        ctx.makeTransition(ctx.getCentering1State(), () -> ctx.getActions().startFirstCentering());
    }

    @Override
    public void calibrate(final Channel<EventReply> chan, final Calibrate calibParams) throws InterruptedException {
        logEvent(this);
        final Maint ctx = getContext();
        if (calibParams == null) {
            chan.write(new PromptReply("Null parameter for calibrate() event."));
        }
        else if (ctx.getActions().readyForCalibration()) {
            chan.write(ACCEPTED);
            ctx.makeTransition(ctx.getCalibratingState(), () -> ctx.getActions().relay(calibParams));
        }
        else {
            chan.write(new PromptReply("Not ready for calibration. See the log."));
        }
    }

    @Override
    public void toggleSafetyCheck(final Channel<EventReply> chan) throws InterruptedException {
        logEvent(this);
        chan.write(ACCEPTED);
        getContext().getActions().relay(new ToggleSafetyCheck());
    }

    private static PromptReply badAxisReply(final String bad) {
        final String valid = Stream.of(Axis.values())
            .map(ax -> "'" + ax.getName() + "'")
            .collect(Collectors.joining(", "));
        return new PromptReply(String .format("Bad axis name '%s'. Valid names: %s.", bad, valid));
    }
}
