package org.lsst.ccs.subsystem.shutter.plc;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;

import static org.lsst.ccs.subsystem.shutter.plc.Tools.getBoolean;
import static org.lsst.ccs.subsystem.shutter.plc.Tools.putBoolean;

import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus.AxisStatus;

/**
 * Holds a ShutterStatusPLC message sent from the PLC. Holds an instance
 * of {@code ShutterStatus} internally.
 * @see ShutterStatus
 * @author tether
 */
public class ShutterStatusPLC extends MsgToCCS {

    final ShutterStatus status;

    /**
     * Constructs from a sequence number and a {@code ShutterStatus} value.
     * @param sequence A message sequence number.
     * @param status A {@code ShutterStatus} value
     * @see ShutterStatus
     */
    public ShutterStatusPLC(
        final int sequence,
        final ShutterStatus status
    )
    {
        super(sequence);
        this.status = status;
    }

    /**
     * Reads and converts the PLC form of the message.
     * @param data Holds the PLC message data.
     */
    public ShutterStatusPLC(final ByteBuffer data) {
        super(data);
        final int motionProfile = data.getInt();
        final boolean isCalibrated = getBoolean(data);
        final int smState = data.getInt();
        final Map<ShutterSide, AxisStatus> axes = new EnumMap<>(ShutterSide.class);
        axes.put(ShutterSide.fromAxis(Axis.AXIS0), decodeAxisStatus(data));
        axes.put(ShutterSide.fromAxis(Axis.AXIS1), decodeAxisStatus(data));
        final boolean isSafetyOn = getBoolean(data);
        final List<Integer> temperature = new ArrayList<>(3);
        temperature.add(data.getInt());
        temperature.add(data.getInt());
        temperature.add(data.getInt());
        this.status = new ShutterStatus(motionProfile, isCalibrated, smState, axes, isSafetyOn, temperature);
    }

    private static AxisStatus decodeAxisStatus(final ByteBuffer data) {
        final double actPos = data.getDouble();
        final double actVel = data.getDouble();
        final double setAcc = data.getDouble();
        final boolean enabled = getBoolean(data);
        final boolean brakeEngaged = getBoolean(data);
        final boolean lowLimit = getBoolean(data);
        final boolean highLimit = getBoolean(data);
        final boolean isHomed = getBoolean(data);
        final int errorID = data.getInt();
        final double motorTemp = data.getDouble();
        return new AxisStatus(actPos, actVel, setAcc, enabled, brakeEngaged,
            lowLimit, highLimit, isHomed, errorID, motorTemp);
    }

    @Override
    public void encode(final ByteBuffer data) {
        super.encode(data);
        data.putInt(status.getMotionProfile());
        putBoolean(data, status.isCalibrated());
        data.putInt(status.getSmState());
        for (final Axis ax: Axis.values()) {
            encodeAxisStatus(data, status.getAxisStatus(ShutterSide.fromAxis(ax)));
        }
        putBoolean(data, status.isSafetyOn());
        status.getTemperature().forEach(data::putInt);
    }

    private static void encodeAxisStatus(final ByteBuffer data, final AxisStatus status) {
        data.putDouble(status.getActPos());
        data.putDouble(status.getActVel());
        data.putDouble(status.getSetAcc());
        putBoolean(data, status.isEnabled());
        putBoolean(data, status.isBrakeEngaged());
        putBoolean(data, status.atLowLimit());
        putBoolean(data, status.atHighLimit());
        putBoolean(data, status.isHomed());
        data.putInt(status.getErrorID());
        data.putDouble(status.getMotorTemp());
    }

    /**
     * Gets the object that can be transmitted on the CCS status bus.
     * @return The internal instance of {@code ShutterStatus}.
     */
    public ShutterStatus getStatusBusMessage() {
        return status;
    }

    /**
     * Forwarded to the internal {@code ShutterStatus} instance.
     * @return
     */
    public int getMotionProfile() {
        return status.getMotionProfile();
    }

    /**
     * Forwarded to the internal {@code ShutterStatus} instance.
     * @return
     */
    public boolean isCalibrated() {
        return status.isCalibrated();
    }

    /**
     * Forwarded to the internal {@code ShutterStatus} instance.
     * @return
     */
    public int getSmState() {
        return status.getSmState();
    }

    /**
     * Forwarded to the internal {@code ShutterStatus} instance.
     * @return
     */
    public AxisStatus getAxisStatus(final Axis ax) {
        return status.getAxisStatus(ShutterSide.fromAxis(ax));
    }

    /**
     * Forwarded to the internal {@code ShutterStatus} instance.
     * @return
     */
    public boolean isSafetyOn() {
        return status.isSafetyOn();
    }

    @Override
    public String toString() {
        return
            "ShutterStatusPLC{" +
            super.toString() +
            " status=" +
            status.toString() +
            "}";
    }
}
