package org.lsst.ccs.subsystem.shutter.plc;

import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Provides some small utility functions.
 * @author tether
 */
public class Tools {

    /**
     * Gets a PLC BOOL value from a byte buffer and converts it to a boolean value.
     * @param data Holds the data.
     * @return The boolean value.
     */
    public static boolean getBoolean(final ByteBuffer data) {return data.get() != 0;}

    /**
     * Appends to a byte buffer the PLC BOOL value equivalent to a Java boolean.
     * @param data Holds the data.
     * @param b The boolean value to convert and append.
     */
    public static void putBoolean(final ByteBuffer data, final boolean b)
    {data.put(b ? (byte)1 : (byte)0);}

    private static final long OFFSET_FROM_TAI_TO_UNIX = 946_684_813_000L; // msec
    /**
     * Converts an EtherCAT Distributed Clock time value into
     * a CCS timestamp with millisecond precision.
     * <p>
     * Assuming that the DC time is being kept in sync with a TAI or GPS time source
     * then it's just a question of adjusting the epoch from that of DC time,
     * 01-01-2000T00:00:00 UTC, to that of Unix, 01-01-1970T00:00:00 UTC. We do that
     * by adding 946,684,813,000 milliseconds.
     * <p>
     * @param dcNanos The DC time value measured in nanoseconds since 01-01-2000T00:00:00 UTC.
     * @return The CCS timestamp obtained from {@code CCSTimeStamp.currentTimeFromMillis()}.
     * @see 
     */
    public static CCSTimeStamp fromDcInstant(final long dcNanos) {
        BigInteger foo;
        final long dcMillis = (dcNanos + 0_500_000L) / 1_000_000L;
        // TODO - do proper time conversion.
        // THis method will work only if there have been no leap seconds added
        // between then and now.
        final CCSTimeStamp now = CCSTimeStamp.currentTime();
        final Instant taiNow = now.getTAIInstant();
        final Instant utcNow = now.getUTCInstant();
        final long nleapMillis = Duration.between(utcNow, taiNow).toMillis();
        return CCSTimeStamp.currentTimeFromMillis(dcMillis + OFFSET_FROM_TAI_TO_UNIX - nleapMillis);
    }
    
    /**
     * Converts a CCS timestamp to a DC time value (count of nanoseconds) with
     * millisecond resolution.
     * @param stamp The CCS timestamp.
     * @return The number of nanoseconds since the DC time epoch, see {@link #fromDcInstant(long) }.
     */
    public static long toDcInstant(final CCSTimeStamp stamp) {
        final long unixMillis = stamp.getTAIInstant().toEpochMilli();
        return (unixMillis - OFFSET_FROM_TAI_TO_UNIX) * 1_000_000L;
    }
    
    /**
     * Converts EtherCat time interval, assumed to be the difference of two DC time values,
     * into a {@code Duration}.
     * @param dcNanos The DC time difference, a count of nanoseconds.
     * @return The {@code Duration} value.
     */
    public static Duration fromDcDuration(final long dcNanos) {
        return Duration.ofNanos(dcNanos);
    }
    
    /**
     * Reverses the transformation performed by {@link #toDcDuration(java.time.Duration) }.
     * @param interval The time interval to convert.
     * @return The count of nanoseconds.
     */
    public static long toDcDuration(final Duration interval) {
        return interval.toNanos();
    }

    public static void main(final String[] args) {
        final List<String> dctimes = new ArrayList<>(Arrays.asList(args));
        if (dctimes.isEmpty()) {dctimes.add("619569008118000000");}  // Around 22:00 2019-08-19 UTC.
        for (final String dc: dctimes) {
            final long dcl = Long.parseLong(dc);
            System.out.printf("DC time %s converts to %s.%n", dc,fromDcInstant(dcl));
        }
    }
}
