package org.lsst.ccs.subsystems.shutter.parker;

import java.io.IOException;
import java.io.PrintStream;
import static java.lang.System.out;

import java.util.EnumMap;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;


import org.lsst.ccs.drivers.parker.AcrComm;
import static org.lsst.ccs.drivers.parker.AxisBit.JOG_ACTIVE;
import org.lsst.ccs.drivers.parker.AxisName;
import static org.lsst.ccs.drivers.parker.AxisSingle.JOG_ACCELERATION_SETTING;
import static org.lsst.ccs.drivers.parker.AxisSingle.JOG_DECELERATION_SETTING;
import static org.lsst.ccs.drivers.parker.AxisSingle.JOG_JERK_SETTING;
import static org.lsst.ccs.drivers.parker.AxisSingle.JOG_VELOCITY_SETTING;
import static org.lsst.ccs.drivers.parker.AxisSingle.MINUS_EXCESS_ERROR;
import static org.lsst.ccs.drivers.parker.AxisSingle.PLUS_EXCESS_ERROR;
import static org.lsst.ccs.drivers.parker.AxisSingle.PULSES_PER_UNIT;
import static org.lsst.ccs.drivers.parker.EncoderLong.ENCODER_POSITION;
import org.lsst.ccs.drivers.parker.EncoderName;
import org.lsst.ccs.drivers.parker.LocalString;

import org.lsst.ccs.subsystems.shutter.common.BladePosition;
import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.MovementHistory;
import org.lsst.ccs.subsystems.shutter.common.ShutterConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystems.shutter.common.ShutterSide.MINUSX;

import static org.lsst.ccs.subsystems.shutter.common.ShutterSide.PLUSX;


/**
 *
 *  Performs various tests on shutter blade sets.
 *  @author tether
 *  @author Owen Saxton
 *
 */
public class TestBladeSets {

    private static Map<ShutterSide, BladeSetDrvr> blades;
    private static Map<ShutterSide, BladeSetConfiguration> configs;
    private static ShutterConfiguration shutterConfig;
    private static MovementHistory mhData;


    ////////// Shell command definitions /////////
    // These take care of initialization and argument checking.

    @Command(name="open", description="Opens a new connection to the motor controller.")
    public void open_command(
        @Argument(
            name="configuration",
            description="The name of a shutter configuration,"+
            " treated as a resource bundle name and so found using CCS search rules.")
        final String configuration
    )
        throws IOException
    {
        if (blades != null) {close_command();}
        final ConfigPropConverter converter = new ConfigPropConverter(configuration);
        configs = converter.getBladeSetConfigs();
        shutterConfig = converter.getShutterConfig();
        final ParkerController controller = new ParkerController();
        controller.init(configs, shutterConfig, null);
        blades = new EnumMap<>(ShutterSide.class);
        blades.put(PLUSX, (BladeSetDrvr)controller.getBladeSets().get(PLUSX));
        blades.put(MINUSX, (BladeSetDrvr)controller.getBladeSets().get(MINUSX));
    }

    @Command(name="close", description="Closes any existing controller connection.")
    public void close_command() {
        if (blades != null) {
            blades.get(PLUSX).getComm().cleanup();
            blades = null;
        }
    }

    @Command(name="showConfig", description="Displays the current shutter configuration.")
    public void showConfig_command() {
        if (blades == null) {
            out.println("Use the 'open' command first.");
            return;
        }
        showConfig();
    }

    private static enum OnOff {ON, OFF;}

    @Command(name="drive", description="Enable or disable a motor drive unit.")
    public void drive_command(
        @Argument(name="side", description="The side of the shutter to affect.")
        final ShutterSide side,
        @Argument(name="newState", description="'on' or 'off'")
        final OnOff newState)
    {
        if (blades == null) {
            out.println("Use the 'open' command first.");
            return;
        }
        drive(blades.get(side), newState);
    }

    @Command(
        name="position",
        description="Position a blade set to some proper fraction of its full extent.\n" +
                    "Uses the motion program and an S-curve motion profile.")
    public void position_command(
        @Argument(name="side", description="The side of the shutter to affect.")
        final ShutterSide side,
        @Argument(name="posn", description="The new position, in the interval [0.0, 1.0].")
        final double posn,
        @Argument(
            name="time",
            description="Move time in seconds. 0 for what's configured (default).",
            defaultValue="0")
        final double time,
        @Argument(
            name="fileNameRoot",
            description="The root of the file name for saving recorded values, default=POSN.",
            defaultValue="POSN")
        final String fileNameRoot
    )
    {
        if (blades == null) {
            out.println("Use the 'open' command first.");
        }
        else if (posn < 0.0 || posn > 1.0) {
            out.println("Position must be between 0 and 1 inclusive.");
        }
        else if (time < 0.0) {
            out.println("Time must be >= 0.0.");
        }
        else {
            position(blades.get(side), posn, time, fileNameRoot);
        }
    }

    @Command(name="showPositions", description="Show blade set positions.")
    public void showPositions_command() {
        if (blades == null) {
            out.println("Use the 'open' command first.");
        }
        else {
            showPositions();
        }
    }

    private static enum MotionProfile {TRAP, SCURVE;}

    @Command(
        name="move",
        description="Moves an absolute distance using a direct command. No profile recording.")
    public void move_command(
        @Argument(name="side", description="The side of the shutter to affect.")
        final ShutterSide side,
        @Argument(name="distance", description="The desired change in position in mm.")
        final double distance,
        @Argument(
            name="time",
            description="The move time in seconds (0 for configured value (default)).",
            defaultValue="0")
        final double time,
        @Argument(
            name="profile",
            description="The motion profile type, TRAP or SCURVE (default).",
            defaultValue="SCURVE")
        final MotionProfile profile
    )
    {
        if (blades == null) {
            out.println("Use the 'open' command first.");
        }
        else if (time < 0.0) {
            out.println("Time must be >= 0.0.");
        }
        else {
            move(blades.get(side), distance, time, profile);
        }
    }

    @Command(
        name="movep",
        description="Moves an absolute distance using the motion program. " +
                    "Records controller clock and encoder value.")
    public void movep_command(
        @Argument(name="side", description="The side of the shutter to affect.")
        final ShutterSide side,
        @Argument(name="distance", description="The desired position change in mm.")
        final double distance,
        @Argument(
            name="time",
            description="The move time in seconds, 0 (default) for the configured value.",
            defaultValue="0")
        final double time,
        @Argument(
            name="nsamp",
            description="The number of samples to record, 0 (default) = configured value.",
            defaultValue="0"
        )
        final int nsamp,
        @Argument(
            name="profile",
            description="The motion profile type, TRAP or SCURVE (default).",
            defaultValue="SCURVE")
        final MotionProfile profile,
        @Argument(
            name="fileNameRoot",
            description="The root of the file name used to save recorded values, default=MOVE.",
            defaultValue="MOVE")
        final String fileNameRoot
    )
    {
        if (blades == null) {
            out.println("Use the 'open' command first.");
        }
        else if (time < 0.0) {
            out.println("Time must be >= 0.0.");
        }
        else if (nsamp <0 ) {
            out.println("Number of samples must be >= 0.");
        }
        else {
            movep(blades.get(side), distance, time, nsamp, profile, fileNameRoot);
        }
    }

    @Command(name="step",
             description="Measures the accuracy and precision of motion.")
    public void step_command(
        @Argument(name="side", description="The side of the shutter to affect.")
        final ShutterSide side,
        @Argument(name="count", description="The number of steps to take.")
        final int count,
        @Argument(name="step", description="The step size in encoder counts.")
        final int step,
        @Argument(name="time", description="The time taken for each step (default 0.1 sec).",
                               defaultValue="0.1")
        final double time
    )
    {
        if (blades == null) {
            out.println("Use the 'open' command first.");
        }
        else if (count <= 0) {
            out.println("Count must be > 0.");
        }
        else if (time <= 0.0) {
            out.println("Time must be > 0.0.");
        }
        else {
            step(blades.get(side), count, step, time);
        }
    }

    @Command(name="showTemp", description="Show the motor drive temperatures.")
    public void showTemp_command() {
        if (blades == null) {
            out.println("Use the 'open' command first.");
            return;
        }
        showTemp();
    }

    ////////// Command implementations //////////

    private void showConfig()
    {
        for (ShutterSide side : ShutterSide.values()) {
            final BladeSetConfiguration cfg = configs.get(side);
            out.format("Blade %s Configuration:%n", cfg.getSide());
            out.format("  Controller host:             %s%n", cfg.getControllerHost());
            out.format("  Encoder pulses/mm:           %s%n", cfg.getEncoderCountsPerMm());
            out.format("  Blade home position (mm):    %s%n", cfg.getHomePos());
            out.format("  Blade open position (mm):    %s%n", cfg.getOpenPos());
            out.format("  Blade closed position (mm):  %s%n", cfg.getClosedPos());
            out.format("  Blade movement time (sec):   %s%n", shutterConfig.getMoveTime());
            out.format("  DIO port to controller:      %s%n", cfg.getDioPortToController());
            out.format("  DIO line to controller:      %s%n", cfg.getDioLineToController());
            out.format("  DIO port from controller:    %s%n", cfg.getDioPortFromController());
            out.format("  DIO line from controller:    %s%n", cfg.getDioLineFromController());
        }
    }

    private void drive(final BladeSetDrvr blade, final OnOff newState)
    {
        if (newState == OnOff.ON) {
            blade.enableDrive();
        }
        else {
            blade.disableDrive();
        }
    }

    private void position(
        final BladeSetDrvr blade,
        final double relPosn,
        final double time,
        final String moveName
    )
    {
        final double t = (time > 0.0) ? time : shutterConfig.getMoveTime();
        long tStamp = System.currentTimeMillis();
        String fileP = genFileName(blade.getSide(), moveName, "_P.dat", tStamp);
        PrintStream oflP = null;

        /*
        **  Open the output files
        */
        try {
            oflP = new PrintStream(fileP);
        }
        catch (IOException e) {
            out.println(e);
            if (oflP != null) oflP.close();
            return;
        }

        /*
        **  Perform the move, capturing position data
        */
        showMove("Started", blade);
        mhData = blade.moveToPosition(relPosn, t);
        showMove("Ended", blade);

        /*
        **  Return if failure
        */
        if (mhData == null) {
            oflP.close();
            return;
        }

        /*
        **  Output the sampled positions
        */
        showStop(mhData.getStatus());
        writePosnData(mhData, oflP);

        /*
        **  Close the output files
        */
        oflP.close();
        out.println("File " + fileP + " created.");
    }

    private void showPositions() {
        for (final ShutterSide side: ShutterSide.values()) {
            final BladeSetDrvr blade = blades.get(side);
            final EncoderName encoder = blade.getEncoder();
            final AcrComm acr = blade.getComm();
            final String line =
                String.format("%s(%s): encoder=%d, relative=%6.4f, absolute=%6.1f mm",
                              side,
                              blade.getAxis(),
                              acr.get(encoder, ENCODER_POSITION),
                              blade.getRelativePosition(),
                              blade.getAbsolutePosition());
            out.println(line);
        }
    }


    private void move(
        final BladeSetDrvr blade,
        final double dist,
        final double time,
        final MotionProfile profile)
    {

        final double t =
            (time > 0.0) ? time : shutterConfig.getMoveTime();
        final int type =
            (profile == MotionProfile.SCURVE)
            ? ParkerConstants.MOV_TYP_SCURVE
            : ParkerConstants.MOV_TYP_TRAP;

        // The controller converts distance to encoder counts using a positive'
        // PPU, so we have to correct for the case where increasing
        // absolute position means decreasing the encoder count.
        final double flip = Math.signum(blade.getBladeSetConfiguration().getEncoderCountsPerMm());
        showMove("Started", blade);
        final int status = blade.move(type, flip*dist, t);
        showMove("Ended", blade);
        showStop(status);
    }

    private void movep(
        final BladeSetDrvr blade,
        final double dist,
        final double time,
        final int nsamp,
        final MotionProfile profile,
        final String moveName
    )
    {
        final double t = (time > 0.0) ? time : shutterConfig.getMoveTime();
        final int configuredNsamp =
            (int)
            Math.ceil(
                (1000.0 *t)    // Dt is in ms.
                /
                (shutterConfig.getMinSampleDt() + t*shutterConfig.getSampleDtScale())
        );
        final int n = (nsamp > 0) ? nsamp : configuredNsamp;
        final int type =
            (profile == MotionProfile.SCURVE)
            ? ParkerConstants.MOV_TYP_SCURVE
            : ParkerConstants.MOV_TYP_TRAP;
        final String file = genFileName(blade.getSide(), moveName, "_P.dat");
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Perform the move, capturing position data
        */
        showMove("Started", blade);
        // The controller converts distance to encoder counts using a positive'
        // PPU, so we have to correct for the case where increasing
        // absolute position means decreasing the encoder count.
        final double flip = Math.signum(blade.getBladeSetConfiguration().getEncoderCountsPerMm());
        MovementHistory data = blade.moveP(type, flip*dist, t, n);
        if (data == null) {
            ofl.close();
            return;
        }
        showMove("Ended", blade);
        //showMove(data.getEndPosition(), data.getEndTime() / 1000);
        showStop(data.getStatus());

        /*
        **  Output the sampled position values
        */
        writePosnData(data, ofl);

        /*
        **  Close the output file
        */
        ofl.close();
        out.println("File " + file + " created");
    }

    private void step(
        final BladeSetDrvr blade,
        final int count,
        final int step,
        final double time)
    {

        /*
        **  Configure and enable the drive. Note that distance units will
        **  be encoder counts, i.e., Pulses Per Unit = 1.0.
        */
        final double vel = Math.abs(2.0 * step / time);
        final double acc = Math.abs(8.0 * step / (time * time));
        final double jrk = acc * acc / vel;

        final AxisName axis = blade.getAxis();
        final AcrComm acr = blade.getComm();
        final EncoderName encoder = blade.getEncoder();
        final BladeSetConfiguration config = configs.get(blade.getSide());
        // Reset the PPU for the axis. Since that will make the EXC values
        // be interpreted as counts we have to rescale them as well.
        final double oldPPU = acr.get(axis, PULSES_PER_UNIT);
        acr.set(axis, PULSES_PER_UNIT, 1.0);
        acr.set(axis, PLUS_EXCESS_ERROR, oldPPU*acr.get(axis, PLUS_EXCESS_ERROR));
        acr.set(axis, MINUS_EXCESS_ERROR, oldPPU*acr.get(axis, MINUS_EXCESS_ERROR));
        // Set motion parameters.
        acr.set(axis, JOG_VELOCITY_SETTING, vel);
        acr.set(axis, JOG_ACCELERATION_SETTING, acc);
        acr.set(axis, JOG_DECELERATION_SETTING, acc);
        acr.set(axis, JOG_JERK_SETTING, jrk);
        // Note that which of the encoder values to follow is the lowest depends
        // on the sign of encoderPulsesPerMm.
        int enc1 = (int)Math.round(blade.absoluteToEncoder(config.getLowestPos()));
        int enc2 = (int)Math.round(blade.absoluteToEncoder(config.getHighestPos()));
        // Set software travel limits.
        acr.sendStr(
            String.format(
                "%s slm (%d, %d)",
                 axis.reference(), Math.max(enc1, enc2), Math.min(enc1, enc2)));
        blade.enableDrive();

        /*
        **  Loop to do steps and read encoder positions
        */
        long nStep, minm = Integer.MAX_VALUE, maxm = Integer.MIN_VALUE,
            pPosn, cPosn, fPosn, sumSq = 0;
        pPosn = fPosn = acr.get(encoder, ENCODER_POSITION);
        for (nStep = 0; nStep < count; nStep++) {
            final String command = String.format("%s jog inc %d", axis.reference(), step);
            acr.sendStr(command);
            do {
                try {Thread.sleep(200);}
                catch (InterruptedException exc) {
                    Thread.currentThread().interrupt();
                    return;
                }

            } while (acr.get(axis, JOG_ACTIVE));
            if (showStop(blade.getSide()) != null) break;
            cPosn = (int)acr.get(encoder, ENCODER_POSITION);
            long cStep = cPosn - pPosn;
            if (cStep > maxm) maxm = cStep;
            if (cStep < minm) minm = cStep;
            sumSq += cStep * cStep;
            pPosn = cPosn;
        }

        /*
        **  Calculate and display results
        */
        double avg = (double)(pPosn - fPosn) / (nStep == 0 ? 1 : nStep);
        double var = (double)sumSq / (nStep == 0 ? 1 : nStep) - avg * avg;
        out.format("Steps = %s, counts = %s, min = %s, max = %s, "
                     + "avg = %.2f, sigma = %.4f%n",
                   nStep, pPosn - fPosn, minm == Integer.MAX_VALUE ? 0 : minm,
                   maxm == Integer.MIN_VALUE ? 0 : maxm, avg, Math.sqrt(var));

        /*
        **  Reset PPU & soft limits, and disable the drive
        */
        acr.set(axis, PULSES_PER_UNIT, oldPPU);
        acr.set(axis, PLUS_EXCESS_ERROR, acr.get(axis, PLUS_EXCESS_ERROR) / oldPPU);
        acr.set(axis, MINUS_EXCESS_ERROR, acr.get(axis, MINUS_EXCESS_ERROR) / oldPPU);
        // Note that the controller calculates user coordinates as just encoder/ppu
        // assuming that that ppu > 0.
        enc1 /= Math.abs(config.getEncoderCountsPerMm());
        enc2 /= Math.abs(config.getEncoderCountsPerMm());
        acr.sendStr(
            String.format(
                "%s slm (%d, %d)",
                 axis.reference(), Math.max(enc1, enc2), Math.min(enc1, enc2)));
    }


   /**
    ***************************************************************************
    **
    **  Processes the TEMPSHOW command
    **
    ***************************************************************************
    */
    private void showTemp()
    {
        for (final ShutterSide side: ShutterSide.values()) {
        out.format("Drive temperature %s(%s): %.1f%n",
                   side,
                   blades.get(side).getAxis(),
                   blades.get(side).getDriveTemperature());
        }
    }


   /**
    ***************************************************************************
    **
    **  Writes a file of sampled position data
    **
    ***************************************************************************
    */
    private void writePosnData(MovementHistory data, PrintStream ofl)
    {
        long sTime = data.getStartTime() / 1000;
        List<BladePosition> pList = data.getPositions();
        ofl.println(
            "\"time(abs millisec)\", \"time(rel millisec)\", \"pos(abs mm)\", \"posn(rel))\"");
        for (final BladePosition pItem: pList) {
            long pTime = pItem.getTime() / 1000;
            ofl.format("%12s, %5s, %8.2f, %8.5f%n", pTime, pTime - sTime,
                       pItem.getPosition(), pItem.getRelPosition());
        }
    }


   /**
    ***************************************************************************
    **
    **  Generates a file name containing blade index and time stamp
    **
    ***************************************************************************
    */
    private String genFileName(final ShutterSide side, String head, String tail, long tStamp)
    {
        GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(tStamp);
        return head + "_" + side.toString() + "_"
                 + String.format("%tY%<tm%<td_%<tH%<tM%<tS", cal) + tail;
    }


    private String genFileName(final ShutterSide side, String head, String tail)
    {
        return genFileName(side, head, tail, System.currentTimeMillis());
    }


   /**
    ***************************************************************************
    **
    **  Shows the move start position and time
    **
    ***************************************************************************
    */
    private void showMove(final String end, final BladeSetDrvr blade)
    {
        final GregorianCalendar cal = new GregorianCalendar();
        cal.setTimeInMillis(System.currentTimeMillis());
        out.format("%s move: posn = %8.2f mm, time = %tF %<tT.%<tL%n",
                   end, blade.getAbsolutePosition(), cal);
    }

   /**
    ***************************************************************************
    **
    **  Shows the reason for a premature stop, if any
    **
    ***************************************************************************
    */
    private String showStop(int status)
    {
        String stopped = getStop(status);
        if (stopped != null) out.println(stopped);

        return stopped;
    }

    private String showStop(final ShutterSide side)
    {
        return showStop(blades.get(side).getMoveStatus());
    }


   /**
    ***************************************************************************
    **
    **  Gets the message text for a premature movement stop
    **
    ***************************************************************************
    */
    private String getStop(int status)
    {
        if (status == 0) return null;
        else return MovementStatus.message(status);
    }
}
