package org.lsst.ccs.subsystems.shutter.parker;


import static org.lsst.ccs.subsystems.shutter.parker.ParkerConstants.*;

import org.lsst.ccs.subsystems.shutter.common.BladeSet;
import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterController;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;

import org.lsst.ccs.drivers.iocard.AccesDio;
import org.lsst.ccs.drivers.parker.AcrComm;
import org.lsst.ccs.drivers.parker.AxisName;
import org.lsst.ccs.drivers.parker.UserParameter;

import static org.lsst.ccs.drivers.parker.AxisBit.ENABLE_NEGATIVE_HARD_LIMIT;
import static org.lsst.ccs.drivers.parker.AxisBit.ENABLE_NEGATIVE_SOFT_LIMIT;
import static org.lsst.ccs.drivers.parker.AxisBit.ENABLE_POSITIVE_HARD_LIMIT;
import static org.lsst.ccs.drivers.parker.AxisBit.ENABLE_POSITIVE_SOFT_LIMIT;
import static org.lsst.ccs.drivers.parker.AxisBit.INVERT_HOME_REGION_SIGNAL;
import static org.lsst.ccs.drivers.parker.AxisBit.INVERT_NEGATIVE_HARD_LIMIT_SIGNAL;
import static org.lsst.ccs.drivers.parker.AxisBit.INVERT_POSITIVE_HARD_LIMIT_SIGNAL;
import static org.lsst.ccs.drivers.parker.AxisBit.JOG_ACTIVE;
import static org.lsst.ccs.drivers.parker.AxisBit.KILL_ALL_MOTION;
import static org.lsst.ccs.drivers.parker.AxisName.AXIS0;
import static org.lsst.ccs.drivers.parker.AxisName.AXIS1;
import static org.lsst.ccs.drivers.parker.AxisSingle.DERIVATIVE_GAIN;
import static org.lsst.ccs.drivers.parker.AxisSingle.INTEGRAL_GAIN;
import static org.lsst.ccs.drivers.parker.AxisSingle.NEGATIVE_TORQUE_LIMIT;
import static org.lsst.ccs.drivers.parker.AxisSingle.POSITIVE_TORQUE_LIMIT;
import static org.lsst.ccs.drivers.parker.AxisSingle.PROPORTIONAL_GAIN;
import static org.lsst.ccs.drivers.parker.AxisSingle.PULSES_PER_UNIT;
import static org.lsst.ccs.drivers.parker.EncoderLong.ENCODER_POSITION;
import static org.lsst.ccs.drivers.parker.EncoderName.ENC0;
import static org.lsst.ccs.drivers.parker.EncoderName.ENC1;
import static org.lsst.ccs.drivers.parker.MasterBit.KILL_ALL_MOVES;
import static org.lsst.ccs.drivers.parker.MasterName.MASTER0;
import static org.lsst.ccs.drivers.parker.SystemBit.SAMPLE_ON_EDGE;
import static org.lsst.ccs.drivers.parker.SystemBit.SAMPLING_ARMED;
import static org.lsst.ccs.drivers.parker.SystemUnsigned.SAMPLE_TIMER_PERIOD;
import static org.lsst.ccs.drivers.parker.SystemUnsigned.SYSTEM_CLOCK;

import org.lsst.ccs.framework.ConfigurableComponent;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import static org.lsst.ccs.drivers.parker.SystemBit.CLEAR_LATCHED_MOTION_DISABLE;

/**
 * Initializes the ACR controller and creates two blade-set objects for use
 * by the subsystem.
 * @author tether
 */
public class ParkerController extends ConfigurableComponent implements ShutterController {

    private Map<ShutterSide, BladeSet> bladeSets = null;
    private AcrComm acr = null;
    private AccesDio dio = null;

    /**
     * Initializes the ACR controller and creates the two blade-set objects.
     * @param bsetConfigs The blade set configurations the two sides.
     * @param shutterConfig The general shutter configuration.
     * @param hallConfig The Hall sensor configuration.
     */
    @Override
    public void init (
        final Map<ShutterSide, BladeSetConfiguration> bsetConfigs,
        final ShutterConfiguration shutterConfig,
        final List<HallConfiguration> hallConfigs
    )
        throws java.io.IOException
    {
        // Require that the two controller IP addresses be the same so that we
        // need only a single instance of AcrComm.
        if (!bsetConfigs.get(ShutterSide.PLUSX).getControllerHost()
            .equals(bsetConfigs.get(ShutterSide.PLUSX).getControllerHost())
            )
        {
            throw new IllegalArgumentException("There should be only one Parker controller.");
        }
        this.acr = AcrComm.newInstance(
            shutterConfig.getController(),
            bsetConfigs.get(ShutterSide.PLUSX).getControllerHost(),
            Optional.empty() // No logging.
        );
        //this.dio = new AccesDio();
        //this.dio.init(-1, -1); // -1 means probe the PCI bus by vendor ID.
        // TODO: Configure the DIO card ports we use.
        acrInitialize(bsetConfigs);

        // Now create the blade sets.
        final Map<ShutterSide, BladeSet> bsets = new EnumMap<>(ShutterSide.class);
        for (ShutterSide side : ShutterSide.values()) {
            bsets.put(
                side,
                new BladeSetDrvr(
                    side,
                    bsetConfigs.get(side),
                    shutterConfig,
                    hallConfigs,
                    acr,
                    dio
                ));
        }
        this.bladeSets = bsets;
    }

    /**
     * Gets the two blades set control objects created during initialization.
     * @return Either null if init() failed, or a map containing one
     * object per shutter side.
     */
    @Override
    public Map<ShutterSide, BladeSet> getBladeSets() {return bladeSets;}


    /**
     * Initializes the motor controller:
     * <ul>
     * <li>Attaches encoder 0(1) and DAC 0(1) to axis 0(1).</li>
     * <li>Attaches  axis 0(1) to master 0 as slave 0(1).</li>
     * <li>Attaches master0 to the motion program.</li>
     * <li>Sets these parameters for each axis:</li>
     * <li>
     *     <ul>
     *     <li>Pulses (encoder counts) Per Unit</li>
     *     <li>Deceleration to use upon hitting a software or hardware travel limit (5G)</li>
     *     <li>Software travel limits</li>
     *     <li>PID gains for motor position feedback</li>
     *     <li>Motor torque limits (one for each direction of travel)(10A)</li>
     *     </ul>
     * </li>
     * <li> Sets the following flags for each axis:
     *     <ul>
     *     <li>Enable hardware travel limits (both directions)</li>
     *     <li>Enable software travel limits (both directions)</li>
     *     </ul>
     * </li>
     * <li> Clears the following flags for each axis:
     *     <ul>
     *     <li>Kill All Motion</li>
     *     <li>Use inverted logic for the Home Region signal</li>
     *     <li>Use inverted logic for the hardware travel limit signals</li>
     *     </ul>
     * </li>
     * <li>Clears the master's Kill ALl Moves flag.</li>
     * <li>Allocates space for the user parameters and the motion programs.</li>
     * <li>Uploads the standard motion program.</li>
     * </ul>
     * @param bsetConfigs The blade set configurations the two sides.
     * @param shutterConfig The general shutter configuration.
     * @param hallConfig The Hall sensor configuration.
     */
    private void acrInitialize(
        final Map<ShutterSide, BladeSetConfiguration> bsetConfigs
    )
    {
        /*
        **  Do basic initialization
        */

           // Global init. of the controller.
        acr.sendStr("sys");
        // Stop all running programs and all motions.
        acr.sendStr("halt all");
        // Set all programs to empty.
        acr.sendStr("new all");
            /** If the Motion Enable input ever lost voltage we have to clear the latched motion disable. */
        acr.set(CLEAR_LATCHED_MOTION_DISABLE, true);
        // The master kill-moves flag must be clear or the master will ignore
        // any motion commands.
        acr.set(MASTER0, KILL_ALL_MOVES, false);
        for (ShutterSide side: ShutterSide.values()) {
            final BladeSetConfiguration config = bsetConfigs.get(side);
            final AxisName axis = AxisName.valueOf("AXIS" + config.getAxis());
            final String axisref = axis.reference();
            // Turn off the motor driver for axis 0.
            acr.sendStr(axisref + " drive off");
            // Enable hardware motion (End Of Travel) limits.
            acr.set(axis, ENABLE_POSITIVE_HARD_LIMIT, true);
            acr.set(axis, ENABLE_NEGATIVE_HARD_LIMIT, true);
            // Enable software motion limits.
            acr.set(axis, ENABLE_POSITIVE_SOFT_LIMIT, true);
            acr.set(axis, ENABLE_NEGATIVE_SOFT_LIMIT, true);
            // Do NOT use active-low logic for the home and EOT sensors.
            acr.set(axis, INVERT_HOME_REGION_SIGNAL, false);
            acr.set(axis, INVERT_POSITIVE_HARD_LIMIT_SIGNAL, false);
            acr.set(axis, INVERT_NEGATIVE_HARD_LIMIT_SIGNAL, false);
            // Make sure that the axis move-killing bits are clear, otherwise no motion commands
            // will be obeyed.
            acr.set(axis, KILL_ALL_MOTION, false);
            acr.set(axis, DERIVATIVE_GAIN, config.getFeedbackD());
            // Enable motor control for the axis.
            acr.sendStr(axisref+" on");
        }

        /*
        **  Load the motion program. Note that DIM is an executable statement, not a
        **  static declaration; space allocation for scalar variables and arrays
        **  takes place each time the program is run provided you use CLEAR
        **  to remove old allocations and prevent double-dimming errors.
        */
        acr.sendStr(MOVE_PROG.reference());
        // Accept program text until the next PEND.
        acr.sendStr("program");
        // Mark progress.
        acr.sendStr(P_PROGRESS.reference() + " = " + PCD_STARTED);
        // Deallocate all memory allocated for local variables and arrays.
        acr.sendStr("clear");
        // Reset to defaults all system parameters and flags dealing with sampling.
        acr.sendStr("samp clear");
        // Allocate long (32-bit) integer local variables, named LV0 - LV9.
        acr.sendStr("dim lv(10)");
        // Set up parameter sampling. The user parameters P_NUM_INT_PARAMS and P_NUM_FLOAT_PARAMS
        // respectively hold the number of long (int) and single (float)
        // controller parameters that will be sampled. Each sampled parameter
        // gets an array of the appropriate type large enough to hold the number of samples
        // specified by user parameter P_NUM_SAMPLES. The index numbers of the parameters
        // to sample start at user parameter P_FIRST_PARAM_NUMBER, first those of the long parameters
        // and then those of the single parameters.
        acr.sendStr("dim la(" + P_NUM_INT_PARAMS.reference() + ")");
        acr.sendStr("dim sa(" + P_NUM_FLOAT_PARAMS.reference() + ")");
        // For each array of longs ...
        acr.sendStr("for lv0 = 0 to (" + P_NUM_INT_PARAMS.reference() + " - 1) step 1");
        //     Give the array one element per desired sample.
        acr.sendStr("dim la(lv0)(" + P_NUM_SAMPLES.reference() + ")");
        //     The i'th long parameter will be stored in array la<i>.
        acr.sendStr("samp(lv0) base la(lv0)");
        //     The index number of the parameter is in user parameter P_FIRST_PARAM_NUMBER+i
        acr.sendStr("samp(lv0) src p(p(lv0 + " + P_FIRST_PARAM_NUMBER.index() + "))");
        //     End of loop.
        acr.sendStr("next");
        // The same sampling setup for the float parameters.
        acr.sendStr("for lv0 = 0 to (" + P_NUM_FLOAT_PARAMS.reference() + " - 1) step 1");
        acr.sendStr("dim sa(lv0)(" + P_NUM_SAMPLES.reference() + ")");
        acr.sendStr("samp(lv0 + " + P_NUM_INT_PARAMS.reference() + ") base sa(lv0)");
        acr.sendStr(
            "samp(lv0 + "
            + P_NUM_INT_PARAMS.reference()
            + ") src p(p(lv0 + "
            + P_NUM_INT_PARAMS.reference()
            + " + "
            + P_FIRST_PARAM_NUMBER.index()
            + "))");
        acr.sendStr("next");

        // Set up sample timing and triggering.
        // The axis number will be in a user parameter.
        final String axisIndirect =
            String.format("axis(%s)", P_AXIS_NUMBER.reference());
        // Continuous-mode sampling: when started, sample at every servo interrupt
        // or at regular intervals.
        acr.sendStr("clr " + SAMPLE_ON_EDGE.reference());
        // Start sampling when JOG ACTIVE goes up. Calculate the index of the right
        // bit using the axis number and the fixed separation between the bit
        // indexes for the axes.
        acr.sendStr(String.format(
                "samp trg +(%d + %s * %d)",
                JOG_ACTIVE.index(AXIS0),
                P_AXIS_NUMBER.reference(),
                JOG_ACTIVE.index(AXIS1) - JOG_ACTIVE.index(AXIS0)));
        // Set sampling period in milliseconds (held in user param. P_SAMPLE_INTERVAL)
        acr.sendStr(SAMPLE_TIMER_PERIOD.reference() + " = " + P_SAMPLE_INTERVAL.reference());
        // End of sampling setup.
        acr.sendStr(P_PROGRESS.reference() + " = " + PCD_SAMPLING_SET_UP);

        // If the motor driver is not left on in between moves, turn it on now.
        acr.sendStr("if (" + P_DRIVE_ON.reference() + " = 0) then "+axisIndirect+" drive on");
        //  Delay (dwell) for one-tenth second.
        acr.sendStr("dwl 0.1");
        // Save the current system time and the position encoder value.
        acr.sendStr("lv0 = " + SYSTEM_CLOCK.reference());
        acr.sendStr(String.format(
                "lv1 = p(%d + %s * %d)",
                ENCODER_POSITION.index(ENC0),
                P_AXIS_NUMBER.reference(),
                ENCODER_POSITION.index(ENC1) - ENCODER_POSITION.index(ENC0)));
        // Enable sampling.
        acr.sendStr("set " + SAMPLING_ARMED.reference());
        // Start the motion. Move the distance given by parameter P_MOVE_DISTANCE using the
        // currently set jog parameters, set by configMove() or initMove().
        acr.sendStr(axisIndirect + " jog inc (" + P_MOVE_DISTANCE.reference() + ")");
        acr.sendStr(P_PROGRESS.reference() + " = " + PCD_MOTION_STARTED);
        // Wait for motion to complete.
        acr.sendStr(String.format(
                "inh -bit(%d + %s * %d)",
                JOG_ACTIVE.index(AXIS0),
                P_AXIS_NUMBER.reference(),
                JOG_ACTIVE.index(AXIS1) - JOG_ACTIVE.index(AXIS0)));
        acr.sendStr(P_PROGRESS.reference() + " = " + PCD_MOTION_COMPLETE);
        // Save the new position encoder value.
        acr.sendStr(String.format(
                "lv2 = p(%d + %s * %d)",
                ENCODER_POSITION.index(ENC0),
                P_AXIS_NUMBER.reference(),
                ENCODER_POSITION.index(ENC1) - ENCODER_POSITION.index(ENC0)));
        // Wait for all the samples to be written.
        acr.sendStr("inh -" + SAMPLING_ARMED.reference());
        acr.sendStr(P_PROGRESS.reference() + " = " + PCD_SAMPLING_COMPLETE);
        // If the motor driver is not left on in between moves, turn it off now.
        acr.sendStr("if (" + P_DRIVE_ON.reference() + " = 0) then " + axisIndirect + " drive off");
        acr.sendStr(P_PROGRESS.reference() + " = " + PCD_DONE);
        acr.sendStr("endp");

        // Get back to the system-level prompt.
        acr.sendStr("sys");
    }

}
