package org.lsst.ccs.subsystems.shutter.parker;

import org.lsst.ccs.subsystems.shutter.CCSConfigurationService;

import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;

import org.lsst.ccs.utilities.conv.InputConversionEngine;

import org.lsst.ccs.utilities.logging.Logger;


import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;


/**
 * Reads a Java properties file and converts the configuration information strings it
 * contains into proper configuration objects.
 * @author tether
 */
public class ConfigPropConverter {
    
    private static final Logger log = Logger.getLogger(ConfigPropConverter.class.getPackage().getName());
    
    private final InputConversionEngine engine;
    
    private final ResourceBundle bundle;
    
    /**
     * Reads the properties file.
     * @param propFileName The name of the properties file w/o directory name or extension.
     * The file is found using the class path mechanism (using ResourceBundle).
     */
    public ConfigPropConverter(String propFileName) {
        this.engine = new InputConversionEngine();
        this.bundle = ResourceBundle.getBundle(propFileName);
    }
    
    /**
     * Converts the property string for both sides to proper blade set configuration objects.
     * @return The map of side to configuration object, or null if either
     * side is missing or invalid.
     */
    public Map<ShutterSide, BladeSetConfiguration> getBladeSetConfigs() {
        // Run the input conversion engine on the appropriate property value,
        // then use the same conversion used for the shutter subsystem proper.
        Map<ShutterSide, BladeSetConfiguration> result = new EnumMap<>(ShutterSide.class);
        for (ShutterSide side : ShutterSide.values()) {
            try {
                final Map<String, String> raw = (Map<String, String>) engine.convertArgToType(
                    bundle.getString(side.name().toLowerCase() + "//config"),
                    (new HashMap<String, String>()).getClass()
                );
                result.put(
                    side,
                    CCSConfigurationService.convertBladeSetConfiguration(log, side, raw)
                );
            }
            catch (Exception exc) {
                log.error("Badly formed or missing configuration for blade set "
                          + side.toString() + ".", exc);
                return null;
            }
        }
        return result;
    }
    
    /**
     * Converts the property string for the Hall configuration into proper configuration objects.
     * @return The list of configuration objects, or null if any are missing or invalid.
     */
    public List<HallConfiguration> getHallConfigs() {
        // Run the input conversion engine on the appropriate property value,
        // then use the same conversion used for the shutter subsystem proper.
        try {
            final List<String> raw = (List<String>)engine.convertArgToType(
                bundle.getString("hall//config"),
                (new ArrayList<String>()).getClass()
            );
            return CCSConfigurationService.convertHallConfigurations(log, raw);
        }
        catch (Exception exc) {
            log.error("Badly formed or missing Hall configuration.", exc);
            return null;
        }
    }
    
    /**
     * Converts the property string for the general shutter configuration into a proper configuration object.
     * @return The configuration object, or null if it's missing or invalid.
     */
    public ShutterConfiguration getShutterConfig() {
        // Run the input conversion engine on the appropriate property value,
        // then use the same conversion used for the shutter subsystem proper.
        try {
            final Map<String, String> raw = (Map<String, String>)engine.convertArgToType(
                bundle.getString("shutter//config"),
                (new HashMap<String, String>()).getClass()
            );
            return CCSConfigurationService.convertShutterConfiguration(log, raw);
        }
        catch (Exception exc) {
            log.error("Badly formed or missing configuration for shutter.", exc);
            return null;
        }
    }
}
