package org.lsst.ccs.subsystems.shutter.parker;

import static org.lsst.ccs.subsystems.shutter.parker.ParkerConstants.*;

import org.lsst.ccs.subsystems.shutter.common.BladePosition;
import org.lsst.ccs.subsystems.shutter.common.BladeSet;
import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;
import org.lsst.ccs.subsystems.shutter.common.MovementHistory;

import org.lsst.ccs.drivers.iocard.AccesDio;

import org.lsst.ccs.drivers.parker.AcrComm;
import org.lsst.ccs.drivers.parker.LocalLongArray;
import org.lsst.ccs.drivers.parker.LocalSingleArray;

import static org.lsst.ccs.drivers.parker.EncoderLong.*;
import static org.lsst.ccs.drivers.parker.AxisBit.*;
import static org.lsst.ccs.drivers.parker.AxisSingle.*;
import static org.lsst.ccs.drivers.parker.AxisUnsigned.*;
import static org.lsst.ccs.drivers.parker.LocalLong.*;
import static org.lsst.ccs.drivers.parker.LocalLongArray.*;
import static org.lsst.ccs.drivers.parker.MasterBit.*;
import static org.lsst.ccs.drivers.parker.MasterName.*;
import static org.lsst.ccs.drivers.parker.SystemUnsigned.*;
import static org.lsst.ccs.drivers.parker.UserParameter.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.lsst.ccs.drivers.parker.AxisName;
import org.lsst.ccs.drivers.parker.EncoderName;
import static org.lsst.ccs.drivers.parker.ProgramBit.PROGRAM_RUNNING;
import static org.lsst.ccs.drivers.parker.ProgramUnsigned.PROGRAM_LINE_NUMBER;


/**
 ***************************************************************************
 **
 **  Routines to perform shutter blade operations.
 **
 **  <p>Definitely NOT thread-safe because any method might manipulate the hardware.
 **  Use should be confined to a single thread, or at least one at a time.
 **
 **  @author Owen Saxton
 **  @author tether
 **
 ***************************************************************************
 */
public final class BladeSetDrvr implements BladeSet {

   /**
    ***************************************************************************
    **
    **  Inner class to hold an item of raw Hall sensor data
    **
    ***************************************************************************
    */
    public static class HallItem {
        /** The time of the transition in nanoseconds (System.nanoTime()). */
        public long     time;

        /** The ID number of the sensor (0, 1, 2 ...). */
        public int      sensor;

        /** Is the sensor open, that is, is there no magnet close by? */
        public boolean  open;

    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold a set of raw Hall sensor data
    **
    ***************************************************************************
    */
    public static class Hall {

        /** The movement start time in nanoseconds (System.nanoTime()). */
        public long        startTime;

        /** The number of Hall transitions that took place during the move. Will
            be greater than data.length if we ran out of space for transitions.
        */
        public int         count;

        /** The current state of all the switches, one bit each. This is set at the start of the move
            and updated after every transition.
        */
        public int         state;

        /** All the HallItems collected so far. An array is used for speedy access. */
        public HallItem[]  data;

        /** Constructs an instance capable of holding up to 100 HallItems. */
        public Hall()
        {
            this(100);
        }

        /**
         * Constructs an instance that can hold a given number of HallItems.
         * @param maxItem the limit on the number of items
         */
        public Hall(int maxItem)
        {
            data = new HallItem[maxItem];
            for (int j = 0; j < maxItem; j++)
                data[j] = new HallItem();
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold an item of sampled data from the motor controller.
    **  A number of integer and float parameters may be sampled at the same time.
    **  The order of the parameter values in each array is implied by the sampling
    **  configuration; the parameters sampled are not otherwise identified.
    **
    ***************************************************************************
    */
    public static class Sample {
        /** The sample time in microseconds since the start of the move, based
            on the motor controller's clock.
        */
        public long      time;

        /** The array of sampled integer parameter values. */
        public long[]    iVals;

        /** The array of sampled float parameter values. */
        public double[]  fVals;
    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold a set of Sample instances along with some
    **  general info about the move.
    **
    ***************************************************************************
    */
    public static class SampData {

        /** The status of the move at completion. See {@link MovementStatus}.*/
        public int                  status;

        /** The move start time, a system time stamp in microseconds. */
        public long                 startTime;

        /** The absolute position in mm at the start of the move. */
        public double               startPosn;

        /** The move end time, a system time stamp in microseconds. */
        public long                 endTime;

        /** The absolute position in mm at the end of the move. */
        public double               endPosn;

        /** The collection of data samples from the controller. */
        public ArrayList<Sample >   sList = new ArrayList<>();
    }


   /*
    ***************************************************************************
    **
    **  Private fields and constructor.
    **
    ***************************************************************************
    */
    // The configuration for this blade set.
    private final BladeSetConfiguration config;

    // The configuration for the shutter as a whole.
    private final ShutterConfiguration shutterConfig;

    // The Hall sensor configuration.
    private final List<HallConfiguration> hallConfigs;

    // Driver for the Acces I/O card.
    private final AccesDio dio;

    // Driver for the Parker ACR motor controller.
    private final AcrComm acr;

    // Which side of the shutter is this blade set on?
    private final ShutterSide side;

    // Which axis does this object control?
    private final AxisName axis;

    // Which encoder is used by our axis.
    private final EncoderName encoder;


    /**
     * Saves the side of the camera X axis to which the blade set belongs,
     * save configuration info and hardware control objects. We
     * assume that AXISn uses ENCn and DACn.
     * @param side the side.
     * @param bsetConfig the blade set-specific configuration.
     * @param shutterConfig configuration info for the shutter in general.
     * @param hallConfigs the list of Hall configurations, one per sensor.
     * @param acr the object used to communicate with the ACR controller.
     * @param dio the interface to the digital I/O card.
     */
    public BladeSetDrvr(
        final ShutterSide side,
        final BladeSetConfiguration bsetConfig,
        final ShutterConfiguration shutterConfig,
        final List<HallConfiguration> hallConfigs,
        final AcrComm acr,
        final AccesDio dio)
    {
        this.side = side;
        this.config = bsetConfig;
        this.shutterConfig = shutterConfig;
        this.hallConfigs = hallConfigs;
        this.acr = acr;
        this.dio = dio;
        this.axis = AxisName.valueOf("AXIS" + config.getAxis());
        this.encoder = EncoderName.values()[axis.index()];
    }


    @Override
    /** {@inheritDoc} */
    public ShutterSide getSide()
    {
        return side;
    }


   /**
    ***************************************************************************
    **
    **  Moves the blade set to a given relative (i.e. fractional) position.
    **  Calls moveH() requesting S-curve motion with the sample count
    **  set from the configuration.
    **  @param relPosition the target relative position
    **  @param time the duration of the motion in seconds
    **  @return the resulting movement history
    **
    ***************************************************************************
    */
    @Override
    public MovementHistory moveToPosition(double relPosition, double time)
    {
        final double sampleDt = shutterConfig.getMinSampleDt()
            + time * shutterConfig.getSampleDtScale(); // Dt in ms
        final int numSamples = (int)Math.ceil(1000.0 * time / sampleDt);
        // The controller converts distance to encoder counts using a positive'
        // PPU, so we have to correct for the case where increasing
        // absolute position means decreasing the encoder count.
        final double flip = Math.signum(config.getEncoderCountsPerMm());
        return moveP(MOV_TYP_SCURVE,
                            flip*(relativeToAbsolute(relPosition) - getAbsolutePosition()),
                            time, numSamples);
    }


    @Override
    /** {@inheritDoc} */
    public double getRelativePosition()
    {
        return absoluteToRelative(getAbsolutePosition());
    }

    @Override
    /** {@inheritDoc} */
    public BladeSetConfiguration getBladeSetConfiguration()
    {
        return config;
    }


   /**
    ***************************************************************************
    **
    **  Gets the current absolute position in mm.
    **  @return the position
    **
    ***************************************************************************
    */
    public double getAbsolutePosition()
    {
        final long encval = acr.get(encoder, ENCODER_POSITION);
        return encoderToAbsolute(encval);
    }

   /**
    ***************************************************************************
    **
    **  Turns on the motor drive current.
    *
    ***************************************************************************
    */
    public void enableDrive()
    {
        acr.sendStr(axis.reference() + " drive on");
    }


   /**
    ***************************************************************************
    **
    **  Turns off the motor drive current.
    *
    ***************************************************************************
    */
    public void disableDrive()
    {
        acr.sendStr(axis.reference() + " drive off");
    }

   /**
    ***************************************************************************
    **
    **  Performs a simple move without enabling Hall transition interrupts
    **  or motor controller parameter sampling. We don't use the normal motion program
    **  in the controller, instead we issue the AXISn JOG INC command directly.
    **  @param type the move type, MOV_TYP_SCURVE or MOV_TYP_TRAP
    **  @param dist the signed distance to move (change of absolute position in mm)
    **  @param time the duration of the move in seconds
    **  @return the end-of-move status code (see {@link MovementStatus})
    **
    ***************************************************************************
    */
    public int move(int type, double dist, double time)
    {

        // Return if distance is zero.
        if (dist == 0.0) return 0;

        // Configure and enable the drive.
        configMove(type, dist, time);

        // Do the move and wait for motion to stop.
        acr.sendStr(axis.reference() + "jog inc " + dist);
        do {
            sleep(0.010);
        } while (acr.get(axis, JOG_ACTIVE));

        return getMoveStatus();
    }


   /**
    ***************************************************************************
    **
    **  Performs a move accompanied by parameter sampling in the motor controller.
    **  @param type see {@link #move(int, double, double, double)}
    **  @param dist see {@link #move(int, double, double, double)}
    **  @param time see {@link #move(int, double, double, double)}
    **  @param nSamp the number of times to perform sampling
    **  @param iParms the ID numbers of the integer parameters to record at each sampling
    **  @param fParms the ID numbers of the float parameters to record at each sampling
    **  @return a SampData filled with the sample data (if any)
    **
    ***************************************************************************
    */
    public SampData moveD(int type, double dist, double time,
                          int nSamp, int[] iParms, int[] fParms)
    {

        // Set the parameters used by the move program and start it
        // so that it may set up the motion and the sampling.
        int nsamp = initMove(type, dist, time, nSamp, iParms, fParms);
        SampData data = new SampData();
        data.startPosn = getAbsolutePosition();

        // Tell the move program that it may start the motion (and sampling)
        // Wait for motion and sampling to finish.
        data.startTime = runMove(nsamp <= 0, null);
        data.endTime = System.currentTimeMillis();
        data.endPosn = getAbsolutePosition();

        // Return if error or null move.
        if (nsamp <= 0) {
            data.status = -nsamp;
            return data;
        }

        // Retrieve the sampled parameters from local arrays of the move program.
        long[][] iVals = new long[iParms.length][];
        double[][] fVals = new double[fParms.length][];

        final long time0 = acr.get(MOVE_PROG, LV0);  // Scalar lv0: motion start time
        final LocalLongArray[] la = LocalLongArray.values();
        for (int j = 0; j < iParms.length; j++)
            iVals[j] = acr.get(MOVE_PROG, la[j]); // Array la<j>[], samples of j'th int param.
        final LocalSingleArray[] sa = LocalSingleArray.values();
        for (int j = 0; j < fParms.length; j++)
            fVals[j] = acr.get(MOVE_PROG, sa[j]); // Array sa<j>[], samples of j'th float param.

        // Set movement completion status.
        data.status = getMoveStatus();

        // Get the index of sampled time; -1 if time is not being sampled.
        int tIndx = -1;
        for (int j = 0; j < iParms.length; j++)
            if (iParms[j] == SYSTEM_CLOCK.index()) tIndx = j;

        // Create a set of Sample objects where each contains all the
        // parameter values sampled at the same time (have same second
        // index in iVals/fVals). If one of the sampled parameters
        // was the system clock then convert its samples from
        // absolute controller time to time since start of motion, and
        // change the time unit from milliseconds to microseconds.
        for (int j = 0; j < nsamp; j++) {
            Sample sItem = new Sample();
            sItem.time = 0;
            sItem.iVals = new long[iParms.length];
            for (int k = 0; k < iParms.length; k++) {
                sItem.iVals[k] = iVals[k][j];
                if (k == tIndx) {
                    sItem.iVals[k] -= time0;         // Subtract start time
                    sItem.time = 1000 * sItem.iVals[k]; // millisec to microsec
                }
            }
            sItem.fVals = new double[fParms.length];
            for (int k = 0; k < fParms.length; k++)
                sItem.fVals[k] = fVals[k][j];
            data.sList.add(sItem);
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Performs a move using the standard motion program, recording
    **  periodic data samples in the motor controller. Sets the motor controller to record
    **  a given number of samples at fixed intervals. In each sample: controller millisecond
    **  clock, motor encoder value, axis status flags (not used).
    **  @param type see {@link #move(int, double, double, double)}
    **  @param dist see {@link #move(int, double, double, double)}
    **  @param time see {@link #move(int, double, double, double)}
    **  @param nSamp the number of times to perform sampling
    **  @return a movement history with an empty list of Hall transitions
    ***************************************************************************
    */
    public MovementHistory moveP(int type, double dist, double time, int nSamp)
    {

        // Set the parameters used by the move program and run it.
        final int[] iParms = {SYSTEM_CLOCK.index(), ENCODER_POSITION.index(encoder),
                        PRIMARY_AXIS_FLAGS.index(axis)},
              fParms = {};
        final double sPosition = getRelativePosition();
        final int nsamp = initMove(type, dist, time, nSamp, iParms, fParms);
        final long sTime = 1000 * runMove(nsamp <= 0, null);
        final long eTime = 1000 * System.currentTimeMillis();

        // Return if error or null move.
        if (nsamp <= 0) {
            return new MovementHistory(
                getSide(),
                -nsamp,
                sTime,
                getRelativePosition(),
                eTime,
                getRelativePosition(),
                Collections.emptyList(),
                Collections.emptyList()
            );
        }

        // Get the sampled positions and times.
        List<BladePosition> pList = new ArrayList<>();
        int status = getPositionData(sTime, nsamp, pList);
        if (status == 0) status = getMoveStatus();

        return new MovementHistory(
            getSide(),
            status,
            sTime,
            sPosition,
            eTime,
            getRelativePosition(),
            pList,
            Collections.emptyList()
        );
    }

   /**
    ***************************************************************************
    **
    **  Gets the configuration object being used.
    **  @return the configuration
    ***************************************************************************
    */
    public BladeSetConfiguration getConfig()
    {
        return config;
    }

    /** Gets the axis controlled by this object. */
    public AxisName getAxis() {return axis;}

    /** Gets the encoder our axis uses for motor position feedback. */
    public EncoderName getEncoder() {return encoder;}

   /**
    ***************************************************************************
    **
    **  Gets the AccesDio object being used.
    **  @return the I/O card object
    ***************************************************************************
    */
    public AccesDio getDio()
    {
        return dio;
    }


   /**
    ***************************************************************************
    **
    **  Gets the motor communications object being used.
    **  @return the AcrComm object
    ***************************************************************************
    */
    public AcrComm getComm()
    {
        return acr;
    }


   /**
    ***************************************************************************
    **
    **  Gets the status of the last move from the motor controller: normal
    **  or premature stop reason.
    **  @return 0 for success or a code from {@link MovementStatus}
    ***************************************************************************
    */
    public int getMoveStatus()
    {
        // If a motion failed for some reason the kill-all-motion bit is
        // set for the axis which in turn sets kill-all-moves for the
        // master to which the axis is attached.
        if (!acr.get(MASTER0, KILL_ALL_MOVES))
            return 0;

        // We failed! Find out why.

        if (acr.get(axis, POSITIONING_ERROR))
            return MovementStatus.STS_EXCS_POSN_ERROR;

        if (acr.get(axis, HIT_POSITIVE_HARD_LIMIT))
            return MovementStatus.STS_POS_EOT_LIMIT;

        if (acr.get(axis, HIT_NEGATIVE_HARD_LIMIT))
            return MovementStatus.STS_NEG_EOT_LIMIT;

        if (acr.get(axis, HIT_POSITIVE_SOFT_LIMIT))
            return MovementStatus.STS_POS_SOFT_LIMIT;

        if (acr.get(axis, HIT_NEGATIVE_SOFT_LIMIT))
            return MovementStatus.STS_NEG_SOFT_LIMIT;

        return MovementStatus.STS_UNKN_MOVE_TRUNC;
    }


   /**
    ***************************************************************************
    **
    **  Gets the drive temperature.
    **  @return the temperature in degrees Celsius
    **
    ***************************************************************************
    */
    public double getDriveTemperature()
    {
        return acr.get(axis, DRIVE_TEMPERATURE);
    }

   /**
    ***************************************************************************
    **
    **  Retrieve position samples from the standard move program
    **  in the motor controller. The position samples stored in the
    **  program's arrays are used only up and including to the point when
    **  the "jog active" axis status flag becomes to zero. Times are converted
    ** to microseconds since start of move.
    **  @param sTime the move start time, a system time stamp in microseconds
    **  @param nsamp the dimension of each of the program's sample arrays la0, la1, la2
    **  @param pList the list to which to add the BladePosition instances
    **
    ***************************************************************************
    */
    private int getPositionData(long sTime, int nsamp,
                                List<BladePosition> pList)
    {

        /*
        **  Retrieve the sampled positions and times
        */
        long time0 = acr.get(MOVE_PROG, LV0);
        long[] times = acr.get(MOVE_PROG, LA0);
        long[] posns = acr.get(MOVE_PROG, LA1);
        long[] flags = acr.get(MOVE_PROG, LA2);
        if (times.length+posns.length+flags.length < 3 * nsamp)
            return MovementStatus.STS_MSSG_PGM_VBLES;

        /*
        **  Create the returned sampled values
        */
        for (int j = 0; j < nsamp; j++) {
            final double posn = encoderToAbsolute(posns[j]);
            pList.add(
                new BladePosition(
                    sTime + 1000 * (times[j] - time0),
                    posn,
                    absoluteToRelative(posn)
                )
            );
            if ((flags[j] & JOG_ACTIVE.flagParameterMask()) == 0) break;
        }

        return 0;
    }

   /**
    ***************************************************************************
    **
    **  Configures the motor controller for a move with no sampling, such as
    *   performed by {@link #move(int, double, double, double)}, which see
    **  for a description of the parameters. The following motor controller parameters are set:
    **  <ul>
    **  <li>Pulses per unit (mm)</li>
    **  <li>Jog velocity limit, mm/sec</li>
    **  <li>Jog acceleration limit, mm/sec**2</li>
    **  <li>Jog deceleration limit, mm/sec**2</li>
    **  <li>Jog jerk, mm/sec**3. Non-zero only for S-curve motion</li>
    **  </ul>
    **  The motor drive is switched on if it was off.
    **
    ***************************************************************************
    */
    private void configMove(int type, double dist, double time)
    {
        double vel, acc, jrk = 0.0;
        vel = Math.abs(2.0 * dist / time);
        acc = Math.abs(4.0 * dist / (time * time));
        if (type == MOV_TYP_SCURVE) {
            acc *= 2.0;
            jrk = acc * acc / vel;
        }
        acr.set(axis, PULSES_PER_UNIT, Math.abs(config.getEncoderCountsPerMm()));
        acr.set(axis, JOG_VELOCITY_SETTING, vel);
        acr.set(axis, JOG_ACCELERATION_SETTING, acc);
        acr.set(axis, JOG_DECELERATION_SETTING, acc);
        acr.set(axis, JOG_JERK_SETTING, jrk);
        enableDrive();
    }


   /**
    ***************************************************************************
    **
    **  Sets parameters for the standard move program. The jog motion profile
    **  parameters will be set directly. The move target and sampling
    **  information are set in the user parameters for use by the
    **  motion program.
    **  <p>
    **  See {@link #moveD(int, double, double, double, int, int[], int[])}
    **  for a description of the parameters.
    **
    ***************************************************************************
    */
    private int initMove(int type, double dist, double time,
                         int nSamp, int[] iParms, int[] fParms)
    {
        double[] parms = new double[P_FIRST_PARAM_NUMBER.index() + iParms.length + fParms.length];
        double vel, acc, jrk = 0.0;

        /*
        **  Return if distance is zero
        */
        if (dist == 0.0) return 0;

        /*
        **  Set up the parameter values used by the move program
        */
        parms[P_AXIS_NUMBER.index()] = axis.index();
        parms[P_MOVE_DISTANCE.index()] = dist;
        parms[P_PROGRESS.index()] = PCD_NEVER_STARTED;
        vel = Math.abs(2 * dist / time);
        acc = Math.abs(4 * dist / (time * time));
        if (type == MOV_TYP_SCURVE) {
            acc *= 2.0;
            jrk = acc * acc / vel;
        }
        if (nSamp > 0) {
            parms[P_SAMPLE_INTERVAL.index()] = Math.floor(1000.0 * time / nSamp);
            parms[P_NUM_SAMPLES.index()] = Math.floor(1000.0 * time / parms[P_SAMPLE_INTERVAL.index()] + 5.0);
        }
        else {
            parms[P_SAMPLE_INTERVAL.index()] = 1.0;
            parms[P_NUM_SAMPLES.index()] = 1.0;
        }
        parms[P_NUM_INT_PARAMS.index()] = iParms.length;
        parms[P_NUM_FLOAT_PARAMS.index()] = fParms.length;
        for (int j = 0; j < iParms.length; j++)
            parms[P_FIRST_PARAM_NUMBER.index() + j] = iParms[j];
        for (int j = 0; j < fParms.length; j++)
            parms[P_FIRST_PARAM_NUMBER.index() + iParms.length + j] = fParms[j];

        /*
        **  Check that there's enough space for samples
        */
        int nWord = 13 + ((int)parms[P_NUM_SAMPLES.index()] + 2)
                            * (iParms.length + fParms.length);
        if (nWord > 1024)
            return -MovementStatus.STS_INSUFF_VBLE_MEM;

        // Set the axis motion parameters in the controller.
        acr.set(axis, PULSES_PER_UNIT, Math.abs(config.getEncoderCountsPerMm()));
        acr.set(axis, JOG_VELOCITY_SETTING, vel);
        acr.set(axis, JOG_ACCELERATION_SETTING, acc);
        acr.set(axis, JOG_DECELERATION_SETTING, acc);
        acr.set(axis, JOG_JERK_SETTING, jrk);
        // Set the motion program's arguments in the user parameters.
        acr.set(USER0, parms);

        return (int)parms[P_NUM_SAMPLES.index()];
    }


   /**
    ***************************************************************************
    **
    **  Carries out the move with parameter sampling.
    *   <p>Precondition: The parameters for the move have been stored in the
    **  controller, including the user parameters used as arguments for the motion
    **  program.
    **  <p>Postcondition:The motion will have stopped and all samples will have been
    **  collected.
    **
    ***************************************************************************
    */
    private long runMove(boolean dummy, Hall hall)
    {
        /*
        **  Get the start times, standard and high-resolution
        */
        long sTime = System.currentTimeMillis();
        if (hall != null)
            hall.startTime = 1000 * sTime - System.nanoTime() / 1000;
        if (dummy) return sTime;

        acr.sendStr("run " + MOVE_PROG.reference());

        // Wait for the program to stop running. If not stopped by an
        // error, the motion program will exit only after motion has stopped
        // and all samples have been taken.
        do {
            sleep(0.1);
        } while(acr.get(MOVE_PROG, PROGRAM_RUNNING));

        // Examine the progress code set by the motion program.
        final int progress = (int)acr.get(P_PROGRESS);
        final int lastLine = (int)acr.get(MOVE_PROG, PROGRAM_LINE_NUMBER);
        if (progress != PCD_DONE) {
            throw new IllegalStateException(
                String.format(
                    "Motion program execution was incomplete. Progress code = %d, last line = %d.",
                    progress, lastLine));
        }
        return sTime;
    }


   /**
    ***************************************************************************
    **
    **  Sleeps for a time
    **
    ***************************************************************************
    */
    private void sleep(double secs)
    {
        long nsecs = (long)(1e9 * secs);
        try {
            Thread.sleep(nsecs / 1000000, (int)(nsecs % 1000000));
        }
        catch (java.lang.InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }


    /**
     * Converts a motor position encoder value to an absolute position in mm of the leading
     * edge of the blade set on the camera X axis. The formula used is
     * {@code position = (encoderValue - offset) / countsPerMm},
     * where {@code offset} is the value the encoder has at {@code absolute position == 0.0} and
     * {@code countsPerMm} is the number of encoder counts per mm, which will be negative
     * if the the direction of increasing encoder values is the direction of decreasing
     * absolute position.
     *
     * @param encoderValue the value read from the encoder.
     * @return
     */
    public double encoderToAbsolute(double encoderValue) {
        return (encoderValue - config.getEncoderOffset()) / config.getEncoderCountsPerMm();
    }

    /** {@inheritDoc} */
    @Override
    public double absoluteToEncoder(double absolute) {
        return absolute * config.getEncoderCountsPerMm() + config.getEncoderOffset();
    }

    /** {@inheritDoc} */
    @Override
    public double absoluteToRelative(double absolute) {
        return (absolute - config.getOpenPos())
            / (config.getClosedPos() - config.getOpenPos());
    }

    /** {@inheritDoc} */
    @Override
    public double relativeToAbsolute(double relative) {
        return config.getOpenPos() + relative * (config.getClosedPos() - config.getOpenPos());
    }

    /** {@inheritDoc} */
    @Override
    public boolean isFullyExtended() {
        return Math.abs(getAbsolutePosition() - config.getClosedPos())
            <= shutterConfig.getPosTolerance();
    }

    /** {@inheritDoc} */
    @Override
    public boolean isFullyRetracted() {
        return Math.abs(getAbsolutePosition() - config.getOpenPos())
            <= shutterConfig.getPosTolerance();
    }
}
