package org.lsst.ccs.subsystems.shutter;

import org.lsst.ccs.subsystems.shutter.common.BladeSet;
import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ConfigurationService;
import org.lsst.ccs.subsystems.shutter.common.MovementHistory;
import static org.lsst.ccs.subsystems.shutter.common.ShutterSide.*;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;
import org.lsst.ccs.subsystems.shutter.status.BladePositionResult;
import org.lsst.ccs.subsystems.shutter.status.CloseShutterStatus;
import org.lsst.ccs.subsystems.shutter.status.OpenShutterStatus;
import org.lsst.ccs.subsystems.shutter.status.MoveToPositionStatus;
import org.lsst.ccs.subsystems.shutter.status.MovementHistoryStatus;
import org.lsst.ccs.subsystems.shutter.status.ReadyForActionStatus;
import org.lsst.ccs.subsystems.shutter.status.StatusKey;
import org.lsst.ccs.subsystems.shutter.status.TakeExposureStatus;
import org.lsst.ccs.subsystems.shutter.status.UnsafeMoveStatus;


import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.NORMAL;


import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.utilities.logging.Logger;

import java.io.Serializable;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;
import org.lsst.ccs.subsystems.shutter.common.BladePosition;
import org.lsst.ccs.subsystems.shutter.common.ShutterController;

/**
 * Main module for the shutter subsystem.
 *
 * <p>This module manages two objects that implement the BladeSet interface, one for each side of the
 * shutter. Generally these will either both be instances of BladeSetSimulator or of BladeSetDrvr.
 * Coordinated motions of one or both the blade sets are carried out in response to commands from the
 * CCS command bus. Motion histories for the blade sets are published on the CCS status bus.
 *
 * @author azemoon
 * @author tether
 */
public class ShutterMain extends Module {

    private volatile Map<ShutterSide, BladeSet> bladeSets = null;
    private final Logger mylog = Logger.getLogger(ShutterMain.class.getPackage().getName());

    @Resource(name="configService")
    private volatile ConfigurationService configService = null;

    @Resource(name="controller")
    private volatile ShutterController controller = null;

    private Map<ShutterSide, BladeSetConfiguration> bsetConfigs = null;
    private List<HallConfiguration> hallConfigs = null;
    private ShutterConfiguration shutterConfig = null;

    /**
     * Constructs the main module for the shutter subsystem.
     * @param name the component name (module name)
     * @param bladeSets a map of BladeSet objects, one per shutter side.
     */
    public ShutterMain(String name)
    {
        super(name);
    }

    /**
     * Requires that both blade sets controllers are present and that all required configuration
     * is present and checked.
     */
    @Override
    public void initModule() {
        mylog.info("Initializing the Shutter module ");
    }

    /**
     * Perform hardware initialization.
     */
    @Override
    public void start() {
        mylog.info("Starting the Shutter module ");

        // Check the configuration data.
        bsetConfigs = configService.getBladeSetConfigurations(mylog);
        shutterConfig = configService.getShutterConfiguration(mylog);
        hallConfigs = configService.getHallConfigurations(mylog);

        if (bsetConfigs==null || shutterConfig==null || hallConfigs==null) {
            logFatalMessageAndThrowException("Bad configuration.");
        }
        try {
            controller.init(bsetConfigs, shutterConfig, hallConfigs);
        }
        catch (java.io.IOException exc) {
            throw new RuntimeException(exc);
        }
        bladeSets = controller.getBladeSets();
    }

    private final static ScheduledExecutorService exec = Executors.newSingleThreadScheduledExecutor();

    /**
     * Shut down activity in this module. It uses an executor service which must be shut down.
     */
    @Override
    public void shutdownNow() { super.shutdownNow(); exec.shutdown();}

    /**
     * Gets one of the two BladeSet objects.
     * @param side selects the desired BladeSet
     * @return the selected BladeSet object
     */
    public final BladeSet getBladeSet(ShutterSide side) {
        return bladeSets.get(side);
    }

    /**
     * The default value for the number of seconds each blade set should take to move
     * when taking an exposure.
     * @return the movement time
     */
    public double getBladeMovementTime() {
        return shutterConfig.getMoveTime();
    }

    /**
     * Gets the tolerance used when comparing blade set position against a known position.
     * @return The tolerance value.
     */
    public double getPosTolerance(ShutterSide side) {
        return shutterConfig.getPosTolerance();
    }

    /**
     * Gets the current minimum approach distance for the blade sets (absolute position).
     * @return The min. distance.
     */
    public double getMinApproach() {
        return shutterConfig.getMinApproach();
    }

    /**
     * Gets the current relative position of both blade sets. See {@link BladePosition}. May
     * cause hardware operations.
     * @return the positions
     */
    @Command(type=ACTION,
        level=NORMAL,
        description="Gets the current relative positions of both blade sets")
    public BladePositionResult getBladeSetPositions()
    {
        return new BladePositionResult(
            getBladeSet(PLUSX).getRelativePosition(),
            getBladeSet(MINUSX).getRelativePosition()
        );
    }

    /**
     * Action command to move a single blade set.
     * <p>Engineering mode only. Blocks until movement is complete.
     * @param side           The side of the blade set to move.
     * @param targetPosition  The desired relative position after the move.
     */
    @Command(type=ACTION, level=ENGINEERING1, description="Moves one of the blade sets")
    public void moveToPosition(
            @Argument(name="side", description="Selects the blade set (MINUSX or PLUSX)")
            final ShutterSide side,
            @Argument(name="targetPosition", description="The desired relative position of the blade set")
            final double targetPosition)
    {
        // If we're going to perform an arbitrary blade set movement commanded by
        // some outside agency then we'll have to check safety.
        if (moveIsUnsafe(side, targetPosition)) {
            publish(StatusKey.UNSAFE_MOVE, new UnsafeMoveStatus());
        }
        else {
            publish(StatusKey.MOVE_BLADE_SET,
                    new MoveToPositionStatus(getBladeSet(side).getRelativePosition(),
                                           side,
                                           targetPosition,
                                           getBladeMovementTime()
                    )
            );
            final MovementHistory hist =
                getBladeSet(side).moveToPosition(targetPosition, getBladeMovementTime());
            publish(StatusKey.MOVEMENT, new MovementHistoryStatus(true, true, hist));
        }
        publish(StatusKey.READY, new ReadyForActionStatus(true));
    }

    /**
     * Perform two blade set motions with a given delay between them.
     * @param firstSide which blade set to move first.
     * @param firstTargetPosition where to move the first blade set
     * @param secondTargetPosition where to move the second blade set
     * @param secondStartDelay the delay in seconds between starting the firt move and the second.
     *
     * Explicit scheduling is always needed when simulating the shutter since the simulation uses the system
     * clock time as the starting time. For a real shutter it won't hurt and is sometimes needed.
     */
    private void moveTwoBladeSets(
        ShutterSide firstSide,
        double firstTargetPosition,
        double secondTargetPosition,
        double secondStartDelay)
    {
        final BladeSet first = getBladeSet(firstSide);
        final BladeSet second = getBladeSet(firstSide.opposite());
        final Future<MovementHistory> fuhist1 =
            exec.schedule(
                () -> first.moveToPosition(firstTargetPosition, getBladeMovementTime()),
                0,
                TimeUnit.MILLISECONDS);
        final Future<MovementHistory> fuhist2 =
            exec.schedule(
                () -> second.moveToPosition(secondTargetPosition, getBladeMovementTime()),
                Math.round(1000.0 * secondStartDelay),
                TimeUnit.MILLISECONDS);
        // Wait for the movements to complete.
        try {
            final MovementHistory hist1 = fuhist1.get();
            publish(StatusKey.MOVEMENT, new MovementHistoryStatus(true, false, hist1));
            final MovementHistory hist2 = fuhist2.get();
            publish(StatusKey.MOVEMENT, new MovementHistoryStatus(false, true, hist2));
        }
       catch (InterruptedException e) {
            // This thread was interrupted while waiting. Cancel the tasks.
           fuhist1.cancel(true);
           fuhist2.cancel(true);
       }
        catch (ExecutionException e) {
            // A task threw an exception, obtainable using getCause().
            fuhist1.cancel(true);
            fuhist2.cancel(true);
            launderThrowable(e.getCause());
        }
        finally {
            publish(StatusKey.READY, new ReadyForActionStatus(true));
        }
   }
    /**
     * An action command that opens the shutter for the given exposure time then closes it.
     * Normal and engineering mode. Blocks until the exposure is complete.
     * @param exposureTime the exposure time in seconds
     */
    @Command(type=ACTION, level=NORMAL, description="Performs a timed exposure (open, wait, close)")
    public void takeExposure(
            @Argument(name="exposureTime", description="The duration of the exposure in seconds")
            final double exposureTime)
    {
        final BladePositionResult pos = getBladeSetPositions();
        final ShutterSide firstSide = (pos.getPlusx() > pos.getMinusx()) ? PLUSX : MINUSX;
        publish(StatusKey.TAKE_EXPOSURE,
            new TakeExposureStatus(firstSide, getBladeMovementTime(), exposureTime));
        moveTwoBladeSets(firstSide, 0.0, 1.0, exposureTime);
    }

    /**
     * Closes the shutter. First blade set MINUSX is retracted and then set PLUSX is extended.
     */
   @Command(type=ACTION, level=ENGINEERING1, description="Closes the shutter")
   public void closeShutter() {
       publish(StatusKey.CLOSE_SHUTTER, new CloseShutterStatus(MINUSX, getBladeMovementTime()));
       moveTwoBladeSets(MINUSX, 0.0, 1.0, getBladeMovementTime());
   }

   /**
    * Opens the shutter. First blade set MINUSX is retracted, then set PLUSX.
    */
   @Command(type=ACTION, level=ENGINEERING1, description="Opens the shutter")
   public void openShutter() {
       publish(StatusKey.OPEN_SHUTTER, new OpenShutterStatus(MINUSX, getBladeMovementTime()));
       moveTwoBladeSets(MINUSX, 0.0, 0.0, getBladeMovementTime());
   }

   /**
    * Does nothing at present.
    */
   @Command(type=ACTION, level=ENGINEERING1, description="Performs a shutter motion calibration")
   public void calibrate() {
       publish(StatusKey.READY, new ReadyForActionStatus(true));
   }

   /**
    * Throws any given RuntimeException or Error, else throws IllegalStateException.
    * Used to handle arbitrary Throwables wrapped inside instances of ExcecutionException.
    * Any that are checked exceptions are wrapped in an IllegalStateException, which is unchecked.
    * @param thr The Throwable from an ExecutionException.
    */
   private static void launderThrowable(Throwable thr) {
       if (thr instanceof RuntimeException)
           throw (RuntimeException)thr;
       else if (thr instanceof Error)
           throw (Error)thr;
       else
           throw new IllegalStateException("Not unchecked", thr);
   }

    private void logFatalMessageAndThrowException(String msg) {
        mylog.fatal(msg);
        throw new RuntimeException(msg);
    }

    /** Publishes subsystem-specific  data to the status bus with a key derived from an enum.
        @param key One of the StatusKey enumerators.
        @param status The serializable object to be published.
    */
    private void publish(StatusKey key, Serializable status) {
        getSubsystem()
            .publishSubsystemDataOnStatusBus(new KeyValueData(key.getKey(), status));
    }

    /**
     * Determines if a commanded move of a single blade set is unsafe. The two blade sets
     * must not end up being too close.
     * @param side which blade set to is being moved.
     * @param targetPosition the new position of the blade set (relative).
     * @return the answer
     */
    private boolean moveIsUnsafe(ShutterSide side, double targetPosition) {
        final double abs1 = getBladeSet(side).relativeToAbsolute(targetPosition);
        final BladeSet opp = getBladeSet(side.opposite());
        final double abs2 = opp.relativeToAbsolute(opp.getRelativePosition());
        return Math.abs(abs2 - abs1) < getMinApproach();
    }

}
