package org.lsst.ccs.subsystems.shutter;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;

/**
 * Checks configuration data against a reference for names and types. A set of configuration data,
 * represented as a {@literal Map<String, String>} will be checked against a reference
 * {@literal Constructor}.  Where the constructor specifies int/long or Integer/Long,
 * {@code Integer/Long.decode(String)} is used to check and convert. For Double values
 * {@code Double.valueOf(String) is used. For enums, the string is converted to
 * uppercase and then appropriate {@code valueOf(String)} method is used.
 *
 * @author tether
 */
public final class ConfigDataChecker<T> {

    private final Constructor<T> construct;
    private final List<Parameter> params;

    /**
     * Constructor.
     * @param reference The reference constructor against which to check a configuration.
     */
    public ConfigDataChecker(final Constructor<T> construct) {
        this.construct = construct;
        this.params = Arrays.asList(construct.getParameters());
    }

    /**
     * Checks for missing data by key.
     * @param config The configuration data to be checked.
     * @return A list of the names of missing items.
     */
    public List<String> checkAllPresent(final Map<String, String> config) {
        return params
            .stream()
            .map(Parameter::getName)
            .filter(x -> !config.containsKey(x))
            .collect(Collectors.toList());
    }

    /**
     * Checks that configuration data items have the right types. It's assumed that
     * no items are missing.
     * @see #checkAllPresent(java.util.Map) 
     * @param config The configuration data to check.
     * @return A list of error messages for items that didn't check out.
     */
    public List<String> checkAllTypes(final Map<String, String> config) {
        return params
            .stream()
            .map(p -> checkType(p.getName(), p.getType(), config.get(p.getName())))
            .filter(x -> (x != null))
            .collect(Collectors.toList());
    }

    private String checkType(final String key, final Class<?> type, final String datum) {
        String error = null;
        if (type.equals(int.class) || type.equals(Integer.class)) {
            try {Integer.decode(datum.trim());}
            catch(NumberFormatException exc) {error = key + ": invalid int ";}
        }
        else if (type.equals(long.class) || type.equals(Long.class)) {
            try {Long.decode(datum.trim());}
            catch(NumberFormatException exc) {error = key + ": invalid long ";}
        }
        else if (type.equals(double.class) || type.equals(Double.class)) {
            try {Double.valueOf(datum.trim());}
            catch(NumberFormatException exc) {error = key + ": invalid double ";}
        }
        else if (type.equals(String.class)) {
            // No need to check.
        }
        else if (type.isEnum()) {
            final boolean match
                = Arrays.stream(type.getEnumConstants())
                .map(x -> x.toString())
                .filter(x -> x.equals(datum.toUpperCase().trim()))
                .findAny()
                .isPresent();
            if (!match) {
                error = key + ": invalid for enum " + type.getSimpleName();
            }
        } else {
            error = key + ": conversion to " + type.getSimpleName() + " isn't implemented";
        }
        return error;
    }
    
    /**
     * Converts the set of configuration data items to an object of type {@literal T},
     * provided that none are missing or invalid.
     * Items that are not required to be of enum type are passed along unchanged; enum strings
     * are converted with the correct {@literal valueOf()} method.
     * @see #checkAllPresent(java.util.Map) 
     * @see #checkAllTypes(java.util.Map) 
     * @param config The configuration data to convert.
     * @return The list of converted items. 
     */
    public T convertAll(final Map<String, String> config) {
        // First assemble an array of the actual parameters after any
        // conversions have been performed.
        final Object[] actuals
            = params.stream()
            .map(p -> {
                final String key = p.getName();
                final Class<?> type = p.getType();
                String datum = config.get(key);
                Object converted = null;
                if (type.equals(int.class) || type.equals(Integer.class)) {
                    try {
                        converted = Integer.decode(datum.trim());
                    } catch (NumberFormatException exc) {
                        throw new Error(exc);
                    }
                } else if (type.equals(long.class) || type.equals(Long.class)) {
                    try {
                        converted = Long.decode(datum.trim());
                    } catch (NumberFormatException exc) {
                        throw new Error(exc);
                    }
                } else if (type.equals(double.class) || type.equals(Double.class)) {
                    try {
                        converted = Double.valueOf(datum.trim());
                    } catch (NumberFormatException exc) {
                        throw new Error(exc);
                    }
                } else if (type.equals(String.class)) {
                    converted = datum;
                } else if (type.isEnum()) {
                    try {
                        final Method m = type.getDeclaredMethod("valueOf", String.class);
                        converted = m.invoke(type, datum.toUpperCase().trim());
                    } catch (NoSuchMethodException | IllegalAccessException | InvocationTargetException exc) {
                        throw new Error(exc);
                    }
                }
                return converted;
            })
            .collect(Collectors.toList())
            .toArray();
        // Call the constructor with the actual arguments.
        try {
            return construct.newInstance(actuals);
        } catch (IllegalAccessException | InstantiationException | InvocationTargetException exc) {
            throw new Error(exc);
        }
    }

}
