package org.lsst.ccs.subsystems.shutter;

import org.lsst.ccs.subsystems.shutter.common.ConfigurationService;

import org.lsst.ccs.framework.ConfigurableComponent;

import org.lsst.ccs.utilities.conv.InputConversionEngine;
import org.lsst.ccs.utilities.conv.TypeConversionException;

import org.lsst.ccs.utilities.logging.Logger;

import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;

import java.util.Collections;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Resource;

/**
 * Implements interface ConfigurationService using the CCS Configuration API. An instance
 * of this will be created by the subsystem build script and named "configService". It draws its
 * data from the components named "shutter", "minusx", "plusx" and "hall".
 * @author tether
 */
public final class CCSConfigurationService
    extends ConfigurableComponent
    implements ConfigurationService 
{

    @Resource(name="shutter")
    private volatile ShutterCCSConfig shutterConfig;
    
    @Resource(name="hall")
    private volatile HallCCSConfig hallConfig;
    
    @Resource(name="minusx")
    private volatile BladeSetCCSConfig minusxConfig;
    
    @Resource(name="plusx")
    private volatile BladeSetCCSConfig plusxConfig;
    

    public CCSConfigurationService() {
        super();
    }

    /** {@inheritDoc   */
    @Override
    public List<HallConfiguration> getHallConfigurations(final Logger log) {
        return convertHallConfigurations(log, hallConfig.getConfiguration());
    }

    /** {@inheritDoc   */
    @Override
    public Map<ShutterSide, BladeSetConfiguration> getBladeSetConfigurations(final Logger log) {
        final BladeSetConfiguration plus =
            convertBladeSetConfiguration(log, ShutterSide.PLUSX, plusxConfig.getConfiguration());
        final BladeSetConfiguration minus =
            convertBladeSetConfiguration(log, ShutterSide.PLUSX, minusxConfig.getConfiguration());
        if (plus == null || minus == null) {
            return null;
        }
        else {
            final EnumMap<ShutterSide, BladeSetConfiguration> map = new EnumMap<>(ShutterSide.class);
            map.put(ShutterSide.PLUSX, plus);
            map.put(ShutterSide.MINUSX, minus);
            return Collections.unmodifiableMap(map);
        }
    }

    /** {@inheritDoc   */
    @Override
    public ShutterConfiguration getShutterConfiguration(final Logger log) {
        return convertShutterConfiguration(log, this.shutterConfig.getConfiguration());
    }
    
    
    private static InputConversionEngine engine = new InputConversionEngine();
    
    private static ConfigDataChecker<HallConfiguration> hallChecker = 
        new ConfigDataChecker<>(HallConfiguration.getConstructor());
    
    /** Performs checking and conversion for raw Hall configuration data. This is exposed
     * so that one can use either what's provided by the CCS configuration API or
     * use the result of running the input conversion engine on a properties file.
     * @param log used for recording errors.
     * @param raw the output of the conversion engine or the configuration API.
     * @return either null if there were any errors, or the list of converted configuration items.
     */
    public static List<HallConfiguration>
    convertHallConfigurations(final Logger log, final List<String> raw) {
        List<HallConfiguration> result = null;
        final int[] errorCount = new int[]{0};
        if ((raw == null) || (raw.isEmpty())) {
            log.error("Missing Hall configuration.");
        }
        else {

            // Use InputConversionEngine to split each string into a Map<String, String>.
            // Use null as a placeholder for failures.
            List<Map<String, String>> listOfMaps
                = raw.stream()
                .map(r -> {
                    try {
                        return (Map<String, String>) engine.convertArgToType(
                            r,
                            (new HashMap<String, String>()).getClass());
                    } catch (TypeConversionException exc) {
                        log.error(String.format("Hall configuration is not well formed: %s", r), exc);
                        ++errorCount[0];
                        return null; // Need to preserve position in list for ident assignment below.
                    }
                })
                .collect(Collectors.toList());
            // Add an "ident" entry to each map giving its position in the list.
            final int[] ident = new int[]{0};
            listOfMaps.stream()
                .forEach(m -> {
                    final int i = ident[0]++;
                    if (m != null) m.put("ident", "" + i);
                });
            // Check that each map has a complete set of entries.
            listOfMaps = listOfMaps.stream()
                .filter(m -> m != null)
                .map(m -> {
                    final List<String> missing =
                        hallChecker.checkAllPresent(m);
                    if (missing.isEmpty()) {
                        return m;
                    }
                    else {
                        final String msg = 
                            missing.stream()
                            .collect(Collectors.joining(
                                "\n    ",
                                String.format("Missing parameters for Hall config[%s].\n    ", m.get("ident")),
                                "\n"
                            ));
                        log.error(msg);
                        ++errorCount[0];
                        return null;
                    }
                })
                .filter(m -> m != null)
                .collect(Collectors.toList());
            // Check that each map value is convertible to the desired type.
            listOfMaps = listOfMaps.stream()
                .map(m -> {
                    final List<String> wrong =
                        hallChecker.checkAllTypes(m);
                    if (wrong.isEmpty()) {
                        return m;
                    }
                    else {
                        final String msg = 
                            wrong.stream()
                            .collect(Collectors.joining(
                                "\n    ",
                                String.format("Wrong data types for Hall config[%s].\n    ", m.get("ident")),
                                "\n"
                            ));
                        log.error(msg);
                        ++errorCount[0];
                        return null;
                    }
                })
                .filter(m -> m != null)
                .collect(Collectors.toList());
            // Convert all maps that have not failed prior tests.
            final List<HallConfiguration> converted = listOfMaps.stream()
                .map(hallChecker::convertAll)
                .collect(Collectors.toList());
            // If all maps got converted, store the result.
            if (errorCount[0] == 0) result = converted;
        }
        return result;
    }
    
    private static ConfigDataChecker<BladeSetConfiguration> bsetChecker = 
        new ConfigDataChecker<>(BladeSetConfiguration.getConstructor());
    
    /**
     * Performs checking and conversion for raw blade set configuration data. This is exposed so that one can
     * use either what's provided by the CCS configuration API or use the result of running the input
     * conversion engine on a properties file.
     *
     * @param log used for recording errors.
     * @param raw the output of the conversion engine or the configuration API.
     * @return either null if there were any errors, or converted configuration.
     */
    public static BladeSetConfiguration
        convertBladeSetConfiguration(final Logger log, final ShutterSide side, final Map<String, String> raw) {
        BladeSetConfiguration result = null;
        boolean ok = true;
        if (raw == null) {
            log.error("Missing blade set configuration for " + side + ".");
            ok = false;
        }
        if (ok) {
            // Add a required "side" entry to the map.
            raw.put("side", side.toString());
            // Check that the map has a complete set of entries.
            final List<String> missing = bsetChecker.checkAllPresent(raw);
            if (!missing.isEmpty()) {
                final String msg
                    = missing.stream()
                    .collect(Collectors.joining(
                            "\n    ",
                            String.format("Missing parameters for blade set config %s.\n    ", side.toString()),
                            "\n"
                        ));
                log.error(msg);
                ok = false;
            }
        }
        // Check that each map value is convertible to the desired type.
        if (ok) {
            final List<String> wrong = bsetChecker.checkAllTypes(raw);
            if (!wrong.isEmpty()) {
                final String msg
                    = wrong.stream()
                    .collect(Collectors.joining(
                            "\n    ",
                            String.format("Wrong data types for blade set config %s.\n    ", side.toString()),
                            "\n"
                        ));
                log.error(msg);
                ok = false;
            }
        }
        // Convert if the map has passed prior tests.
        if (ok) {
            result = bsetChecker.convertAll(raw);
        }
        return result;
    }
    
    private static ConfigDataChecker<ShutterConfiguration> shutterChecker = 
        new ConfigDataChecker<>(ShutterConfiguration.getConstructor());

    /**
     * Performs checking and conversion for raw shutter configuration data. This is exposed so that one can
     * use either what's provided by the CCS configuration API or use the result of running the input
     * conversion engine on a properties file.
     *
     * @param log used for recording errors.
     * @param raw the output of the conversion engine or the configuration API.
     * @return either null if there were any errors, or converted configuration.
     */
    public static ShutterConfiguration
        convertShutterConfiguration(final Logger log, final Map<String, String> raw) {
        ShutterConfiguration result = null;
        boolean ok = true;
        if (raw == null) {
            log.error("Missing shutter configuration.");
            ok = false;
        }
        if (ok) {
            // Check that the map has a complete set of entries.
            final List<String> missing = shutterChecker.checkAllPresent(raw);
            if (!missing.isEmpty()) {
                final String msg
                    = missing.stream()
                    .collect(Collectors.joining(
                            "\n    ",
                            "Missing parameters for shutter config.\n    ",
                            "\n"
                        ));
                log.error(msg);
                ok = false;
            }
        }
        // Check that each map value is convertible to the desired type.
        if (ok) {
            final List<String> wrong = shutterChecker.checkAllTypes(raw);
            if (!wrong.isEmpty()) {
                final String msg
                    = wrong.stream()
                    .collect(Collectors.joining(
                            "\n    ",
                            "Wrong data types for shutter config.\n    ",
                            "\n"
                        ));
                log.error(msg);
                ok = false;
            }
        }
        // Convert if the map has passed prior tests.
        if (ok) {
            result = shutterChecker.convertAll(raw);
        }
        return result;
    }
}
