package org.lsst.ccs.subsystems.shutter.parker;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.lsst.ccs.drivers.iocard.AccesDio;
import org.lsst.ccs.drivers.iocard.Helios;

import org.lsst.ccs.drivers.parker.AcrComm;

import org.lsst.ccs.subsystems.shutter.common.BladePosition;
import org.lsst.ccs.subsystems.shutter.common.BladeSet;
import org.lsst.ccs.subsystems.shutter.common.BladeSetCalibration;
import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallTransition;
import org.lsst.ccs.subsystems.shutter.common.MovementHistory;

/**
 ***************************************************************************
 **
 **  Routines to perform shutter blade operations.
 **
 **  <p>Definitely NOT thread-safe because any method might manipulate the hardware.
 **  Use should be confined to a single thread, or at least one at a time.
 **
 **  @author Owen Saxton
 **  @author tether
 **
 ***************************************************************************
 */
public final class BladeSetDrvr implements BladeSet {

   /**
    ***************************************************************************
    **
    **  Inner class to hold an item of raw Hall sensor data
    **
    ***************************************************************************
    */
    public static class HallItem {
        /** The time of the transition in nanoseconds (System.nanoTime()). */
        public long     time;
        
        /** The ID number of the sensor (0, 1, 2 ...). */
        public int      sensor;
        
        /** Is the sensor open, that is, is there no magnet close by? */
        public boolean  open;

    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold a set of raw Hall sensor data
    **
    ***************************************************************************
    */
    public static class Hall {

        /** The movement start time in nanoseconds (System.nanoTime()). */
        public long        startTime;
        
        /** The number of Hall transitions that took place during the move. Will
            be greater than data.length if we ran out of space for transitions.
        */
        public int         count;
        
        /** The current state of all the switches, one bit each. This is set at the start of the move
            and updated after every transition.
        */
        public int         state;
        
        /** All the HallItems collected so far. An array is used for speedy access. */
        public HallItem[]  data;

        /** Constructs an instance capable of holding up to 100 HallItems. */
        public Hall()
        {
            this(100);
        }

        /**
         * Constructs an instance that can hold a given number of HallItems.
         * @param maxItem the limit on the number of items
         */
        public Hall(int maxItem)
        {
            data = new HallItem[maxItem];
            for (int j = 0; j < maxItem; j++)
                data[j] = new HallItem();
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold an item of sampled data from the motor controller.
    **  A number of integer and float parameters may be sampled at the same time.
    **  The order of the parameter values in each array is implied by the sampling
    **  configuration; the parameters sampled are not otherwise identified.
    **
    ***************************************************************************
    */
    public static class Sample {
        /** The sample time in microseconds since the start of the move, based
            on the motor controller's clock.
        */
        public int      time;
        
        /** The array of sampled integer parameter values. */
        public int[]    iVals;
        
        /** The array of sampled float parameter values. */
        public float[]  fVals;
    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold a set of Sample instances along with some
    **  general info about the move.
    **
    ***************************************************************************
    */
    public static class SampData {

        /** The status of the move at completion. See {@link MovementStatus}.*/
        public int                  status;
        
        /** The move start time, a system time stamp in microseconds. */
        public long                 startTime;
        
        /** The absolute position in mm at the start of the move. */
        public double               startPosn;
        
        /** The move end time, a system time stamp in microseconds. */
        public long                 endTime;
        
        /** The absolute position in mm at the end of the move. */
        public double               endPosn;
        
        /** The collection of data samples from the controller. */
        public ArrayList<Sample >   sList = new ArrayList<>();
    }

   /*
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    
    /** Use an S-surve motion profile. */
    public final static int    MOV_TYP_SCURVE = 0;
    
    /** Use a trapezoidal motion profile. */
    public final static int    MOV_TYP_TRAP   = 1;
    
    /** Start looking for home by going in the negative direction (decreasing absolute position). */
    public final static int    HOM_OPT_NDIRN  = 0x01;
    
    /** Make the final approach to home from the positive side (higher absolute position). */
    public final static int    HOM_OPT_PFINAL = 0x02;
    
    /** The limit on the motor torque expressed as amps supplied by the motor driver. */
    public final static double TORQUE_LIM   = 10.0;

   /*
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    
    // User parameter numbers. User parameters are set for the general purpose motion program
    // in the motion controller to use as arguments. User parameters nos. P_PARMS, P_PARMS+1, ... contain
    // the ID numbers of the parameters to be sampled, starting with the integer
    // parameters.
    private final static int P_DRVON = 0, // "Leave drive on" flag.
                             P_DIST  = 1, // The distance to travel, signed, mm.
                             P_SMPTM = 2, // The time between parameter samplings in millisec.
                             P_SMPCT = 3, // The maximum number of samples to take.
                             P_NIPRM = 4, // The number of integer parameters to sample.
                             P_NFPRM = 5, // The number of float parameters to sample.
                             P_PARMS = 6; // The ID no. of the first parameter to sample.
    
    // The ID numbers assigned to various motion programs.
    private final static int MOVE_PROG = 0, // The general purpose motion program with sampling.
                             CAL_PROG  = 1, // The calibration program.
                             TEST_PROG = 2, // A general controller checkup.
                             TEMP_PROG = 3; // Loaded on the fly for special purposes.

   /*
    ***************************************************************************
    **
    **  Private fields and constructor.
    **
    ***************************************************************************
    */
    // The configuration for this blade set.
    private final BladeSetConfiguration config;
    
    // Values copied from the configuration.
    private final int bladeIndex, dioHPort, dioIPort, dioILine, dioOPort, dioOLine;
    
    // Driver for the Acces I/O card.
    private final AccesDio acd;
    
    // Driver for the Helios ADC unit.
    private final Helios hel;
    
    // Driver for the Parker ACR motor controller.
    private final AcrComm acr;

    // Leave the motor drive on after a move?
    private boolean driveOn = false;


    /**
     * Finds the default configuration for the blade set and creates the hardware drivers.
     * @param index the index, 0 or 1, for the blade set.
     */
    public BladeSetDrvr(int index) {
        bladeIndex = index;
        config = new BladeSetConfiguration(index);
        dioHPort = config.getDioHPort();
        dioIPort = config.getDioIPort();
        dioILine = config.getDioILine();
        dioOPort = config.getDioOPort();
        dioOLine = config.getDioOLine();
        acr = new AcrComm(config.getNode(), true);
        acd = new AccesDio();
        hel = new Helios();
    }
   /**
    ***************************************************************************
    **
    **  Completes the initialization of the hardware drivers created by the constructor.
    **
    ***************************************************************************
    */
    @Override
    public void init()
    {
        acrInitialize();
        acd.init(config.getDioAddr(), config.getDioLevel());
        acd.dioConfig(config.getDioConf());
        hel.init(config.getAdAddr(), config.getAdLevel());
        hel.adConfig(3, true, false);
        hel.adSetChan(config.getAdTempChn());
    }


   @Override
    public int getIndex()
    {
        return bladeIndex;
    }


   /**
    ***************************************************************************
    **
    **  Moves the blade set to a given relative (i.e. fractional) position.
    **  Calls moveH() requesting S-curve motion with the sample count
    **  set from the configuration.
    **  @param relPosition the target relative position
    **  @param time the duration of the motion in seconds
    **  @return the resulting movement history
    **
    ***************************************************************************
    */
    @Override
    public MovementHistory moveToPosition(double relPosition, double time)
    {
        return moveH(MOV_TYP_SCURVE,
                            config.position(relPosition) - getPosition(),
                            time, 0.0, config.getNumSamp());
    }


    @Override
    public double getCurrentPosition()
    {
        return config.relPosition(acr.getIntParm(AcrComm.ENC_POSN_PRM));
    }

    @Override
    public BladeSetConfiguration getBladeSetConfiguration()
    {
        return config;
    }


   /**
    ***************************************************************************
    **
    **  Gets the current absolute position in mm.
    **  @return the position
    **
    ***************************************************************************
    */
    public double getPosition()
    {
        return config.position(acr.getIntParm(AcrComm.ENC_POSN_PRM));
    }


   /**
    ***************************************************************************
    **
    **  Turns the motor drive current on and set the "drive always on" flag true.
    **
    ***************************************************************************
    */
    public void setDriveOn()
    {
        enableDrive();
        driveOn = true;
    }


   /**
    ***************************************************************************
    **
    **  Turns the motor drive current off and set the "drive always on" flag false.
    **
    ***************************************************************************
    */
    public void setDriveOff()
    {
        driveOn = false;
        disableDrive();
    }


   /**
    ***************************************************************************
    **
    **  Turns on the motor drive current if the "drive always on" flag is false.
    *
    ***************************************************************************
    */
    public void enableDrive()
    {
        if (!driveOn) acr.sendStr("axis 0 drive on", 0);
    }


   /**
    ***************************************************************************
    **
    **  Turns off the motor drive current if the "drive always on" flag is false.
    *
    ***************************************************************************
    */
    public void disableDrive()
    {
        if (!driveOn) acr.sendStr("axis 0 drive off", 0);
    }


   /**
    ***************************************************************************
    **
    **  Calibrates the Hall switch positions using a series of small movements.
    **  <p>Does a series of small movement steps for each leg and reads the Hall sensors
    **  after each one.  This can take a reasonably long time if the step size is small
    **  enough to be useful. The blade set is moved away from home for the first leg
    **  and toward home for the second, regardless of the signs of dist and step.
    **  @param dist the total blade set travel distance for each leg in mm (sign ignored)
    **  @param step the blade set step in mm (sign ignored)
    **  @param time the duration of each step in seconds
    **  @return the calibration object
    ***************************************************************************
    */
    public BladeSetCalibration calibrate(double dist, double step, double time)
    {
        // Determine the sign that dist must have in order to move away from home.
        dist = Math.copySign(dist, config.getClosed() - config.getOpen());

        // Move away from then toward home.
        final List<BladeSetCalibration.Item> items = new ArrayList<>();
        final int status1 = calibLeg(dist, step, time, items);
        final int status2 = calibLeg(-dist, step, time, items);
        return new BladeSetCalibration(status1 == 0 ? status2 : status1, items);
    }


   /**
    ***************************************************************************
    **
    **  Calibrates the Hall switch positions using continuous slow motion.
    **  <p>Captures Hall sensor transitions while moving the blade set continuously,
    **  thereby mimicking what happens during normal shutter use.  Dynamic effects
    **  are eliminated by specifying a sufficiently long period for the calibration.
    **  The blade set is moved away from home for the first leg and toward home for the second.
    **  @param dist the total blade set travel distance in mm (sign ignored)
    **  @param time the duration of each leg in seconds
    **  @return the calibration object
    ***************************************************************************
    */
    public BladeSetCalibration calibNew(double dist, double time)
    {

        // Give dist the right sign for moving away from home.
        dist = Math.copySign(dist, config.getClosed() - config.getOpen());

        // Get the initial state of the Hall switches.
        Hall hall = new Hall();
        hall.state = acd.dioInp(dioHPort);
        
        // On each Hall transition call this.readHallC() passing the Hall
        // object as argument. readHallC() will add a new item
        // each time it's called.
        acd.attachInt(1 << dioHPort, this, "readHallC", hall);

        // Will the first Hall transition be an opening or a closing? Set the
        // next capture trigger accordingly. readHallC() will change the state
        // of the output line dioOLine for each transition.
        int capMode;
        if (hall.state == 0xff) {
            // All switches are open so the first transition will be a closing.
            acd.dioSetBit(dioOPort, dioOLine);
            capMode = AcrComm.ICM_FALL_3RD;
        }
        else {
            // At least one switch is closed so the first transition will be
            // an opening. This follows from the width of each magnet being much
            // smaller than the distance between magnets.
            acd.dioClrBit(dioOPort, dioOLine);
            capMode = AcrComm.ICM_RISE_3RD;
        }

        /*
        **  Load the calibration movement program
        */
        acr.sendStr("prog" + CAL_PROG, 0);// Select the program we chose for calibration.
        acr.sendStr("program", 0);        // Set program entry mode.
        acr.sendStr("clear", 0);          // Destroy any existing program.
        acr.sendStr("dim lv(10)", 0);     // Ten scalar integers lv0 - lv9
        acr.sendStr("dim la(1)", 0);      // One array la0 for captured values ...
        acr.sendStr("dim la0(100)", 0);   // ... with 100 elements
        acr.sendStr("lv0 = 0", 0);        // Next capture index
        acr.sendStr("lv1 = " + capMode, 0); // Capture trigger mode
        acr.sendStr("set " + AcrComm.CAL_SYNC_BIT, 0); // Advertise that the loop is running
        acr.sendStr("while (1 = 1)", 0);
        acr.sendStr("intcap axis0 (lv1)", 0); // Arm the trigger for the next capture
        acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0); // Wait for next capture to finish
        acr.sendStr("la0(lv0) = p" + AcrComm.CAP_POSN_PRM, 0); // Store encoder value
        acr.sendStr("lv0 = lv0 + 1", 0); // Advance capture index
        // Alternate capture trigger between Hall closing and Hall opening.
        acr.sendStr("lv1 = " + (AcrComm.ICM_RISE_3RD + AcrComm.ICM_FALL_3RD)
                      + " - lv1", 0);
        acr.sendStr("wend", 0); // End loop
        acr.sendStr("endp", 0); // End program entry
        acr.sendStr("sys", 0);  // Back to the sys-level prompt

        // Do a move in each direction, capturing the transitions
        final List<BladeSetCalibration.Item> clist = new ArrayList<>();
        final int status1 = calibLegNew(dist, time, hall, clist);
        final int status2 = calibLegNew(-dist, time, hall, clist);

        // Cancel Hall switch interrupts and the clear output line.
        acd.detachInt();
        acd.dioClrBit(dioOPort, dioOLine);

        return new BladeSetCalibration(status1 == 0 ? status2 : status1, clist);
    }

   /**
    ***************************************************************************
    **
    **  Performs a simple move without enabling Hall transition interrupts
    **  or motor controller parameter sampling. We don't use the normal motion program
    **  in the controller, instead we issue the AXIS0 JOG INC command directly.
    **  @param type the move type, MOV_TYP_SCURVE or MOV_TYP_TRAP
    **  @param dist the signed distance to move (change of absolute position in mm)
    **  @param time the duration of the move in seconds
    **  @param mvFract modifies a trapezoidal velocity profile. A value of 0.0
    **  results in a standard profile whereas 1.0 results in a profile
    **  with discontinuous changes between intervals of constant velocity. Put
    **  another way, increasing the value steepens the sides of the
    **  velocity trapezoids and broadens the tops. Limited to [0.0, 0.9].
    **  Always use 0.0 for S-curve profiles.
    **  @return the end-of-move status code (see {@link MovementStatus})
    **
    ***************************************************************************
    */
    public int move(int type, double dist, double time, double mvFract)
    {

        // Return if distance is zero.
        if (dist == 0.0) return 0;

        // Configure and enable the drive.
        configMove(type, dist, time, mvFract);

        // Do the move and wait for motion to stop.
        acr.sendStr("axis0 jog inc " + config.motorDistance(dist), 0);
        acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);

        // Possibly disable the drive.
        disableDrive();

        return moveStatus();
    }


   /**
    ***************************************************************************
    **
    **  Performs a move accompanied by parameter sampling in the motor controller.
    **  @param type see {@link #move(int, double, double, double)}
    **  @param dist see {@link #move(int, double, double, double)}
    **  @param time see {@link #move(int, double, double, double)}
    **  @param mvFract see {@link #move(int, double, double, double)}
    **  @param nSamp the number of times to perform sampling
    **  @param iParms the ID numbers of the integer parameters to record at each sampling
    **  @param fParms the ID numbers of the float parameters to record at each sampling
    **  @return a SampData filled with the sample data (if any)
    **
    ***************************************************************************
    */
    public SampData moveD(int type, double dist, double time, double mvFract,
                          int nSamp, int[] iParms, int[] fParms)
    {

        // Set the parameters used by the move program and run it.
        int nsamp = initMove(type, dist, time, mvFract, nSamp, iParms, fParms);
        SampData data = new SampData();
        data.startPosn = getPosition();
        data.startTime = runMove(nsamp <= 0, null);
        data.endTime = System.currentTimeMillis();
        data.endPosn = getPosition();

        // Return if error or null move.
        if (nsamp <= 0) {
            data.status = -nsamp;
            return data;
        }

        // Retrieve the sampled parameters.Note that acr.getVars()
        // assigns to elements of its last argument.
        int[][] iVals = new int[iParms.length][nsamp];
        float[][] fVals = new float[fParms.length][nsamp];
        int[] time0 = new int[1];

        int tsamp = acr.getVars(MOVE_PROG, 0, 1, time0);  // Scalar lv0: start time
        for (int j = 0; j < iParms.length; j++)
            tsamp += acr.getVars(MOVE_PROG, j, 0, nsamp, iVals[j]); // Array la<j>[]
        for (int j = 0; j < fParms.length; j++)
            tsamp += acr.getVars(MOVE_PROG, j, 0, nsamp, fVals[j]); // Array sa<j>[]
        if (tsamp != nsamp * (iParms.length + fParms.length) + 1) {
            data.status = MovementStatus.STS_MSSG_PGM_VBLES;
            return data;
        }

        // Set movement completion status.
        data.status = moveStatus();

        // Get the index of sampled time; -1 if time is not being sampled.
        int tIndx = -1;
        for (int j = 0; j < iParms.length; j++)
            if (iParms[j] == AcrComm.SYS_CLOCK_PRM) tIndx = j;

        // Output the sampled values. Collect all the values for a given
        // sample time into one Sample object.
        for (int j = 0; j < nsamp; j++) {
            Sample sItem = new Sample();
            sItem.time = 0;
            sItem.iVals = new int[iParms.length];
            for (int k = 0; k < iParms.length; k++) {
                sItem.iVals[k] = iVals[k][j];
                if (k == tIndx) {
                    sItem.iVals[k] -= time0[0];         // Subtract start time
                    sItem.time = 1000 * sItem.iVals[k]; // millisec to microsec
                }
            }
            sItem.fVals = new float[fParms.length];
            for (int k = 0; k < fParms.length; k++)
                sItem.fVals[k] = fVals[k][j];
            data.sList.add(sItem);
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Performs a move using the standard motion program, recording Hall switch transitions and taking
    **  periodic data samples in the motor controller. Sets the motor controller to record
    **  a given number of samples at fixed intervals. In each sample: controller millisecond
    **  clock, motor encoder value, axis status flags (not used). Since motor sampling
    **  isn't triggered by Hall transitions this method determines positions for
    **  the Hall transitions by matching sensor ID, direction of travel and
    **  transition final state against items in the current calibration.
    **  @param type see {@link #move(int, double, double, double)}
    **  @param dist see {@link #move(int, double, double, double)}
    **  @param time see {@link #move(int, double, double, double)}
    **  @param mvFract see {@link #move(int, double, double, double)}
    **  @param nSamp the number of times to perform sampling
    **  @return the resulting movement history
    ***************************************************************************
    */
    public MovementHistory moveH(int type, double dist, double time,
                                 double mvFract, int nSamp)
    {

        // Read the current states of the Hall switches.
        Hall hall = new Hall();
        hall.state = acd.dioInp(dioHPort);
        
        // Call this.readHallM() on each Hall transition, with our local Hall object
        // as an argument.
        acd.attachInt(1 << dioHPort, this, "readHallM", hall);

        // Set the parameters for the move program and run it.
        final int[] iParms = {AcrComm.SYS_CLOCK_PRM, AcrComm.ENC_POSN_PRM,
                        AcrComm.PRIM_AXIS_PRM},
              fParms = {};
        final double sPosition = getCurrentPosition();
        final int nsamp = initMove(type, dist, time, mvFract, nSamp, iParms, fParms);
        final long sTime = 1000 * runMove(nsamp <= 0, hall);
        final long eTime = 1000 * System.currentTimeMillis();

        // Disable Hall switch interrupts.
        acd.detachInt();

        // Return if error or null move
        if (nsamp <= 0) {
            return new MovementHistory(
                getIndex(),
                -nsamp,
                sTime,
                getCurrentPosition(),
                eTime,
                getCurrentPosition(),
                Collections.emptyList(),
                Collections.emptyList()
            );
        }

        // Get the sampled positions and times.
        List<BladePosition> pList = new ArrayList<>();
        int status = getPositionData(sTime, nsamp, pList);
        if (status == 0) status = moveStatus();
        
        // Starting position (absolute).
        double posn = pList.isEmpty() ? 0.0 : pList.get(0).getPosition();

        // Locate the first calibration item that will be used.
        boolean reverse = (dist < 0.0);
        List<BladeSetCalibration.Item> cList
            = config.getCalibration().getData();
        int cIndex; // Calib. item no.
        // Locate first calibration item whose position is ahead
        // of posn along the direction of travel.
        for (cIndex = 0; cIndex < cList.size(); cIndex++) {
            final BladeSetCalibration.Item cItem = cList.get(cIndex);
            if (cItem.isReverse() ^ reverse) continue;
            if (reverse && cItem.getPosition() < posn) break;
            if (!reverse && cItem.getPosition() > posn) break;
        }

        // Create HallTransitions from the HallItems and the calibration
        // info.
        final List<HallTransition> hList = new ArrayList<>();
        if (hall.count > hall.data.length) {
            // Did we run out of space to store transitions?
            if (status == 0) status = MovementStatus.STS_EXCESS_HALL_TRAN;
            // There was some kind of move error. We got what we got.
            hall.count = hall.data.length;
        }
        // For each
        for (int j = 0; j < hall.count; j++) {
            HallItem item = hall.data[j];
            // Attempt to update the transition position.
            posn = 0.0;
            while (cIndex < cList.size()) {
                BladeSetCalibration.Item cItem = cList.get(cIndex++);
                if (cItem.getSensor() == item.sensor
                      && cItem.isOpen() == item.open) {
                    posn = cItem.getPosition();
                    break;
                }
            }
            HallTransition hItem = new HallTransition(
                item.time / 1000 + hall.startTime,
                item.sensor,
                item.open,
                dist < 0.0,
                posn,
                config.relPosition(posn));
            hList.add(hItem);
        }

        return new MovementHistory(
            bladeIndex,
            status,
            sTime,
            sPosition,
            eTime,
            getCurrentPosition(),
            pList,
            hList
        );
    }


   /**
    ***************************************************************************
    **
    **  Works just like moveH() except that no Hall transitions are collected.
    **  @see #moveH(int, double, double, double, int) 
    **  @return a movement history with an empty list of Hall transitions
    ***************************************************************************
    */
    public MovementHistory moveP(int type, double dist, double time,
                                 double mvFract, int nSamp)
    {

        // Set the parameters used by the move program and run it.
        final int[] iParms = {AcrComm.SYS_CLOCK_PRM, AcrComm.ENC_POSN_PRM,
                        AcrComm.PRIM_AXIS_PRM},
              fParms = {};
        final double sPosition = getCurrentPosition();
        final int nsamp = initMove(type, dist, time, mvFract, nSamp, iParms, fParms);
        final long sTime = 1000 * runMove(nsamp <= 0, null);
        final long eTime = 1000 * System.currentTimeMillis();

        // Return if error or null move.
        if (nsamp <= 0) {
            return new MovementHistory(
                getIndex(),
                -nsamp,
                sTime,
                getCurrentPosition(),
                eTime,
                getCurrentPosition(),
                Collections.emptyList(),
                Collections.emptyList()
            );
        }

        // Get the sampled positions and times.
        List<BladePosition> pList = new ArrayList<>();
        int status = getPositionData(sTime, nsamp, pList);
        if (status == 0) status = moveStatus();

        return new MovementHistory(
            bladeIndex,
            status,
            sTime,
            sPosition,
            eTime,
            getCurrentPosition(),
            pList,
            Collections.emptyList()
        );
    }


   /**
    ***************************************************************************
    **
    **  Finds the home position for this blade set.
    **  See the section on Homing in the ACR Programming Guide.
    **  There are two inputs to the controller that define a Home region; one
    **  defines the negative (low) edge and the other the positive (high) edge.
    **  We do "backup" homing which means that motion continues at speed until
    **  a particular edge is reached then decelerates to a stop. Motion then
    **  resumes in the opposite direction using a different, usually much lower,
    **  speed until the same edge is crossed again. One can require that the final
    **  approach to the edge be from below, so if at this point the motion is
    **  in the required direction it stops immediately. Otherwise we have
    **  to do another deceleration and reversal.
    **
    **  <p>Hitting a motion limit the first time causes motion to reverse. Hitting
    **  one a second time stops motion and causes the homing operation to fail.</p>
    **  @param optns a bit mask of options. The logical OR of zero or more of:
    **  <ul>
    **  <li>{@link #HOM_OPT_NDIRN}</li>
    **  <li>{@link #HOM_OPT_PFINAL}</li>
    **  </ul>
    **  @param vel the velocity at which to begin the home search, mm/sec
    **  @return 0 for success, -1 for failure
    **
    ***************************************************************************
    */
    public int home(int optns, double vel)
    {
        final double fvel = vel / 10.0;
        final double acc = 10.0 * vel;
        final double jrk = 0;
        // Set the no. of position encoder pulses per mm and the motion profile
        // for the initial movement.
        acr.setFloatParm(AcrComm.PPU_PRM, (float)config.ppu());
        acr.setFloatParm(AcrComm.JOG_VEL_PRM, (float)vel);
        acr.setFloatParm(AcrComm.JOG_ACC_PRM, (float)acc);
        acr.setFloatParm(AcrComm.JOG_DEC_PRM, (float)acc);
        acr.setFloatParm(AcrComm.JOG_JRK_PRM, (float)jrk);
        // Set the velocity to be used on the final approach to the home position.
        acr.sendStr("axis0 jog homvf " + fvel, 0);
        // We will use backup-style homing.
        acr.setBit(AcrComm.HOME_BKUP_BIT);
        // We will ignore the negative edge of the home region and go for the positive edge.
        acr.clearBit(AcrComm.HOME_NEDG_BIT);
        // Optionally allow final approach from above (motion in negative direction).
        if ((optns & HOM_OPT_PFINAL) != 0)
            acr.clearBit(AcrComm.HOME_NFNL_BIT); // Forbid
        else
            acr.setBit(AcrComm.HOME_NFNL_BIT);  // Allow
        enableDrive();

        // Do the home and check outcome. "jog -1" specifies initial motion
        // in the negative direction (decreasing absolute position) while "jog 1"
        // specifies the positive direction.
        int status = 0;
        acr.sendStr("axis0 jog home " + ((optns & HOM_OPT_NDIRN) != 0 ? -1 : 1),
                    0);
        while (true) {
            acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0); // Wait until motion is finished
            if (acr.getBit(AcrComm.HOME_FND_BIT) != 0) break; // Success
            if (acr.getBit(AcrComm.HOME_FAIL_BIT) != 0) {     // Failure
                status = -1;
                break;
            }
        }

        // Possibly disable the drive.
        disableDrive();

        return status;
    }


   /**
    ***************************************************************************
    **
    **  Gets the configuration object being used.
    **  @return the configuration
    ***************************************************************************
    */
    public BladeSetConfiguration getConfig()
    {
        return config;
    }


   /**
    ***************************************************************************
    **
    **  Gets the AccesDio object being used.
    **  @return the I/O card object
    ***************************************************************************
    */
    public AccesDio getAcces()
    {
        return acd;
    }


   /**
    ***************************************************************************
    **
    **  Gets the Helios object being used.
    **  @return the ADC unit control object
    ***************************************************************************
    */
    public Helios getHelios()
    {
        return hel;
    }


   /**
    ***************************************************************************
    **
    **  Gets the motor communications object being used.
    **  @return the AcrComm object
    ***************************************************************************
    */
    public AcrComm getComm()
    {
        return acr;
    }


   /**
    ***************************************************************************
    **
    **  Gets the status of the last move from the motor controller: normal
    **  or premature stop reason.
    **  @return 0 for success or a code from {@link MovementStatus}
    ***************************************************************************
    */
    public int moveStatus()
    {
        if (acr.getBit(AcrComm.KILL_MOVE_BIT) == 0)
            return 0;

        if (acr.getBit(AcrComm.POSN_ERR_BIT) != 0)
            return MovementStatus.STS_EXCS_POSN_ERROR;

        if (acr.getBit(AcrComm.POS_EOT_BIT) != 0)
            return MovementStatus.STS_POS_EOT_LIMIT;

        if (acr.getBit(AcrComm.NEG_EOT_BIT) != 0)
            return MovementStatus.STS_NEG_EOT_LIMIT;

        if (acr.getBit(AcrComm.POS_SLM_BIT) != 0)
            return MovementStatus.STS_POS_SOFT_LIMIT;

        if (acr.getBit(AcrComm.NEG_SLM_BIT) != 0)
            return MovementStatus.STS_NEG_SOFT_LIMIT;

        return MovementStatus.STS_UNKN_MOVE_TRUNC;
    }


   /**
    ***************************************************************************
    **
    **  Gets the drive temperature.
    **  @return the temperature in degrees Celsius
    **
    ***************************************************************************
    */
    public double getDriveTemperature()
    {
        return acr.getFloatParm(AcrComm.DRIVE_TEMP_PRM);
    }


   /**
    ***************************************************************************
    **
    **  Gets the ambient temperature around the HCU.
    **  @return the temperature in degrees Celsius
    ***************************************************************************
    */
    public double getAmbientTemperature()
    {
        return 100.0 * hel.adCountToVolts(hel.adSample());
    }


   /**
    ***************************************************************************
    **
    **  Performs Hall switch position calibration in one direction using
    **  many small steps.
    **  @param dist the total change in absolute position in mm
    **  @param step the step size in mm (sign ignored)
    **  @param time the duration of each step in seconds
    **  @param cList the list to which calibration items are to be added
    **
    ***************************************************************************
    */
    private int calibLeg(double dist, double step, double time,
                          List<BladeSetCalibration.Item> cList)
    {

        /*
        **  Configure and enable the drive
        */
        configMove(MOV_TYP_SCURVE, step, time, 0.0);

        /*
        **  Generate command to step the shutter. Need to call config.motorDistance()
        **  to get the direction right.
        */
        step = Math.copySign(step, dist);
        String stepCmnd = "axis0 jog inc (" + config.motorDistance(step) + ")";

        /*
        **  Read the switches
        */
        int oldVal = acd.dioInp(dioHPort);

        /*
        **  Loop while stepping the shutter and reading the switches.
        **  Quit if positive or negative travel limit is reached
        */
        int index = 0, oldSensor = -1, status = 0;
        boolean reverse = (dist < 0.0);
        double position = getPosition(), endPosition = position + dist;
        
        while (reverse ^ ((position - endPosition) < 0.0)) {
            acr.sendStr(stepCmnd, 0);
            acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0); // Wait for step to finish
            position = getPosition();
            int value = acd.dioInp(dioHPort);
            int chan = oldVal ^ value;
            if (chan != 0) {   // If some switch changed state...
                oldVal = value;
                boolean open = (chan & oldVal) != 0; // 0 = closed,  1  = open
                int sensor = Integer.numberOfTrailingZeros(chan);
                if (sensor != oldSensor) {
                    // Same sensor as last time.
                    index = 0;
                    oldSensor = sensor;
                }
                else {
                    // Assign a unique index to each close+open pair for
                    // consecutive transitions of the same sensor.
                    if (!open) index++;
                }
                cList.add(new BladeSetCalibration.Item(sensor, index, position,
                                                       open, reverse));
            }
            if ((status = moveStatus()) != 0) break; // Move failure?
        }

        /*
        **  Possibly disable the drive
        */
        disableDrive();

        /*
        **  Set the status
        */
        return status;
    }


   /**
    ***************************************************************************
    **
    **  Calibrates the Hall switch positions in one direction using
    **  a continuous motion.
    **  @see calibNew()
    **
    ***************************************************************************
    */
    private int calibLegNew(double dist, double time, Hall hall,
                             List<BladeSetCalibration.Item> cList)
    {
        /*
        **  Clear the Hall count
        */
        hall.count = 0;

        /*
        **  Configure and enable the drive, and start the capture program
        */
        configMove(MOV_TYP_SCURVE, dist, time, 0);
        acr.clearBit(AcrComm.CAL_SYNC_BIT);
        acr.sendStr("run prog" + CAL_PROG, 0);
        acr.sendStr("inh " + AcrComm.CAL_SYNC_BIT, 0); // Wait for capture loop to run

        /*
        **  Move the shutter and wait for completion
        */
        acr.sendStr("axis0 jog inc " + config.motorDistance(dist), 0);
        acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
        int status = moveStatus();

        /*
        **  Possibly disable the drive, halt the program and disable capture triggering
        */
        disableDrive();
        acr.sendStr("halt prog" + CAL_PROG, 0);
        acr.sendStr("intcap axis0 off", 0);

        /*
        **  Get the captured encoder positions and check counts. We should
        */
        int[] count = new int[1];
        acr.getVars(CAL_PROG, 0, 1, count); // No. of samples (lv0 from capture program)
        int[] posns = new int[count[0]];
        int nsamp = acr.getVars(CAL_PROG, 0, 0, count[0], posns); // la0[] from capture program
        if (nsamp != count[0]) {
            return MovementStatus.STS_MSSG_PGM_VBLES; // la0[] wasn't as big as it should have been
        }
        if (hall.count > hall.data.length) { // Interrupt count > no. of transitions saved?
            if (status == 0)
                status = MovementStatus.STS_EXCESS_HALL_TRAN; // Move OK, we just ran out of space
            hall.count = hall.data.length; // Move failure, just take what we got.
        }
        if (nsamp != hall.count) { // No. of position samples != no. of Hall transitions?
            if (status == 0)
                status = MovementStatus.STS_UNEQL_HALL_TRAN; // Move OK but data still screwy
            if (nsamp < hall.count)
                hall.count = nsamp; // Move failure, take what samples we got.
        }

        /*
        **  Add the encoder positions and sensor data to the calibration list
        */
        int index = 0, oldSensor = -1;
        boolean reverse = (dist < 0f);
        for (int j = 0; j < hall.count; j++) {
            HallItem item = hall.data[j];
            if (item.sensor != oldSensor) {
                index = 0;
                oldSensor = item.sensor;
            }
            else
                if (!item.open) index++;
            cList.add(new BladeSetCalibration.Item(item.sensor, index,
                                                   config.position(posns[j]),
                                                   item.open, reverse));
        }

        /*
        **  Set the status
        */
        return status;
    }


   /**
    ***************************************************************************
    **
    **  Retrieve position samples from the standard move program
    **  in the motor controller. The position samples stored in the
    **  program's arrays are used only up and including to the point when
    **  the "jog active" axis status flag goes low.
    **  @param sTime the move start time, a system time stamp in microseconds
    **  @param nsamp the dimension of each of the program's sample arrays la0, la1, la2
    **  @param pList the list to which to add the BladePosition instances
    **
    ***************************************************************************
    */
    private int getPositionData(long sTime, int nsamp,
                                List<BladePosition> pList)
    {

        /*
        **  Retrieve the sampled positions and times
        */
        int[] times = new int[nsamp], posns = new int[nsamp],
              flags = new int[nsamp], time0 = new int[1];
        int tsamp = acr.getVars(MOVE_PROG, 0, 1, time0);     // lv0
        tsamp += acr.getVars(MOVE_PROG, 0, 0, nsamp, times); // la0[]
        tsamp += acr.getVars(MOVE_PROG, 1, 0, nsamp, posns); // la1[]
        tsamp += acr.getVars(MOVE_PROG, 2, 0, nsamp, flags); // la2[]
        if (tsamp != 3 * nsamp + 1)
            return MovementStatus.STS_MSSG_PGM_VBLES;

        /*
        **  Create the returned sampled values
        */
        for (int j = 0; j < nsamp; j++) {
            final double posn = config.position(posns[j]);
            pList.add(
                new BladePosition(
                    sTime + 1000 * (times[j] - time0[0]),
                    posn,
                    config.relPosition(posn)
                )
            );
            if ((flags[j] & 0x01000000) == 0) break;
        }

        return 0;
    }


   /**
    ***************************************************************************
    **
    **  Handles Hall switch interrupts for new-style calibrations.
    **  This routine is called whenever a Hall switch changes state. It assumes
    **  that only one Hall switch at a time will change state and that the
    **  open transition (magnet absent) matching the close transition (magnet present)
    **  for the same switch will come before the close transition for the next switch.
    **
    **  <p>The digital output line connected to the third external input of the motor controller
    **  is made to change state, which triggers the taking of a sample by the
    **  calibration program running in the controller. The line state change is
    **  made to mimic that of the Hall switch input line on the I/O card. I suspect
    **  that the reason for this is to allow a person using an oscilliscope
    **  to tell open and close transitions apart.</p>
    **
    **  <p>The count field in the Hall object actually counts interrupts rather
    **  than the number of Hall items available, so it can be used later to
    **  check whether there were too many transitions to record.</p>
    **  @param value not used
    **  @param parm a reference to the Hall object in which to place transition data
    ***************************************************************************
    */
    private void readHallC(int value, Object parm)
    {
        Hall hall = (Hall)parm;              // Collection of Hall data.
        int state = acd.dioInp(dioHPort);    // Current state of all switches.
        int sensor = state ^ hall.state;     // Which bits changed from the prior state?
        if (sensor == 0) return;             // False alarm?

        hall.state = state;                  // Remember state for next interrupt.
        if (hall.count < hall.data.length) { // If we still have unused items ...
            HallItem item = hall.data[hall.count];
            boolean open = (sensor & state) != 0;
            if (open) acd.dioSetBit(dioOPort, dioOLine); // Set output line.
            else acd.dioClrBit(dioOPort, dioOLine);
            item.time = System.nanoTime();
            item.sensor = Integer.numberOfTrailingZeros(sensor);
            item.open = open;
        }
        hall.count++;
    }


   /**
    ***************************************************************************
    **
    **  This works just like readHallC() except that the output line to the 
    **  motor controller isn't used.
    **
    ***************************************************************************
    */
    private void readHallM(int value, Object parm)
    {
        Hall hall = (Hall)parm;
        int state = acd.dioInp(dioHPort);
        int sensor = state ^ hall.state;
        if (sensor == 0) return;

        hall.state = state;
        if (hall.count < hall.data.length) {
            HallItem item = hall.data[hall.count];
            item.time = System.nanoTime();
            item.sensor = Integer.numberOfTrailingZeros(sensor);
            item.open = (sensor & state) != 0;
        }
        hall.count++;
    }


   /**
    ***************************************************************************
    **
    **  Configures the motor controller for a move with no sampling, such as
    *   performed by {@link #move(int, double, double, double)}, which see
    **  for a description of the parameters. The following motor controller parameters are set:
    **  <ul>
    **  <li>Pulses per unit (mm)</li>
    **  <li>Jog velocity limit, mm/sec</li>
    **  <li>Jog acceleration limit, mm/sec**2</li>
    **  <li>Jog deceleration limit, mm/sec**2</li>
    **  <li>Jog jerk, mm/sec**3. Non-zero only for S-curve motion</li>
    **  </ul>
    **  The motor drive is switched on if it was off.
    **
    ***************************************************************************
    */
    private void configMove(int type, double dist, double time, double mvFract)
    {
        double vel, acc, jrk = 0.0;
        if (mvFract < 0.0) mvFract = 0.0;
        else if (mvFract > 0.9) mvFract = 0.9;
        vel = Math.abs(2.0 * dist / ((1.0 + mvFract) * time));
        acc = Math.abs(4.0 * dist / ((1.0 - mvFract * mvFract) * time * time));
        if (type == MOV_TYP_SCURVE) {
            acc *= 2.0;
            jrk = acc * acc / vel;
        }
        acr.setFloatParm(AcrComm.PPU_PRM, (float)config.ppu());
        acr.setFloatParm(AcrComm.JOG_VEL_PRM, (float)vel);
        acr.setFloatParm(AcrComm.JOG_ACC_PRM, (float)acc);
        acr.setFloatParm(AcrComm.JOG_DEC_PRM, (float)acc);
        acr.setFloatParm(AcrComm.JOG_JRK_PRM, (float)jrk);
        enableDrive();
    }


   /**
    ***************************************************************************
    **
    **  Sets parameters for the standard move program and starts it.
    **  See {@link #moveD(int, double, double, double, int, int[], int[])}
    **  for a description of the parameters.
    **
    ***************************************************************************
    */
    private int initMove(int type, double dist, double time, double mvFract,
                         int nSamp, int[] iParms, int[] fParms)
    {
        double[] parms = new double[P_PARMS + iParms.length + fParms.length];
        double vel, acc, jrk = 0.0;

        /*
        **  Return if distance is zero
        */
        if (dist == 0.0) return 0;

        /*
        **  Set up the parameter values used by the move program
        */
        parms[P_DRVON] = driveOn ? 1.0 : 0.0;
        parms[P_DIST] = config.motorDistance(dist);
        if (mvFract < 0) mvFract = 0;
        else if (mvFract > 0.9) mvFract = 0.9;
        vel = Math.abs(2 * dist / ((1 + mvFract) * time));
        acc = Math.abs(4 * dist / ((1 - mvFract * mvFract) * time * time));
        if (type == MOV_TYP_SCURVE) {
            acc *= 2.0;
            jrk = acc * acc / vel;
        }
        if (nSamp > 0) {
            parms[P_SMPTM] = Math.floor(1000 * time / nSamp);
            parms[P_SMPCT] = Math.floor(1000 * time / parms[P_SMPTM] + 5.0);
        }
        else {
            parms[P_SMPTM] = 1.0;
            parms[P_SMPCT] = 1.0;
        }
        parms[P_NIPRM] = iParms.length;
        parms[P_NFPRM] = fParms.length;
        for (int j = 0; j < iParms.length; j++)
            parms[P_PARMS + j] = iParms[j];
        for (int j = 0; j < fParms.length; j++)
            parms[P_PARMS + iParms.length + j] = fParms[j];

        /*
        **  Check that there's enough space for samples
        */
        int nWord = 13 + ((int)parms[P_SMPCT] + 2)
                            * (iParms.length + fParms.length);
        if (nWord > 1024)
            return -MovementStatus.STS_INSUFF_VBLE_MEM;

        /*
        **  Copy the parameter values to the controller
        */
        acr.setFloatParm(AcrComm.PPU_PRM, (float)config.ppu());
        acr.setFloatParm(AcrComm.JOG_VEL_PRM, (float)vel);
        acr.setFloatParm(AcrComm.JOG_ACC_PRM, (float)acc);
        acr.setFloatParm(AcrComm.JOG_DEC_PRM, (float)acc);
        acr.setFloatParm(AcrComm.JOG_JRK_PRM, (float)jrk);
        int count = acr.setUserParms(0, parms, parms.length);
        if (count < parms.length)
            return -MovementStatus.STS_INSUFF_USER_PARM;

        /*
        **  Clear the sync bit, run the move program and wait for sync bit
        */
        acr.clearBit(AcrComm.MOVE_SYNC_BIT);
        acr.sendStr("run prog" + MOVE_PROG, 0);
        acr.sendStr("inh " + AcrComm.MOVE_SYNC_BIT, 0);
        sleep(0.1);

        return (int)parms[P_SMPCT];
    }


   /**
    ***************************************************************************
    **
    **  Carries out the move with parameter sampling
    **
    ***************************************************************************
    */
    private long runMove(boolean dummy, Hall hall)
    {
        /*
        **  Get the start times, standard and high-resolution
        */
        long sTime = System.currentTimeMillis();
        if (hall != null)
            hall.startTime = 1000 * sTime - System.nanoTime() / 1000;
        if (dummy) return sTime;

        /*
        **  Start the move by raising the input line
        */
        acd.dioSetBit(dioOPort, dioOLine);

        /*
        **  Wait for the move and sampling to complete
        */
        acr.sendStr("inh " + AcrComm.JOG_ACTV_BIT, 0);
        acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
        acr.sendStr("inh -" + AcrComm.SAMP_ARM_BIT, 0);

        /*
        **  Lower the input line
        */
        acd.dioClrBit(dioOPort, dioOLine);

        return sTime;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the motor controller. Sets parameters, motion control programs, etc.
    **
    ***************************************************************************
    */
    private void acrInitialize()
    {
        /*
        **  Do basic initialization
        */
        
        // Configuration of the move program.
        acr.sendStr("prog" + MOVE_PROG, 0); 
        // Axis 0 will use encoder 0 for position feedback, DAC 0 for the motor control signal, and encoder 0
        // for velocity feedback.
        acr.sendStr("attach axis0 enc0 dac0 enc0", 0);
        // The program will use master 0. Masters generate motion profiles.
        acr.sendStr("attach master0", 0);
        // The only slave of master 0 will be axis 0, labelled the "x" axis.
        acr.sendStr("attach slave0 axis0 \"x\"", 0);
        
        // Global init. of the controller.
        acr.sendStr("sys", 0);
        // Stop all running programs and all motions.
        acr.sendStr("halt all", 0);
        // Set all programs to empty.
        acr.sendStr("new all", 0);
        // Clear all memory allocation.
        acr.sendStr("clear", 0);
        // Allocate space for global aliases (slots for #DEFINE).
        acr.sendStr("dim def(10)", 0);
        // Allocate 256 global variables. type = double.
        acr.sendStr("dim p(256)", 0);
        // Allocate 32K, 8K, 8K and 8K for the move, calibration, test, and temp programs, respectively.
        acr.sendStr("dim prog" + MOVE_PROG + "(32768)", 0);
        acr.sendStr("dim prog" + CAL_PROG + "(8192)", 0);
        acr.sendStr("dim prog" + TEST_PROG + "(8192)", 0);
        acr.sendStr("dim prog" + TEMP_PROG + "(8192)", 0);
        // Turn off the motor driver for axis 0.
        acr.sendStr("axis0 drive off", 0);
        // Number of encoder Pulses Per Unit of travel. For us the unit is 1 mm.
        acr.setFloatParm(AcrComm.PPU_PRM, (float)config.ppu());
        // Set the  Hardware Limit Deceleration, the deceleration to use when motion hits a hardware limit
        // or when motion is killed. If this is mm/sec^2 then the decel is 5g.
        acr.sendStr("axis0 hldec 50000", 0);
        // Set Software Limits to Motion, positive and negative directions of motion.
        acr.sendStr("axis0 slm (" + config.posSoftLimit() + ","
                      + config.negSoftLimit() + ")", 0);
        // Software Limit Deceleration.
        acr.sendStr("axis0 sldec 50000", 0);
        // Enable hardware motion (End Of Travel) limits.
        acr.setBit(AcrComm.ENAB_PEOT_BIT);
        acr.setBit(AcrComm.ENAB_NEOT_BIT);
        // Enable software motion limits.
        acr.setBit(AcrComm.ENAB_PSLM_BIT);
        acr.setBit(AcrComm.ENAB_NSLM_BIT);
        // Do NOT use inverted logic for the home and EOT sensors; we want
        // these to be ON only when the condition is true.
        acr.clearBit(AcrComm.INVT_HOME_BIT);
        acr.clearBit(AcrComm.INVT_PEOT_BIT);
        acr.clearBit(AcrComm.INVT_NEOT_BIT);
        // Make sure that the Kill Move bit is off, otherwise no motion commands
        // will be obeyed.
        acr.clearBit(AcrComm.KILL_MOVE_BIT);
        // Set the feedback loop gains for axis 0. The motor controller uses the standard
        // Proportional-Integral-Derivative (PID) technique.
        acr.setFloatParm(AcrComm.PGAIN_PRM, 0.04f);
        acr.setFloatParm(AcrComm.IGAIN_PRM, 0.0003f);
        acr.setFloatParm(AcrComm.DGAIN_PRM, 0.001f);
        // Enable motor control for axis 0.
        acr.sendStr("axis0 on", 0);
        // Set the torque limits for both directions of motion, axis 0.
        acr.setFloatParm(AcrComm.POS_TLM_PRM, (float)TORQUE_LIM);
        acr.setFloatParm(AcrComm.NEG_TLM_PRM, -(float)TORQUE_LIM);

        /*
        **  Load the move program
        */
        acr.sendStr("prog" + MOVE_PROG, 0);
        // Accept program text until the next PEND.
        acr.sendStr("program", 0);
        // Deallocate all memory allocated for local variables and arrays.
        acr.sendStr("clear", 0);
        // Reset to defaults all system parameters and flags dealing with sampling.
        acr.sendStr("samp clear", 0);
        // Allocate long (32-bit) integer local variables, named LV0 - LV9.
        acr.sendStr("dim lv(10)", 0);
        // Allocate space for headers of arrays of longs. The space for the actual array elements is
        // allocated below. The number of arrays is determined by parameter no. P_NIPRM. The syntax "(pn)"
        // is a reference to the value of parameter n. It's used when the command syntax would not
        // allow a normal parameter reference of the forn "pn". Parameters 0-4095 are reserved for the user.
        acr.sendStr("dim la(p" + P_NIPRM + ")", 0);
        // Allocation of arrays of "single" (32-bit float).
        acr.sendStr("dim sa(p" + P_NFPRM + ")", 0);
        // For each array of longs ...
        acr.sendStr("for lv0 = 0 to (p" + P_NIPRM + " - 1) step 1", 0);
        //     Give the array one element per desired sample.
        acr.sendStr("dim la(lv0)(p" + P_SMPCT + ")", 0);
        //     Samples from channel 0, 1, ... will be stored in long array 0, 1, ...
        acr.sendStr("samp(lv0) base la(lv0)", 0);
        //     Each sample value for channel lv0 will come from the parameter whose number
        //     is given by parameter lv0 + P_PARMS. Could be either a system or a user parameter.
        acr.sendStr("samp(lv0) src p(p(lv0 + " + P_PARMS + "))", 0);
        //     End of loop.
        acr.sendStr("next", 0);
        // The same deal for samples that are singles (float) instead of integers.
        acr.sendStr("for lv0 = 0 to (p" + P_NFPRM + " - 1) step 1", 0);
        acr.sendStr("dim sa(lv0)(p" + P_SMPCT + ")", 0);
        acr.sendStr("samp(lv0 + p" + P_NIPRM + ") base sa(lv0)", 0);
        acr.sendStr("samp(lv0 + p" + P_NIPRM + ") src p(p(lv0 + p" + P_NIPRM
                      + " + " + P_PARMS + "))", 0);
        acr.sendStr("next", 0);
        // Continuous-mode sampling: when started, sample at every servo interrupt or at regular intervals.
        acr.sendStr("clr " + AcrComm.SAMP_MODE_BIT, 0);
        // Start sampling when motion starts.
        acr.sendStr("samp trg " + AcrComm.JOG_ACTV_BIT, 0);
        // Set sampling period.
        acr.sendStr("p" + AcrComm.SAMP_PERD_PRM + " = p" + P_SMPTM, 0);
        // If the motor driver is not left on in between moves, turn it on now.
        acr.sendStr("if (p" + P_DRVON + " = 0) then axis0 drive on", 0);
        //  Delay (dwell) for one-tenth second.
        acr.sendStr("dwl 0.1", 0);
        acr.sendStr("set " + AcrComm.MOVE_SYNC_BIT, 0);
//        acr.sendStr("inh " + AcrComm.INPUT4_BIT, 0);
        // Enable hardware position capture using the position (first) encoder of axis 0.
        // Capture occurs on the rising edge of the third external input (which is what?).
        acr.sendStr("intcap axis0 " + AcrComm.ICM_RISE_3RD, 0);
        // Wait until the capture-complete bit is set.
        acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0);
        // Save the current system time and the position encoder value for axis 0.
        acr.sendStr("lv0 = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("lv1 = p" + AcrComm.ENC_POSN_PRM, 0);
        // Enable sampling.
        acr.sendStr("set " + AcrComm.SAMP_ARM_BIT, 0);
        // Start the motion. Move the distance given by parameter P_DIST using the
        // currently set jog parameters, set by configMove() or initMove().
        acr.sendStr("axis0 jog inc (p" + P_DIST + ")", 0);
        // Wait for motion to complete.
        acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
        // Save the new position encoder value.
        acr.sendStr("lv2 = p" + AcrComm.ENC_POSN_PRM, 0);
        // Wait for samples to be written.
        acr.sendStr("inh -" + AcrComm.SAMP_ARM_BIT, 0);
        // If the motor driver is not left on in between moves, turn it off now.
        acr.sendStr("if (p" + P_DRVON + " = 0) then axis0 drive off", 0);
        acr.sendStr("endp", 0);

        /*
        **  Reset the command input to be system-wide
        */
        acr.sendStr("sys", 0);
    }


   /**
    ***************************************************************************
    **
    **  Sleeps for a time
    **
    ***************************************************************************
    */
    private void sleep(double secs)
    {
        long nsecs = (long)(1e9 * secs);
        try {
            Thread.sleep(nsecs / 1000000, (int)(nsecs % 1000000));
        }
        catch (java.lang.InterruptedException e) {
        }
    }

}
