package org.lsst.ccs.subsystems.shutter;

import java.io.Serializable;

import java.util.List;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import static org.lsst.ccs.command.annotations.Command.CommandType.ACTION;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.NORMAL;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;

import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.framework.annotations.ConfigChanger;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.subsystems.shutter.common.BladeSet;
import org.lsst.ccs.subsystems.shutter.common.MovementHistory;
import org.lsst.ccs.subsystems.shutter.status.BladePositionResult;
import org.lsst.ccs.subsystems.shutter.status.CloseShutterStatus;
import org.lsst.ccs.subsystems.shutter.status.OpenShutterStatus;
import org.lsst.ccs.subsystems.shutter.status.MoveToPositionStatus;
import org.lsst.ccs.subsystems.shutter.status.MovementHistoryStatus;
import org.lsst.ccs.subsystems.shutter.status.ReadyForActionStatus;
import org.lsst.ccs.subsystems.shutter.status.ShutterConfigStatus;
import org.lsst.ccs.subsystems.shutter.status.StatusKey;
import org.lsst.ccs.subsystems.shutter.status.TakeExposureStatus;
import org.lsst.ccs.subsystems.shutter.status.UnsafeMoveStatus;

/**
 * Main module for the shutter subsystem.
 * 
 * <p>This module manages two objects that implement the BladeSet interface, one for each side of the
 * shutter. Generally these will either both be instances of BladeSetSimulator or of BladeSetDrvr.
 * Coordinated motions of one or both the blade sets are carried out in response to commands from the
 * CCS command bus. Motion histories for the blade sets are published on the CCS status bus.
 * 
 * <p>Configuration parameters:
 * <ul>
 * <li>bladeMovementTime (double) - the duration of blade movements in seconds.</li>
 * <li>bladeSetEndTol (double) - a blade set whose relative position is less than or equal to this is
 *     considered retracted. Likewise one whose position is greater than or equal to 1 - this
 *     is considered fully extended.</li>
 * <li>bladeSetApproachMin (double) - If the blade sets were touching then (relative position set 0)
 *     + (relative position set 1) would equal 1.0. Instead we require that the sum be
 *     less than or equal to this value.</li>
 * </ul>
 * @author azemoon
 * @author tether
 */
public class ShutterModule extends Module {
    
    @ConfigurationParameter(name="bladeMovementTime",
        description="The duration in seconds of any blade set motion")
    private volatile double bladeMovementTime;
    
    @ConfigurationParameter(name="bladeSetEndTol",
        description="Tolerance, as relative position, " +
            "for deciding whether a blade set is at its opened or closed position")
    private volatile double bladeSetEndTol;
    
    @ConfigurationParameter(name="bladeSetApproachMin",
        description="The minimum distance, as relative position, to keep between"+
            " the blade set edges")
    private volatile double bladeSetApproachMin;

    private final List<BladeSet> bladeSets;
    private final Logger mylog = Logger.getLogger("org.lsst.ccs.subsystem.shutter");
    
    /**
     * Constructs the main module for the shutter subsystem.
     * @param name the component name (module name)
     * @param bladeSets a list of BladeSet objects not yet assigned to positions in the shutter
     */
    public ShutterModule(
        String name,
        List<BladeSet> bladeSets)
    {
        super(name);
        this.bladeSets = bladeSets;
        
        // These defaults should be overridden by a configuration.
        this.bladeMovementTime   = 1.0;
        this.bladeSetEndTol      = 0.001;
        this.bladeSetApproachMin = 0.001;
    }

    /**
     * Checks that the module has been constructed correctly. There must be two blade sets.
     */
    @Override
    public void initModule() {
        mylog.info("[ShutterModule] Initializing the Shutter module ");

        // Check that we have two BladeSets
        if (bladeSets.size() != 2) {
            logFatalMessageAndThrowException("[ShutterModule] The Shutter Module requires two bladeSets when built");
        }
        
    }

    /**
     * Perform hardware initialization.
     */
    @Override
    public void start() {
        getBladeSet(0).init();
        getBladeSet(1).init();
    }
    
    private final static ScheduledExecutorService exec = Executors.newSingleThreadScheduledExecutor();

    /**
     * Shut down activity in this module. It uses an executor service which must be shut down.
     */
    @Override
    public void shutdownNow() { super.shutdownNow(); exec.shutdown();}

    /**
     * Gets one of the two BladeSet objects.
     * @param index selects the desired BladeSet
     * @return the selected BladeSet object
     * @exception IndexOutOfBoundsException if the index is not zero or one
     */
    public final BladeSet getBladeSet(int index) {
        return bladeSets.get(index);
    }

    /**
     * The default value for the number of seconds each blade set should take to move
     * when taking an exposure.
     * @return the movement time
     */
    public double getBladeMovementTime() {
        return bladeMovementTime;
    }

    /**
     * Gets the tolerance used when comparing blade set position against 0 or 1.
     * @return The tolerance value.
     */
    public double getBladeSetEndTol() {
        return this.bladeSetEndTol;
    }
    
    /**
     * Gets the current minimum approach distance for the blade sets (relative position).
     * @return The min. distance.
     */
    public double getBladeSetApproachMin() {
        return this.bladeSetApproachMin;
    }
    
    /**
     * Gets the current relative position of both blade sets. See {@link BladePosition}. May
     * cause hardware operations.
     * @return the positions
     */
    @Command(type=ACTION,
        level=NORMAL,
        description="Gets the current relative positions of both blade sets")
    public BladePositionResult getBladeSetPositions()
    {
        return new BladePositionResult(
            getBladeSet(0).getCurrentPosition(),
            getBladeSet(1).getCurrentPosition()
        );
    }

    /**
     * Action command to move a single blade set.
     * <p>Engineering mode only. Blocks until movement is complete.
     * @param index           The index of the blade set to move.
     * @param targetPosition  The desired relative position after the move.
     */
    @Command(type=ACTION, level=ENGINEERING1, description="Moves one of the blade sets")
    public void moveToPosition(
            @Argument(name="index", description="The blade set index (0 or 1)")
            final int index,
            @Argument(name="targetPosition", description="The desired relative position of the blade set")
            final double targetPosition)
    {
        // If we're going to perform an arbitrary blade set movement commanded by
        // some outside agency then we'll have to check safety.
        if (moveIsUnsafe(index, targetPosition)) {
            publish(StatusKey.UNSAFE_MOVE, new UnsafeMoveStatus());
        }
        else {
            publish(StatusKey.MOVE_BLADE_SET,
                    new MoveToPositionStatus(getBladeSet(index).getCurrentPosition(),
                                           index,
                                           targetPosition,
                                           getBladeMovementTime()
                    )
            );
            final MovementHistory hist =
                getBladeSet(index).moveToPosition(targetPosition, getBladeMovementTime());
            publish(StatusKey.MOVEMENT, new MovementHistoryStatus(true, true, hist));
        }
        publish(StatusKey.READY, new ReadyForActionStatus(true));
    }

    /**
     * Perform two blade set motions with a given delay between them.
     * @param ifirst the index of the first blade set to move
     * @param firstTargetPosition where to move the first blade set
     * @param secondTargetPosition where to move the second blade set
     * @param secondStartDelay the delay in seconds between starting the firt move and the second.
     * 
     * Explicit scheduling is always needed when simulating the shutter since the simulation uses the system
     * clock time as the starting time. For a real shutter it won't hurt and is sometimes needed.
     */
    private void moveTwoBladeSets(
        int ifirst,
        double firstTargetPosition,
        double secondTargetPosition,
        double secondStartDelay)
    {
        final BladeSet first = getBladeSet(ifirst);
        final BladeSet second = getBladeSet(1 - ifirst);
        final Future<MovementHistory> fuhist1 =
            exec.schedule(() -> first.moveToPosition(firstTargetPosition, bladeMovementTime),
                0, TimeUnit.MILLISECONDS);
        final Future<MovementHistory> fuhist2 =
            exec.schedule(() -> second.moveToPosition(secondTargetPosition, bladeMovementTime),
                Math.round(1000.0 * secondStartDelay), TimeUnit.MILLISECONDS);
        // Wait for the movements to complete.
        try {
            final MovementHistory hist1 = fuhist1.get();
            publish(StatusKey.MOVEMENT, new MovementHistoryStatus(true, false, hist1));
            final MovementHistory hist2 = fuhist2.get();
            publish(StatusKey.MOVEMENT, new MovementHistoryStatus(false, true, hist2));
        }
       catch (InterruptedException e) {
            // This thread was interrupted while waiting. Cancel the tasks.
           fuhist1.cancel(true);
           fuhist2.cancel(true);
       }
        catch (ExecutionException e) {
            // A task threw an exception, obtainable using getCause().
            fuhist1.cancel(true);
            fuhist2.cancel(true);
            launderThrowable(e.getCause());
        }
        finally {
            publish(StatusKey.READY, new ReadyForActionStatus(true));
        }
   }
    /**
     * An action command that opens the shutter for the given exposure time then closes it.
     * Normal and engineering mode. Blocks until the exposure is complete.
     * @param exposureTime the exposure time in seconds
     */
    @Command(type=ACTION, level=NORMAL, description="Performs a timed exposure (open, wait, close)")
    public void takeExposure(
            @Argument(name="exposureTime", description="The duration of the exposure in seconds")
            final double exposureTime)
    {
        final BladePositionResult pos = this.getBladeSetPositions();
        final int ifirst = (pos.getPos0() > pos.getPos1()) ? 0 : 1;
        publish(StatusKey.TAKE_EXPOSURE, new TakeExposureStatus(ifirst, bladeMovementTime, exposureTime));
        moveTwoBladeSets(ifirst, 0.0, 1.0, exposureTime);
    }

    /**
     * Closes the shutter. First blade set 0 is retracted and then set 1 is extended.
     */
   @Command(type=ACTION, level=ENGINEERING1, description="Closes the shutter")
   public void closeShutter() {
       publish(StatusKey.CLOSE_SHUTTER, new CloseShutterStatus(0, getBladeMovementTime()));
       moveTwoBladeSets(0, 0.0, 1.0, bladeMovementTime);
   }

   /**
    * Opens the shutter. First blade set 0 is retracted, then set 1.
    */
   @Command(type=ACTION, level=ENGINEERING1, description="Opens the shutter")
   public void openShutter() {
       publish(StatusKey.OPEN_SHUTTER, new OpenShutterStatus(0, getBladeMovementTime()));
       moveTwoBladeSets(0, 0.0, 0.0, bladeMovementTime);
   }

   /**
    * Does nothing at present.
    */
   @Command(type=ACTION, level=ENGINEERING1, description="Performs a shutter motion calibration")
   public void calibrate() {
       publish(StatusKey.READY, new ReadyForActionStatus(true));
   }
   
   /**
    * Throws any given RuntimeException or Error, else throws IllegalStateException.
    * Used to handle arbitrary Throwables wrapped inside instances of ExcecutionException.
    * Any that are checked exceptions are wrapped in an IllegalStateException, which is unchecked.
    * @param thr The Throwable from an ExecutionException.
    */
   private static void launderThrowable(Throwable thr) {
       if (thr instanceof RuntimeException)
           throw (RuntimeException)thr;
       else if (thr instanceof Error)
           throw (Error)thr;
       else
           throw new IllegalStateException("Not unchecked", thr);
   }

    private void logFatalMessageAndThrowException(String msg) {
        mylog.fatal(msg);
        throw new RuntimeException(msg);
    }
    
    /** Publishes subsystem-specific  data to the status bus with a key derived from an enum.
        @param key One of the StatusKey enumerators.
        @param status The serializable object to be published.
    */
    private void publish(StatusKey key, Serializable status) {
        getSubsystem()
            .publishSubsystemDataOnStatusBus(new KeyValueData(key.getKey(), status));
    }
    
    /**
     * Is a blade set fully retracted?
     * @param index the index of the blade set (0 or 1).
     * @return the answer
     */
    private boolean bladeSetIsRetracted(int index) {
        return Math.abs(getBladeSet(index).getCurrentPosition()) < this.getBladeSetEndTol();
    }
    
    /**
     * Is a blade set fully extended?
     * @param index the index of the blade set (0 or 1).
     * @return the answer
     */
    private boolean bladeSetIsExtended(int index) {
        return Math.abs(1.0 - getBladeSet(index).getCurrentPosition()) < this.getBladeSetEndTol();
    }
    
    /**
     * Determines if a commanded move of a single blade set is unsafe. The two blade sets
     * must not end up being too close.
     * @param index the index of the blade set to be moved.
     * @param targetPosition
     * @return the answer
     */
    private boolean moveIsUnsafe(int index, double targetPosition) {
        return (1.0 - targetPosition - getBladeSet(1 - index).getCurrentPosition()) <=
            this.getBladeSetApproachMin();
    }
}
