package org.lsst.ccs.subsystems.shutter.driver;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.drivers.iocard.AccesDio;
import org.lsst.ccs.drivers.iocard.Helios;
import org.lsst.ccs.drivers.parker.AcrComm;
import org.lsst.ccs.subsystems.shutter.common.BladePosition;
import org.lsst.ccs.subsystems.shutter.common.HallTransition;
import org.lsst.ccs.subsystems.shutter.common.MovementHistory;
import org.lsst.ccs.subsystems.shutter.interfaces.BladeSet;
import org.lsst.ccs.subsystems.shutter.interfaces.HallSensorListener;
import org.lsst.ccs.subsystems.shutter.interfaces.MovementHistoryListener;

/**
 ***************************************************************************
 **
 **  Routines to perform shutter blade operations
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class BladeSetDrvr implements BladeSet {

   /**
    ***************************************************************************
    **
    **  Inner class to hold an item of raw Hall sensor data
    **
    ***************************************************************************
    */
    public static class HallItem {

        public long     time;    // Transition time (nano timer)
        public int      sensor;  // Sensor ID
        public boolean  open;    // True if sensor opened; false if closed

    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold a set of raw Hall sensor data
    **
    ***************************************************************************
    */
    public static class Hall {

        public long        startTime;  // Start time (nano timer)
        public int         count;      // Count of items in use
        public int         state;      // State of the switches
        public HallItem[]  data;       // Array of data items

        public Hall()
        {
            this(100);
        }

        public Hall(int maxItem)
        {
            data = new HallItem[maxItem];
            for (int j = 0; j < maxItem; j++)
                data[j] = new HallItem();
        }

    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold an item of sampled data
    **
    ***************************************************************************
    */
    public static class Sample {
        public int      time;    // Microseconds since move start time
        public int[]    iVals;   // Array of sampled integer parameter values
        public float[]  fVals;   // Array of sampled float parameter values
    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold a set of special move data samples
    **
    ***************************************************************************
    */
    public static class SampData {

        public int                  status;     // Move completion status
        public long                 startTime;  // Move start time
        public float                startPosn;  // Move start position
        public long                 endTime;    // Move end time
        public float                endPosn;    // Move end position
        public ArrayList<Sample >   sList = new ArrayList<Sample>();
                                                // List of sampled data
    }

   /**
    ***************************************************************************
    **
    **  Public constants
    **
    ***************************************************************************
    */
    public final static int   MOV_TYP_SCURVE = 0,
                              MOV_TYP_TRAP   = 1,
                              HOM_OPT_NDIRN  = 0x01,
                              HOM_OPT_PFINAL = 0x02;
    public final static float TORQUE_LIM   = 10.0f;

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int P_DRVON = 0,
                             P_DIST  = 1,
                             P_SMPTM = 2,
                             P_SMPCT = 3,
                             P_NIPRM = 4,
                             P_NFPRM = 5,
                             P_PARMS = 6,
                             MOVE_PROG = 0,
                             CAL_PROG  = 1,
                             TEST_PROG = 2,
                             TEMP_PROG = 3;

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private BladeSetConfiguration config = new BladeSetConfiguration();
    private BladeSetCalibration calData;
    private ArrayList<HallSensorListener> hsListeners = new ArrayList();
    private ArrayList<MovementHistoryListener> mhListeners = new ArrayList();
    private int bladeIndex, dioHPort, dioIPort, dioILine, dioOPort, dioOLine;
    private boolean driveOn = false;
    private AccesDio acd;
    private Helios hel;
    private AcrComm acr;


   /**
    ***************************************************************************
    **
    **  Sets the blade set index, and hence its configuration
    **
    ***************************************************************************
    */
    @Override
    public void setIndex(int index)
    {
        config.setIndex(index);
        bladeIndex = index;
        dioHPort = config.getDioHPort();
        dioIPort = config.getDioIPort();
        dioILine = config.getDioILine();
        dioOPort = config.getDioOPort();
        dioOLine = config.getDioOLine();

        acr = new AcrComm(config.getNode(), true);
        acrInitialize();

        acd = new AccesDio();
        acd.init(config.getDioAddr(), config.getDioLevel());
        acd.dioConfig(config.getDioConf());

        hel = new Helios();
        hel.init(config.getAdAddr(), config.getAdLevel());
        hel.adConfig(3, true, false);
        hel.adSetChan(config.getAdTempChn());
    }


   /**
    ***************************************************************************
    **
    **  Gets the blade set index
    **
    ***************************************************************************
    */
    @Override
    public int getIndex()
    {
        return bladeIndex;
    }


   /**
    ***************************************************************************
    **
    **  Moves the blade set to a given relative (i.e. fractional) position
    **
    ***************************************************************************
    */
    @Override
    public void moveToPosition(float relPosition, float time)
    {
        MovementHistory
            history = moveH(MOV_TYP_SCURVE,
                            config.position(relPosition) - getPosition(),
                            time, 0.0f, config.getNumSamp());
        for (MovementHistoryListener l: mhListeners)
            l.movementHistoryFinalized(history);
    }


   /**
    ***************************************************************************
    **
    **  Returns the current blade set relative (i.e. fractional) position
    **
    ***************************************************************************
    */
    @Override
    public float getCurrentPosition()
    {
        return config.relPosition(acr.getIntParm(AcrComm.ENC_POSN_PRM));
    }


   /**
    ***************************************************************************
    **
    **  Returns the configuration object being used
    **
    ***************************************************************************
    */
    public BladeSetConfiguration getBladeSetConfiguration()
    {
        return config;
    }


   /**
    ***************************************************************************
    **
    **  Adds a movement history listener
    **
    ***************************************************************************
    */
    @Override
    public void addMovementHistoryListener(MovementHistoryListener l)
    {
        mhListeners.add(l);
    }


   /**
    ***************************************************************************
    **
    **  Removes a movement history listener
    **
    ***************************************************************************
    */
    @Override
    public void removeMovementHistoryListener(MovementHistoryListener l)
    {
        mhListeners.remove(l);
    }


   /**
    ***************************************************************************
    **
    **  Adds a Hall sensor listener
    **
    ***************************************************************************
    */
    @Override
    public void addHallSensorListener(HallSensorListener l)
    {
        hsListeners.add(l);
    }


   /**
    ***************************************************************************
    **
    **  Removes a Hall sensor listener
    **
    ***************************************************************************
    */
    @Override
    public void removeHallSensorListener(HallSensorListener l)
    {
        hsListeners.remove(l);
    }


   /**
    ***************************************************************************
    **
    **  Gets the current blade set position as a distance
    **
    ***************************************************************************
    */
    public float getPosition()
    {
        return config.position(acr.getIntParm(AcrComm.ENC_POSN_PRM));
    }


   /**
    ***************************************************************************
    **
    **  Turns the drive on and leaves it on
    **
    ***************************************************************************
    */
    public void setDriveOn()
    {
        enableDrive();
        driveOn = true;
    }


   /**
    ***************************************************************************
    **
    **  Turns the drive off
    **
    ***************************************************************************
    */
    public void setDriveOff()
    {
        driveOn = false;
        disableDrive();
    }


   /**
    ***************************************************************************
    **
    **  Enables the drive
    *
    ***************************************************************************
    */
    public void enableDrive()
    {
        if (!driveOn) acr.sendStr("axis 0 drive on", 0);
    }


   /**
    ***************************************************************************
    **
    **  Disables the drive
    *
    ***************************************************************************
    */
    public void disableDrive()
    {
        if (!driveOn) acr.sendStr("axis 0 drive off", 0);
    }


   /**
    ***************************************************************************
    **
    **  Calibrates the Hall switch positions, moving in both directions
    **
    ***************************************************************************
    */
    public BladeSetCalibration calibrate(float dist, float step, float time)
    {
        BladeSetCalibration calib = new BladeSetCalibration();

        dist = Math.copySign(dist, config.getClosed() - config.getOpen());
        calibLeg(dist, step, time, calib);
        calibLeg(-dist, step, time, calib);

        return calib;
    }


   /**
    ***************************************************************************
    **
    **  Calibrates the Hall switch positions (new test)
    **
    ***************************************************************************
    */
    public BladeSetCalibration calibNew(float dist, float time)
    {

        BladeSetCalibration calib = new BladeSetCalibration();

        /*
        **  Set the first move to be in the open-to-closed direction
        */
        dist = Math.copySign(dist, config.getClosed() - config.getOpen());

        /*
        **  Set up Hall switch reading via switch interrupts
        */
        Hall hall = new Hall();
        hall.state = acd.dioInp(dioHPort);
        acd.attachInt(1 << dioHPort, this, "readHallC", hall);

        /*
        **  Set initial state of output line and expected transition
        */
        int capMode;
        if (hall.state == 0xff) {
            acd.dioSetBit(dioOPort, dioOLine);
            capMode = AcrComm.ICM_FALL_3RD;
        }
        else {
            acd.dioClrBit(dioOPort, dioOLine);
            capMode = AcrComm.ICM_RISE_3RD;
        }

        /*
        **  Load the capture program
        */
        acr.sendStr("prog" + CAL_PROG, 0);
        acr.sendStr("program", 0);
        acr.sendStr("clear", 0);
        acr.sendStr("dim lv(10)", 0);
        acr.sendStr("dim la(1)", 0);
        acr.sendStr("dim la0(100)", 0);
        acr.sendStr("lv0 = 0", 0);
        acr.sendStr("lv1 = " + capMode, 0);
        acr.sendStr("set " + AcrComm.CAL_SYNC_BIT, 0);
        acr.sendStr("while (1 = 1)", 0);
        acr.sendStr("intcap axis0 (lv1)", 0);
        acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0);
        acr.sendStr("la0(lv0) = p" + AcrComm.CAP_POSN_PRM, 0);
        acr.sendStr("lv0 = lv0 + 1", 0);
        acr.sendStr("lv1 = " + (AcrComm.ICM_RISE_3RD + AcrComm.ICM_FALL_3RD)
                      + " - lv1", 0);
        acr.sendStr("wend", 0);
        acr.sendStr("endp", 0);
        acr.sendStr("sys", 0);

        /*
        **  Do a move in each direction, capturing the transitions
        */
        calibLegNew(dist, time, hall, calib);
        calibLegNew(-dist, time, hall, calib);

        /*
        **  Cancel Hall switch interrupts and clear output line
        */
        acd.detachInt();
        acd.dioClrBit(dioOPort, dioOLine);

        return calib;
    }


   /**
    ***************************************************************************
    **
    **  Sets the Hall calibration data
    **
    ***************************************************************************
    */
    public void setCalib(BladeSetCalibration list)
    {
        calData = list;
    }


   /**
    ***************************************************************************
    **
    **  Gets the Hall calibration data
    **
    ***************************************************************************
    */
    public BladeSetCalibration getCalib()
    {
        return calData;
    }


   /**
    ***************************************************************************
    **
    **  Performs a move
    **
    ***************************************************************************
    */
    public int move(int type, float dist, float time, float mvFract)
    {

        /*
        **  Return if distance is zero
        */
        if (dist == 0f) return 0;

        /*
        **  Configure and enable the drive
        */
        configMove(type, dist, time, mvFract);

        /*
        **  Do the move and wait for motion to stop
        */
        acr.sendStr("axis0 jog inc " + config.motorDistance(dist), 0);
        acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);

        /*
        **  Disable the drive
        */
        disableDrive();

        return moveStatus();
    }


   /**
    ***************************************************************************
    **
    **  Performs a move, recording various parameters periodically
    **
    ***************************************************************************
    */
    public SampData moveD(int type, float dist, float time, float mvFract,
                          int nSamp, int[] iParms, int[] fParms)
    {

        /*
        **  Set the parameters used by the move program and run it
        */
        int nsamp = initMove(type, dist, time, mvFract, nSamp, iParms, fParms);
        SampData data = new SampData();
        data.startPosn = getPosition();
        data.startTime = runMove(nsamp <= 0, null);
        data.endTime = System.currentTimeMillis();
        data.endPosn = getPosition();

        /*
        **  Return if error or null move
        */
        if (nsamp <= 0) {
            data.status = -nsamp;
            return data;
        }

        /*
        **  Retrieve the sampled parameters
        */
        int[][] iVals = new int[iParms.length][nsamp];
        float[][] fVals = new float[fParms.length][nsamp];
        int[] time0 = new int[1];

        int tsamp = acr.getVars(MOVE_PROG, 0, 1, time0);
        for (int j = 0; j < iParms.length; j++)
            tsamp += acr.getVars(MOVE_PROG, j, 0, nsamp, iVals[j]);
        for (int j = 0; j < fParms.length; j++)
            tsamp += acr.getVars(MOVE_PROG, j, 0, nsamp, fVals[j]);
        if (tsamp != nsamp * (iParms.length + fParms.length) + 1) {
            data.status = MovementStatus.STS_MSSG_PGM_VBLES;
            return data;
        }

        /*
        **  Set movement completion status
        */
        data.status = moveStatus();

        /*
        **  Get index of sampled time; -1 if time not being sampled
        */
        int tIndx = -1;
        for (int j = 0; j < iParms.length; j++)
            if (iParms[j] == AcrComm.SYS_CLOCK_PRM) tIndx = j;

        /*
        **  Output the sampled values
        */
        for (int j = 0; j < nsamp; j++) {
            Sample sItem = new Sample();
            sItem.time = 0;
            sItem.iVals = new int[iParms.length];
            for (int k = 0; k < iParms.length; k++) {
                sItem.iVals[k] = iVals[k][j];
                if (k == tIndx) {
                    sItem.iVals[k] -= time0[0];
                    sItem.time = 1000 * sItem.iVals[k];
                }
            }
            sItem.fVals = new float[fParms.length];
            for (int k = 0; k < fParms.length; k++)
                sItem.fVals[k] = fVals[k][j];
            data.sList.add(sItem);
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Performs a move, recording both Hall switch transitions and periodic
    **  encoder values
    **
    ***************************************************************************
    */
    public MovementHistory moveH(int type, float dist, float time,
                                 float mvFract, int nSamp)
    {

        /*
        **  Set up Hall switch reading via switch interrupts
        */
        Hall hall = new Hall();
        hall.state = acd.dioInp(dioHPort);
        acd.attachInt(1 << dioHPort, this, "readHallM", hall);

        /*
        **  Set the parameters used by the move program and run it
        */
        int[] iParms = {AcrComm.SYS_CLOCK_PRM, AcrComm.ENC_POSN_PRM,
                        AcrComm.PRIM_AXIS_PRM},
              fParms = {};
        int nsamp = initMove(type, dist, time, mvFract, nSamp, iParms, fParms);
        MovementHistory data = new MovementHistory();
        data.setBladeSetIndex(bladeIndex);
        data.setStartPosition(getPosition());
        long sTime = 1000 * runMove(nsamp <= 0, hall);
        data.setStartTime(sTime);
        data.setEndTime(1000 * System.currentTimeMillis());
        data.setEndPosition(getPosition());

        /*
        **  Disable Hall switch interrupts
        */
        acd.detachInt();

        /*
        **  Return if error or null move
        */
        if (nsamp <= 0) {
            data.setStatus(-nsamp);
            return data;
        }

        /*
        **  Get the sampled positions and times
        */
        List<BladePosition> pList = data.getMovementProfile();
        int status = getPositionData(sTime, nsamp, pList);
        if (status == 0) status = moveStatus();
        data.setStatus(status);
        
        float posn = pList.isEmpty() ? 0f : pList.get(0).getPosition();

        /*
        **  Locate the first calibration item to be used
        */
        boolean reverse = (dist < 0f);
        ArrayList<BladeSetCalibration.Item> cList
            = config.getCalibration().getData();
        int cIndex;
        for (cIndex = 0; cIndex < cList.size(); cIndex++) {
            BladeSetCalibration.Item cItem = cList.get(cIndex);
            if (cItem.isReverse() ^ reverse) continue;
            if (reverse && cItem.getPosition() < posn) break;
            if (!reverse && cItem.getPosition() > posn) break;
        }

        /*
        **  Create the returned Hall sensor data
        */
        List<HallTransition> hList = data.getHallTransitions();
        if (hall.count > hall.data.length) {
            if (status == 0)
                data.setStatus(MovementStatus.STS_EXCESS_HALL_TRAN);
            hall.count = hall.data.length;
        }
        for (int j = 0; j < hall.count; j++) {
            HallTransition hItem = new HallTransition();
            HallItem item = hall.data[j];
            posn = 0f;
            while (cIndex < cList.size()) {
                BladeSetCalibration.Item cItem = cList.get(cIndex++);
                if (cItem.getSensor() == item.sensor
                      && cItem.isOpen() == item.open) {
                    posn = cItem.getPosition();
                    break;
                }
            }
            hItem.setTransitionTime(item.time / 1000 + hall.startTime);
            hItem.setSensorId(item.sensor);
            hItem.setOpen(item.open);
            hItem.setReverse(dist < 0f);
            hItem.setPosition(posn);
            hItem.setRelPosition(config.relPosition(posn));
            hList.add(hItem);
        }

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Performs a move, sampling the position periodically
    **
    ***************************************************************************
    */
    public MovementHistory moveP(int type, float dist, float time,
                                 float mvFract, int nSamp)
    {

        /*
        **  Set the parameters used by the move program and run it
        */
        int[] iParms = {AcrComm.SYS_CLOCK_PRM, AcrComm.ENC_POSN_PRM,
                        AcrComm.PRIM_AXIS_PRM},
              fParms = {};
        int nsamp = initMove(type, dist, time, mvFract, nSamp, iParms, fParms);
        MovementHistory data = new MovementHistory();
        data.setBladeSetIndex(bladeIndex);
        data.setStartPosition(getPosition());
        long sTime = 1000 * runMove(nsamp <= 0, null);
        data.setStartTime(sTime);
        data.setEndTime(1000 * System.currentTimeMillis());
        data.setEndPosition(getPosition());

        /*
        **  Return if error or null move
        */
        if (nsamp <= 0) {
            data.setStatus(-nsamp);
            return data;
        }

        /*
        **  Get the sampled positions and times
        */
        int status = getPositionData(sTime, nsamp, data.getMovementProfile());
        if (status == 0) status = moveStatus();
        data.setStatus(status);

        return data;
    }


   /**
    ***************************************************************************
    **
    **  Homes the shutter
    **
    ***************************************************************************
    */
    public int home(int optns, float vel)
    {
        /*
        **  Configure and enable the drive
        */
        float fvel = vel / 10F;
        float acc = 10F * vel;
        float jrk = 0;
        acr.setFloatParm(AcrComm.PPU_PRM, config.ppu());
        acr.setFloatParm(AcrComm.JOG_VEL_PRM, vel);
        acr.setFloatParm(AcrComm.JOG_ACC_PRM, acc);
        acr.setFloatParm(AcrComm.JOG_DEC_PRM, acc);
        acr.setFloatParm(AcrComm.JOG_JRK_PRM, jrk);
        acr.sendStr("axis0 jog homvf " + fvel, 0);
        acr.setBit(AcrComm.HOME_BKUP_BIT);
        acr.clearBit(AcrComm.HOME_NEDG_BIT);
        if ((optns & HOM_OPT_PFINAL) != 0)
            acr.clearBit(AcrComm.HOME_NFNL_BIT);
        else
            acr.setBit(AcrComm.HOME_NFNL_BIT);
        enableDrive();

        /*
        **  Do the home and check outcome
        */
        int status = 0;
        acr.sendStr("axis0 jog home " + ((optns & HOM_OPT_NDIRN) != 0 ? -1 : 1),
                    0);
        while (true) {
            acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
            if (acr.getBit(AcrComm.HOME_FND_BIT) != 0) break;
            if (acr.getBit(AcrComm.HOME_FAIL_BIT) != 0) {
                status = -1;
                break;
            }
        }

        /*
        **  Disable the drive
        */
        disableDrive();

        return status;
    }


   /**
    ***************************************************************************
    **
    **  Returns the configuration object being used
    **
    ***************************************************************************
    */
    public BladeSetConfiguration getConfig()
    {
        return config;
    }


   /**
    ***************************************************************************
    **
    **  Returns the AccesDio object being used
    **
    ***************************************************************************
    */
    public AccesDio getAcces()
    {
        return acd;
    }


   /**
    ***************************************************************************
    **
    **  Returns the Helios object being used
    **
    ***************************************************************************
    */
    public Helios getHelios()
    {
        return hel;
    }


   /**
    ***************************************************************************
    **
    **  Returns the motor communications object being used
    **
    ***************************************************************************
    */
    public AcrComm getComm()
    {
        return acr;
    }


   /**
    ***************************************************************************
    **
    **  Gets the status of a move: normal; or premature stop reason
    **
    ***************************************************************************
    */
    public int moveStatus()
    {
        if (acr.getBit(AcrComm.KILL_MOVE_BIT) == 0)
            return 0;

        if (acr.getBit(AcrComm.POSN_ERR_BIT) != 0)
            return MovementStatus.STS_EXCS_POSN_ERROR;

        if (acr.getBit(AcrComm.POS_EOT_BIT) != 0)
            return MovementStatus.STS_POS_EOT_LIMIT;

        if (acr.getBit(AcrComm.NEG_EOT_BIT) != 0)
            return MovementStatus.STS_NEG_EOT_LIMIT;

        if (acr.getBit(AcrComm.POS_SLM_BIT) != 0)
            return MovementStatus.STS_POS_SOFT_LIMIT;

        if (acr.getBit(AcrComm.NEG_SLM_BIT) != 0)
            return MovementStatus.STS_NEG_SOFT_LIMIT;

        return MovementStatus.STS_UNKN_MOVE_TRUNC;
    }


   /**
    ***************************************************************************
    **
    **  Gets the drive temperature
    **
    ***************************************************************************
    */
    public float getDriveTemperature()
    {
        return acr.getFloatParm(AcrComm.DRIVE_TEMP_PRM);
    }


   /**
    ***************************************************************************
    **
    **  Gets the ambient temperature
    **
    ***************************************************************************
    */
    public float getAmbientTemperature()
    {
        return 100f * hel.adCountToVolts(hel.adSample());
    }


   /**
    ***************************************************************************
    **
    **  Performs Hall switch position calibration in one direction
    **
    ***************************************************************************
    */
    private void calibLeg(float dist, float step, float time,
                          BladeSetCalibration calib)
    {

        /*
        **  Configure and enable the drive
        */
        configMove(MOV_TYP_SCURVE, step, time, 0f);

        /*
        **  Generate command to step the shutter
        */
        step = Math.copySign(step, dist);
        String stepCmnd = "axis0 jog inc (" + config.motorDistance(step) + ")";

        /*
        **  Read the switches
        */
        int oldVal = acd.dioInp(dioHPort);

        /*
        **  Loop while stepping the shutter and reading the switches
        **  Quit if positive or negative travel limit is reached
        */
        int index = 0, oldSensor = -1, status = 0;
        boolean reverse = (dist < 0F);
        float position = getPosition(), endPosition = position + dist;
        ArrayList<BladeSetCalibration.Item> cList = calib.getData();

        while (reverse ^ ((position - endPosition) < 0f)) {
            acr.sendStr(stepCmnd, 0);
            acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
            position = getPosition();
            int value = acd.dioInp(dioHPort);
            int chan = oldVal ^ value;
            if (chan != 0) {
                oldVal = value;
                boolean open = (chan & oldVal) != 0;
                int sensor = Integer.numberOfTrailingZeros(chan);
                if (sensor != oldSensor) {
                    index = 0;
                    oldSensor = sensor;
                }
                else
                    if (!open) index++;
                cList.add(new BladeSetCalibration.Item(sensor, index, position,
                                                       open, reverse));
            }
            if ((status = moveStatus()) != 0) break;
        }

        /*
        **  Disable the drive
        */
        disableDrive();

        /*
        **  Set the status
        */
        if (calib.getStatus() == 0)
            calib.setStatus(status);
    }


   /**
    ***************************************************************************
    **
    **  Calibrates the Hall switch positions in one direction
    **
    ***************************************************************************
    */
    private void calibLegNew(float dist, float time, Hall hall,
                             BladeSetCalibration calib)
    {
        /*
        **  Clear the Hall count
        */
        hall.count = 0;

        /*
        **  Configure and enable the drive, and start the capture program
        */
        configMove(MOV_TYP_SCURVE, dist, time, 0F);
        acr.clearBit(AcrComm.CAL_SYNC_BIT);
        acr.sendStr("run prog" + CAL_PROG, 0);
        acr.sendStr("inh " + AcrComm.CAL_SYNC_BIT, 0);

        /*
        **  Move the shutter and wait for completion
        */
        acr.sendStr("axis0 jog inc " + config.motorDistance(dist), 0);
        acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
        int status = moveStatus();

        /*
        **  Disable the drive, halt the program and stop the capture
        */
        disableDrive();
        acr.sendStr("halt prog" + CAL_PROG, 0);
        acr.sendStr("intcap axis0 off", 0);

        /*
        **  Get the captured encoder positions and check counts
        */
        int[] count = new int[1];
        acr.getVars(CAL_PROG, 0, 1, count);
        int[] posns = new int[count[0]];
        int nsamp = acr.getVars(CAL_PROG, 0, 0, count[0], posns);
        if (nsamp != count[0]) {
            calib.setStatus(MovementStatus.STS_MSSG_PGM_VBLES);
            return;
        }
        if (hall.count > hall.data.length) {
            if (status == 0)
                status = MovementStatus.STS_EXCESS_HALL_TRAN;
            hall.count = hall.data.length;
        }
        if (nsamp != hall.count) {
            if (status == 0)
                status = MovementStatus.STS_UNEQL_HALL_TRAN;
            if (nsamp < hall.count)
                hall.count = nsamp;
        }

        /*
        **  Add the encoder positions and sensor data to the calibration list
        */
        int index = 0, oldSensor = -1;
        boolean reverse = (dist < 0f);
        ArrayList<BladeSetCalibration.Item> cList = calib.getData();
        for (int j = 0; j < hall.count; j++) {
            HallItem item = hall.data[j];
            if (item.sensor != oldSensor) {
                index = 0;
                oldSensor = item.sensor;
            }
            else
                if (!item.open) index++;
            cList.add(new BladeSetCalibration.Item(item.sensor, index,
                                                   config.position(posns[j]),
                                                   item.open, reverse));
        }

        /*
        **  Set the status
        */
        if (calib.getStatus() == 0)
            calib.setStatus(status);
    }


   /**
    ***************************************************************************
    **
    **  Adds sampled position data to a list
    **
    ***************************************************************************
    */
    private int getPositionData(long sTime, int nsamp,
                                List<BladePosition> pList)
    {

        /*
        **  Retrieve the sampled positions and times
        */
        int[] times = new int[nsamp], posns = new int[nsamp],
              flags = new int[nsamp], time0 = new int[1];
        int tsamp = acr.getVars(MOVE_PROG, 0, 1, time0);
        tsamp += acr.getVars(MOVE_PROG, 0, 0, nsamp, times);
        tsamp += acr.getVars(MOVE_PROG, 1, 0, nsamp, posns);
        tsamp += acr.getVars(MOVE_PROG, 2, 0, nsamp, flags);
        if (tsamp != 3 * nsamp + 1)
            return MovementStatus.STS_MSSG_PGM_VBLES;

        /*
        **  Create the returned sampled values
        */
        for (int j = 0; j < nsamp; j++) {
            BladePosition pItem = new BladePosition();
            pItem.setTime(sTime + 1000 * (times[j] - time0[0]));
            float posn = config.position(posns[j]);
            pItem.setPosition(posn);
            pItem.setRelPosition(config.relPosition(posn));
            pList.add(pItem);
            if ((flags[j] & 0x01000000) == 0) break;
        }

        return 0;
    }


   /**
    ***************************************************************************
    **
    **  Reads the Hall switches while calibrating, and record any changes
    **
    **  <p>
    **  This routine is called whenever a Hall switch changes
    **
    ***************************************************************************
    */
    private void readHallC(int value, Object parm)
    {
        Hall hall = (Hall)parm;
        int state = acd.dioInp(dioHPort);
        int sensor = state ^ hall.state;
        if (sensor == 0) return;

        hall.state = state;
        if (hall.count < hall.data.length) {
            HallItem item = hall.data[hall.count];
            boolean open = (sensor & state) != 0;
            if (open) acd.dioSetBit(dioOPort, dioOLine);
            else acd.dioClrBit(dioOPort, dioOLine);
            item.time = System.nanoTime();
            item.sensor = Integer.numberOfTrailingZeros(sensor);
            item.open = open;
        }
        hall.count++;
    }


   /**
    ***************************************************************************
    **
    **  Reads the Hall switches while moving, and record any changes
    **
    **  <p>
    **  This routine is called whenever a Hall switch changes
    **
    ***************************************************************************
    */
    private void readHallM(int value, Object parm)
    {
        Hall hall = (Hall)parm;
        int state = acd.dioInp(dioHPort);
        int sensor = state ^ hall.state;
        if (sensor == 0) return;

        hall.state = state;
        if (hall.count < hall.data.length) {
            HallItem item = hall.data[hall.count];
            item.time = System.nanoTime();
            item.sensor = Integer.numberOfTrailingZeros(sensor);
            item.open = (sensor & state) != 0;
        }
        hall.count++;
    }


   /**
    ***************************************************************************
    **
    **  Configures the drive for a move
    **
    ***************************************************************************
    */
    private void configMove(int type, float dist, float time, float mvFract)
    {
        float vel, acc, jrk = 0F;
        if (mvFract < 0F) mvFract = 0F;
        else if (mvFract > 0.9F) mvFract = 0.9F;
        vel = Math.abs(2F * dist / ((1F + mvFract) * time));
        acc = Math.abs(4F * dist / ((1F - mvFract * mvFract) * time * time));
        if (type == MOV_TYP_SCURVE) {
            acc *= 2F;
            jrk = acc * acc / vel;
        }
        acr.setFloatParm(AcrComm.PPU_PRM, config.ppu());
        acr.setFloatParm(AcrComm.JOG_VEL_PRM, vel);
        acr.setFloatParm(AcrComm.JOG_ACC_PRM, acc);
        acr.setFloatParm(AcrComm.JOG_DEC_PRM, acc);
        acr.setFloatParm(AcrComm.JOG_JRK_PRM, jrk);
        enableDrive();
    }


   /**
    ***************************************************************************
    **
    **  Sets parameters for the move program and start it
    **
    ***************************************************************************
    */
    private int initMove(int type, float dist, float time, float mvFract,
                         int nSamp, int[] iParms, int[] fParms)
    {
        double[] parms = new double[P_PARMS + iParms.length + fParms.length];
        float vel, acc, jrk = 0F;

        /*
        **  Return if distance is zero
        */
        if (dist == 0f) return 0;

        /*
        **  Set up the parameter values used by the move program
        */
        parms[P_DRVON] = driveOn ? 1.0 : 0.0;
        parms[P_DIST] = config.motorDistance(dist);
        if (mvFract < 0) mvFract = 0;
        else if (mvFract > 0.9F) mvFract = 0.9F;
        vel = Math.abs(2 * dist / ((1 + mvFract) * time));
        acc = Math.abs(4 * dist / ((1 - mvFract * mvFract) * time * time));
        if (type == MOV_TYP_SCURVE) {
            acc *= 2;
            jrk = acc * acc / vel;
        }
        if (nSamp > 0) {
            parms[P_SMPTM] = Math.floor(1000 * time / nSamp);
            parms[P_SMPCT] = Math.floor(1000 * time / parms[P_SMPTM] + 5.0);
        }
        else {
            parms[P_SMPTM] = 1.0;
            parms[P_SMPCT] = 1.0;
        }
        parms[P_NIPRM] = iParms.length;
        parms[P_NFPRM] = fParms.length;
        for (int j = 0; j < iParms.length; j++)
            parms[P_PARMS + j] = iParms[j];
        for (int j = 0; j < fParms.length; j++)
            parms[P_PARMS + iParms.length + j] = fParms[j];

        /*
        **  Check that there's enough space for samples
        */
        int nWord = 13 + ((int)parms[P_SMPCT] + 2)
                            * (iParms.length + fParms.length);
        if (nWord > 1024)
            return -MovementStatus.STS_INSUFF_VBLE_MEM;

        /*
        **  Copy the parameter values to the controller
        */
        acr.setFloatParm(AcrComm.PPU_PRM, config.ppu());
        acr.setFloatParm(AcrComm.JOG_VEL_PRM, vel);
        acr.setFloatParm(AcrComm.JOG_ACC_PRM, acc);
        acr.setFloatParm(AcrComm.JOG_DEC_PRM, acc);
        acr.setFloatParm(AcrComm.JOG_JRK_PRM, jrk);
        int count = acr.setUserParms(0, parms, parms.length);
        if (count < parms.length)
            return -MovementStatus.STS_INSUFF_USER_PARM;

        /*
        **  Clear the sync bit, run the move program and wait for sync bit
        */
        acr.clearBit(AcrComm.MOVE_SYNC_BIT);
        acr.sendStr("run prog" + MOVE_PROG, 0);
        acr.sendStr("inh " + AcrComm.MOVE_SYNC_BIT, 0);
        sleep(0.1F);

        return (int)parms[P_SMPCT];
    }


   /**
    ***************************************************************************
    **
    **  Carries out the move with parameter sampling
    **
    ***************************************************************************
    */
    private long runMove(boolean dummy, Hall hall)
    {
        /*
        **  Get the start times, standard and high-resolution
        */
        long sTime = System.currentTimeMillis();
        if (hall != null)
            hall.startTime = 1000 * sTime - System.nanoTime() / 1000;
        if (dummy) return sTime;

        /*
        **  Start the move by raising the input line
        */
        acd.dioSetBit(dioOPort, dioOLine);

        /*
        **  Wait for the move and sampling to complete
        */
        acr.sendStr("inh " + AcrComm.JOG_ACTV_BIT, 0);
        acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
        acr.sendStr("inh -" + AcrComm.SAMP_ARM_BIT, 0);

        /*
        **  Lower the input line
        */
        acd.dioClrBit(dioOPort, dioOLine);

        return sTime;
    }


   /**
    ***************************************************************************
    **
    **  Initializes the motor controller
    **
    ***************************************************************************
    */
    private void acrInitialize()
    {
        /*
        **  Do basic initialization
        */
        acr.sendStr("prog" + MOVE_PROG, 0);
        acr.sendStr("attach axis0 enc0 dac0 enc0", 0);
        acr.sendStr("attach master0", 0);
        acr.sendStr("attach slave0 axis0 \"x\"", 0);
        acr.sendStr("sys", 0);
        acr.sendStr("halt all", 0);
        acr.sendStr("new all", 0);
        acr.sendStr("clear", 0);
        acr.sendStr("dim def(10)", 0);
        acr.sendStr("dim p(256)", 0);
        acr.sendStr("dim prog" + MOVE_PROG + "(32768)", 0);
        acr.sendStr("dim prog" + CAL_PROG + "(8192)", 0);
        acr.sendStr("dim prog" + TEST_PROG + "(8192)", 0);
        acr.sendStr("dim prog" + TEMP_PROG + "(8192)", 0);
        acr.sendStr("axis0 drive off", 0);
        acr.setFloatParm(AcrComm.PPU_PRM, config.ppu());
        acr.sendStr("axis0 hldec 50000", 0);
        acr.sendStr("axis0 slm (" + config.posSoftLimit() + ","
                      + config.negSoftLimit() + ")", 0);
        acr.sendStr("axis0 sldec 50000", 0);
        acr.setBit(AcrComm.ENAB_PEOT_BIT);
        acr.setBit(AcrComm.ENAB_NEOT_BIT);
        acr.setBit(AcrComm.ENAB_PSLM_BIT);
        acr.setBit(AcrComm.ENAB_NSLM_BIT);
        acr.clearBit(AcrComm.INVT_HOME_BIT);
        acr.clearBit(AcrComm.INVT_PEOT_BIT);
        acr.clearBit(AcrComm.INVT_NEOT_BIT);
        acr.clearBit(AcrComm.KILL_MOVE_BIT);
        acr.setFloatParm(AcrComm.PGAIN_PRM, 0.04f);
        acr.setFloatParm(AcrComm.IGAIN_PRM, 0.0003f);
        acr.setFloatParm(AcrComm.DGAIN_PRM, 0.001f);
        acr.sendStr("axis0 on", 0);
        acr.setFloatParm(AcrComm.POS_TLM_PRM, TORQUE_LIM);
        acr.setFloatParm(AcrComm.NEG_TLM_PRM, -TORQUE_LIM);

        /*
        **  Load the move program
        */
        acr.sendStr("prog" + MOVE_PROG, 0);
        acr.sendStr("program", 0);
        acr.sendStr("clear", 0);
        acr.sendStr("samp clear", 0);
        acr.sendStr("dim lv(10)", 0);
        acr.sendStr("dim la(p" + P_NIPRM + ")", 0);
        acr.sendStr("dim sa(p" + P_NFPRM + ")", 0);
        acr.sendStr("for lv0 = 0 to (p" + P_NIPRM + " - 1) step 1", 0);
        acr.sendStr("dim la(lv0)(p" + P_SMPCT + ")", 0);
        acr.sendStr("samp(lv0) base la(lv0)", 0);
        acr.sendStr("samp(lv0) src p(p(lv0 + " + P_PARMS + "))", 0);
        acr.sendStr("next", 0);
        acr.sendStr("for lv0 = 0 to (p" + P_NFPRM + " - 1) step 1", 0);
        acr.sendStr("dim sa(lv0)(p" + P_SMPCT + ")", 0);
        acr.sendStr("samp(lv0 + p" + P_NIPRM + ") base sa(lv0)", 0);
        acr.sendStr("samp(lv0 + p" + P_NIPRM + ") src p(p(lv0 + p" + P_NIPRM
                      + " + " + P_PARMS + "))", 0);
        acr.sendStr("next", 0);
        acr.sendStr("clr " + AcrComm.SAMP_MODE_BIT, 0);
        acr.sendStr("samp trg " + AcrComm.JOG_ACTV_BIT, 0);
        acr.sendStr("p" + AcrComm.SAMP_PERD_PRM + " = p" + P_SMPTM, 0);
        acr.sendStr("if (p" + P_DRVON + " = 0) then axis0 drive on", 0);
        acr.sendStr("dwl 0.1", 0);
        acr.sendStr("set " + AcrComm.MOVE_SYNC_BIT, 0);
//        acr.sendStr("inh " + AcrComm.INPUT4_BIT, 0);
        acr.sendStr("intcap axis0 " + AcrComm.ICM_RISE_3RD, 0);
        acr.sendStr("inh " + AcrComm.CAP_CMPL_BIT, 0);
        acr.sendStr("lv0 = p" + AcrComm.SYS_CLOCK_PRM, 0);
        acr.sendStr("lv1 = p" + AcrComm.ENC_POSN_PRM, 0);
        acr.sendStr("set " + AcrComm.SAMP_ARM_BIT, 0);
        acr.sendStr("axis0 jog inc (p" + P_DIST + ")", 0);
        acr.sendStr("inh -" + AcrComm.JOG_ACTV_BIT, 0);
        acr.sendStr("lv2 = p" + AcrComm.ENC_POSN_PRM, 0);
        acr.sendStr("inh -" + AcrComm.SAMP_ARM_BIT, 0);
        acr.sendStr("if (p" + P_DRVON + " = 0) then axis0 drive off", 0);
        acr.sendStr("endp", 0);

        /*
        **  Reset the command input to be system-wide
        */
        acr.sendStr("sys", 0);
    }


   /**
    ***************************************************************************
    **
    **  Sleeps for a time
    **
    ***************************************************************************
    */
    private void sleep(float secs)
    {
        long nsecs = (long)(1e9F * secs);
        try {
            Thread.sleep(nsecs / 1000000, (int)(nsecs % 1000000));
        }
        catch (java.lang.InterruptedException e) {
        }
    }

}
