package org.lsst.ccs.subsystems.shutter.driver;

import java.io.PrintStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.Scanner;
import java.util.ArrayList;
import java.util.NoSuchElementException;

/**
 ***************************************************************************
 **
 **  Class containing blade configuration data
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class BladeSetConfiguration implements Serializable {

    private final static float POS_SOFT_LIM = 1.05f,
                               NEG_SOFT_LIM = -0.05f;

    int     index;      // Index of the BladeSet
    String  node;       // Node name of motor controller
    float   ppMm;       // Motor encoder pulses per mm of travel
    float   homePosn;   // Blade home position (mm)
    float   openPosn;   // Blade open position (mm)
    float   clsdPosn;   // Blade closed position (mm)
    int     numSamp;    // Number of encoder samples to take
    float   moveTime;   // Duration of a BladeSet move (sec)
    int     dioAddr;    // DIO board I/O space address
    int     dioLevel;   // DIO board IRQ level
    int     dioConf;    // DIO board ports configuration value
    int     dioHPort;   // DIO board port for Hall sensor input
    int     dioOPort;   // DIO board port for output to motor
    int     dioOLine;   // DIO board line for output to motor
    int     dioIPort;   // DIO board port for input from motor
    int     dioILine;   // DIO board line for input from motor
    int     adAddr;     // AD board I/O space address
    int     adLevel;    // AD board IRQ level
    int     adTempChn;  // AD board channel for ambient temperature

    BladeSetCalibration calib = new BladeSetCalibration();
                        // Hall switch calibration data

    private static final String[] CALIB_FILE = {"Calibration_0.dat",
                                                "Calibration_1.dat"};

    private static final PrintStream out = System.out;

    public void setIndex(int index)
    {
        this.index = index;
        if (index == 0) {
            node       = "192.168.100.1";
            ppMm       = -49.3504f;
            homePosn   = 0.0f;
            openPosn   = 0.0f;
            clsdPosn   = 760.0f;
            moveTime   = 1.0f;
            numSamp    = 100;
            dioAddr    = 0x340;
            dioLevel   = 6;
            dioConf    = 0x07;
            dioHPort   = 0;
            dioIPort   = 2;
            dioILine   = 0;
            dioOPort   = 2;
            dioOLine   = 4;
            adAddr     = 0x280;
            adLevel    = 5;
            adTempChn  = 0;
        }
        else {
            node       = "192.168.100.1";
            ppMm       = 49.3504f;
            homePosn   = 760.0f;
            openPosn   = 760.0f;
            clsdPosn   = 0.0f;
            moveTime   = 1.0f;
            numSamp    = 100;
            dioAddr    = 0x340;
            dioLevel   = 6;
            dioConf    = 0x07;
            dioHPort   = 0;
            dioIPort   = 2;
            dioILine   = 0;
            dioOPort   = 2;
            dioOLine   = 4;
            adAddr     = 0x280;
            adLevel    = 5;
            adTempChn  = 0;
        }
        readCalibration();
    }


    public int getIndex()
    {
        return index;
    }


    public void setCalibration(BladeSetCalibration calib)
    {
        this.calib = calib;
    }


    public BladeSetCalibration getCalibration()
    {
        return calib;
    }


    public String getNode()
    {
        return node;
    }


    public float getPpMm()
    {
        return ppMm;
    }


    public float getHome()
    {
        return homePosn;
    }


    public float getOpen()
    {
        return openPosn;
    }


    public float getClosed()
    {
        return clsdPosn;
    }


    public float getMoveTime()
    {
        return moveTime;
    }


    public int getNumSamp()
    {
        return numSamp;
    }


    public int getDioAddr()
    {
        return dioAddr;
    }


    public int getDioLevel()
    {
        return dioLevel;
    }


    public int getDioConf()
    {
        return dioConf;
    }


    public int getDioHPort()
    {
        return dioHPort;
    }


    public int getDioIPort()
    {
        return dioIPort;
    }


    public int getDioILine()
    {
        return dioILine;
    }


    public int getDioOPort()
    {
        return dioOPort;
    }


    public int getDioOLine()
    {
        return dioOLine;
    }


    public int getAdAddr()
    {
        return adAddr;
    }


    public int getAdLevel()
    {
        return adLevel;
    }


    public int getAdTempChn()
    {
        return adTempChn;
    }


    public float ppu()
    {
        return Math.abs(ppMm);
    }


    public float motorDistance(float dist)
    {
        return Math.signum(ppMm) * dist;
    }


    public float position(int encoder)
    {
        return homePosn + encoder / ppMm;
    }


    public float position(float relPosition)
    {
        return openPosn + relPosition * (clsdPosn - openPosn);
    }


    public float relPosition(int encoder)
    {
        return (encoder / ppMm + homePosn - openPosn) / (clsdPosn - openPosn);
    }


    public float relPosition(float position)
    {
        return (position - openPosn) / (clsdPosn - openPosn);
    }


    public float posSoftLimit()
    {
        float reach = Math.signum(ppMm) * (clsdPosn - openPosn);
        return ((reach > 0.0f) ? POS_SOFT_LIM : NEG_SOFT_LIM) * reach;
    }


    public float negSoftLimit()
    {
        float reach = Math.signum(ppMm) * (clsdPosn - openPosn);
        return ((reach > 0.0f) ? NEG_SOFT_LIM : POS_SOFT_LIM) * reach;
    }


   /**
    ***************************************************************************
    **
    **  Read a file of Hall sensor calibration data (transition positions)
    **
    ***************************************************************************
    */
    public int readCalibration()
    {
        return readCalibration(CALIB_FILE[index]);
    }

    public int readCalibration(String file)
    {

        /*
        **  Open the file
        */
        FileInputStream ifl = null;
        try {
            ifl = new FileInputStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return -1;
        }

        /*
        **  Parse the data and add it to the list
        */
        calib.setStatus(0);
        ArrayList<BladeSetCalibration.Item> list = calib.getData();
        list.clear();
        Scanner inp = new Scanner(ifl);
        int line = 0, field = 0, sensor = 0, indx = 0;
        float posn = 0f;
        boolean open = false, reverse = false;
        try {
            while (inp.hasNext()) {
                switch (field) {
                case 0:
                    reverse = inp.next("[+-]").equals("-");
                    break;
                case 1:
                    sensor = inp.nextInt();
                    break;
                case 2:
                    indx = inp.nextInt();
                    break;
                case 3:
                    open = inp.next("[OC]").equals("O");
                    break;
                case 4:
                    posn = inp.nextFloat();
                    list.add(new BladeSetCalibration.Item(sensor, indx, posn,
                                                          open, reverse));
                    field = -1;
                    line++;
                    inp.nextLine();
                }
                field++;
            }
        }
        catch (NoSuchElementException e) {
            out.format("Invalid field (%s) in line %s\n", field + 1, line + 1);
            list.clear();
            field = 0;
            line = 0;
        }

        if (field != 0) {
            out.format("File ended unexpectedly in line %s\n", line + 1);
            list.clear();
            line = 0;
        }

        /*
        **  Close the file
        */
        try {
            ifl.close();
        }
        catch (IOException e) {
        }

        return line;
    }


   /**
    ***************************************************************************
    **
    **  Write a file of Hall sensor calibration data (transition positions)
    **
    ***************************************************************************
    */
    public void writeCalibration()
    {
        writeCalibration(calib, CALIB_FILE[index]);
    }

    public void writeCalibration(BladeSetCalibration calib)
    {
        writeCalibration(calib, CALIB_FILE[index]);
    }

    public void writeCalibration(String file)
    {
        writeCalibration(calib, file);
    }

    public static void writeCalibration(BladeSetCalibration calib, String file)
    {
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Write the data to the output file
        */
        ArrayList<BladeSetCalibration.Item> list = calib.getData();
        for (int j = 0; j < list.size(); j++) {
            BladeSetCalibration.Item tran = list.get(j);
            ofl.format("%s %s %s %s %8.2f\n", tran.isReverse() ? "-" : "+",
                       tran.getSensor(), tran.getIndex(),
                       tran.isOpen() ? "O" : "C", tran.getPosition());
        }

        /*
        **  Close the output file
        */
        ofl.close();
    }

}
