package org.lsst.ccs.subsystem.shutter.gui;

import java.util.Arrays;
import java.util.logging.Logger;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusAggregator;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.MessagingService;
import org.lsst.ccs.subsystem.shutter.common.Constants;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.gui.ShutterPlugin.ShutterPage;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;

/**
 * Detects the presence or absence of the worker subsystem on the CCS status bus
 * and listens for any status messages from it. Not thread-safe, use only from the dispatcher task.
 * @author tether
 */
public class StatusListener implements PageActions, AgentStatusListener, StatusMessageListener  {
    private static final Logger LOG = Logger.getLogger(StatusListener.class.getName());

    private final ShutterPage page;
    
    private String workerName;

    public StatusListener(final ShutterPage page) {
        this.page = page;
        this.workerName = null;
    }

    ////////// Implementation of PageActions //////////

    @Override
    public void disconnect() {
        LOG.fine("disconnect()");
        final AgentStatusAggregator agg = page.getConsole().getStatusAggregator();
        agg.removeListener(this);
        final AgentMessagingLayer aml =
            page.getConsole().getAgentService(MessagingService.class).getMessagingAccess();
        aml.removeStatusMessageListener(this);
    }

    @Override
    public void connect() {
        LOG.fine("connect()");
        if (workerName == null) {
            throw new NullPointerException("Worker subsystem name has not been set.");
        }
        final AgentStatusAggregator agg = page.getConsole().getStatusAggregator();
        agg.addListener(this, Arrays.asList(workerName), null);
        final AgentMessagingLayer aml =
            page.getConsole().getAgentService(MessagingService.class).getMessagingAccess();
        aml.addStatusMessageListener(
            this,
            msg -> msg.getOriginAgentInfo().getName().equals(workerName));
    }
    
    @Override
    public void setWorkerName(final String workerName) {
        LOG.fine("setWorkerName()");
        this.workerName = workerName;
    }

    ////////// Implementation of AgentStatusListener //////////

    @Override
    public void disconnect(AgentStatusEvent event) {
        LOG.fine("Worker is offline.");
        page.getDispatcher().showWorkerIsUnreachable("OFFLINE");
    }

    @Override
    public void connect(AgentStatusEvent event) {
        LOG.fine("Worker is online.");
        page.getDispatcher().showWorkerIsReachable();
    }

    ////////// Implementation of StatusMessageListener //////////

    @Override
    public void onStatusMessage(final StatusMessage msg) {
        LOG.fine(msg.toString());
        final PhysicalState pstate =
            (PhysicalState)
            msg
            .getState()
            .getComponentState(Constants.PHYSICAL_STATE_PATH, PhysicalState.class);
        final SoftwareState sstate =
            (SoftwareState)
            msg
            .getState()
            .getComponentState(Constants.SOFTWARE_STATE_PATH, SoftwareState.class);
        page.getDispatcher().showStateBundle(pstate, sstate);
        if (msg instanceof StatusSubsystemData) {
            final StatusSubsystemData sd = (StatusSubsystemData)msg;
            final String key = sd.getDataKey();
            if (key.equals(ShutterStatus.class.getSimpleName())) {
                final ShutterStatus status =
                    (ShutterStatus)
                    sd.getSubsystemData()
                    .getValue();
                page.getDispatcher().showStatus(status);
            }
            else if (key.equals(MotionDone.class.getSimpleName())) {
                final MotionDone status =
                    (MotionDone)
                    sd.getSubsystemData()
                    .getValue();
                page.getDispatcher().showTrajectory(status);
            }
        }

    }

}
