package org.lsst.ccs.subsystem.shutter.gui;


import data.DiscretePlotData;
import data.MetaData;
import data.MutablePlotData;
import data.PlotDataListener;
import data.SuggestedRange;
import java.util.ArrayList;
import java.util.List;
import org.freehep.util.Value;
import util.ListenerList;

import org.lsst.ccs.subsystem.shutter.sim.PredictedPosition;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;


/**
 * Contains the data set for a predicted trajectory of a blade set. This is a mutable data set
 * whose contents may be removed or added to. Those objects needing to use this data set
 * must register with the data set as plot-data listeners in order to be informed when
 * the contents of the data set have changed.
 * <p>
 * Each data point has four dimensions: time, acceleration, velocity and position. The times
 * are relative to the start of motion.
 * <p>
 * This class isn't thread safe and methods should be called only from the event dispatch
 * thread, for example via {@code invokeLater}.
 * @see PlotDataListener
 * @author tonyj
 * @author tether
 */
public class PredictionPlotData implements DiscretePlotData, MutablePlotData {

    public static final int TIME = 0;
    public static final int ACCELERATION = 1;
    public static final int VELOCITY = 2;
    public static final int POSITION = 3;

    private final List<PredictionPoint> data = new ArrayList<>();
    private final ListenerList<PlotDataListener> listeners = new ListenerList<>();
    private final String[] names = {"Time", "Acceleration", "Velocity", "Position"};

    public PredictionPlotData() {}

    /**
     * Clears away any old data and cause the plot pane to be repainted.
     */
    public void reset() {
       data.clear();
       firePlotDataChanged();
    }

    /**
     * Adds new data and causes the plot pane to be repainted. All the data
     * for the plot must be added at once or the scaling  of acceleration
     * and velocity won't work properly.
     * <p>
     * Acceleration values will be scaled by dividing by the max absolute value.
     * Likewise for the velocity values. Position values are assumed to be
     * already scaled so that a full stroke has size 1.0.
     * @param pos a stream of PredictedPosition from simulated motion
     */
    public void addData(final List<PredictedPosition> pos, final CCSTimeStamp t0) {
        final double vscale =
            pos.stream()
                .mapToDouble(p->Math.abs(p.getVelocity()))
                .max()
                .orElse(1.0);
        final double ascale =
            pos.stream()
                .mapToDouble(p->Math.abs(p.getAcceleration()))
                .max()
                .orElse(1.0);
        for (final PredictedPosition p: pos) {
            data.add(
                new PredictionPoint(
                    p.timeDiff(t0).toMillis() * 1e-3,
                    p.getAcceleration() / ascale,
                    p.getVelocity() / vscale,
                    p.getPosition()));                
        }
        firePlotDataChanged();
    }

    /**
     * Gets the count of data points currently stored.
     * @return The number of points.
     */
    @Override
    public int getNPoints() {
        return data.size();
    }

    /**
     * Gets the number of dimensions, that is, the number of values per data point.
     * @return The dimension count.
     */
    @Override
    public int getNDimensions() {
        return names.length;
    }

    /**
     * Gets the name of one of the dimensions of a data point.
     * @param index The dimension index {@code d}, where {@code 0 <= d < getNDimensions()}.
     * @return The name of the requested dimension.
     * @throws IndexOutOfBoundsException if {@code index} is out of bounds.
     */
    @Override
    public String names(int index) {
        return names[index];
    }

    /**
     * Returns the data type (class object) for the values of a given dimension.
     * @param index The dimension index, ignored by this method.
     * @return {@code Double.TYPE}, always.
     */
    @Override
    public Class types(int index) {
        return Double.TYPE;
    }

    /**
     * Gets the value of a given data point at a given dimension.
     * @param value A generic container for values of primitive type. In this case
     * it will always be given a double value, so use {@code value.getDouble()}.
     * @param dim The dimension index {@code d}, where {@code 0 <= d < getNDimensions()}.
     * @param index The data point index {@code i}, where {@code 0 <= i < getNPoints(}.
     * @throws IndexOutOfBoundsException if either {@code dim} or {@code index}
     * is out of bounds.
     */
    @Override
    public void getValue(Value value, int dim, int index) {
        final PredictionPoint point = data.get(index);
        switch (dim) {
            case TIME: value.set(point.getTime()); break;
            case ACCELERATION: value.set(point.getAcceleration()); break;
            case VELOCITY: value.set(point.getVelocity()); break;
            case POSITION: value.set(point.getPosition()); break;
            default: throw new IndexOutOfBoundsException("dim is out of bounds.");
        }
    }

    /**
     * Gets the metadata for this plot data object.
     * @return null since there is no metadata.
     */
    @Override
    public MetaData getMetaData() {
        return null;
    }

    @Override
    public String getTitle() {
        return "Predicted trajectory";
    }

    @Override
    public SuggestedRange getSuggestedRange() {
        return null;
    }

    @Override
    public void addPlotDataListener(PlotDataListener listener) {
        listeners.addListener(listener);
    }

    @Override
    public void removePlotDataListener(PlotDataListener listener) {
        listeners.removeListener(listener);
    }

    private void firePlotDataChanged() {
        if (!listeners.isEmpty()) {
            for (PlotDataListener l : listeners.getListeners()) {
                l.dataChanged();
            }
        }
    }

    /**
     * Locking isn't implemented.
     * @return nothing.
     * @throws UnsupportedOperationException
     */
    @Override
    public Object lock() {
        throw new UnsupportedOperationException("Locking isn't implemented.");
    }

    private static class PredictionPoint {

        private final double time;
        private final double acceleration;
        private final double velocity;
        private final double position;

        /**
         * Construct the plot data point.
         * @param date More or less absolute time in seconds since some epoch.
         * @param acceleration The acceleration scaled to fall between -1.0 and 1.0.
         * @param velocity The velocity scaled to fall between -1.0 and 1.0.
         * @param position The position scaled to fall between 0.0, fully retracted,
         * and 1.0, fully extended.
         */
        public PredictionPoint(double time, double acceleration, double velocity, double position) {
            this.time = time;
            this.acceleration = acceleration;
            this.velocity = velocity;
            this.position = position;
        }

        /**
         * Gets the scaled acceleration.
         * @return A value between -1 and 1.
         */
        public double getAcceleration() {
            return acceleration;
        }

        /**
         * Gets the time.
         * @return Time in seconds.
         */
        public double getTime() {
            return time;
        }

        /**
         * Gets the position.
         * @return A value between 0 and 1.
         */
        public double getPosition() {
            return position;
        }

        /**
         * Gets the velocity.
         * @return A value between -1 and 1.
         */
        public double getVelocity() {
            return velocity;
        }
    }

}
