package org.lsst.ccs.subsystem.shutter.gui;

import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.gconsole.agent.command.CommandTask;
import org.lsst.ccs.gconsole.services.command.CommandService;
import org.lsst.ccs.messaging.CommandRejectedException;
import org.lsst.ccs.subsystem.shutter.common.Axis;
import org.lsst.ccs.subsystem.shutter.common.Constants;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;

/**
 * Sends commands out to the worker subsystem and receives the acks and nacks.
 * @author tether
 */
public class Commander implements PluginActions {
    private static final Logger LOG = Logger.getLogger(Commander.class.getName());

    private final ShutterPlugin plugin;
    private final CommandService command;

    public Commander(final ShutterPlugin plugin) {
        this.plugin = plugin;
        this.command = new CommandService();
    }

    ////////// Implementation of PluginActions //////////

    @Override
    public void takeExposure(double exposureTime) {
        LOG.fine(() -> "takeExposure(" + exposureTime + ")");
        issueCommand("takeExposure", exposureTime);
    }

    @Override
    public void openShutter() {
        LOG.fine("openShutter()");
        issueCommand("openShutter");
    }

    @Override
    public void closeShutter() {
        LOG.fine("closeShutter()");
        issueCommand("closeShutter");
    }
    
    @Override
    public void changeBrakeState(final ShutterSide side, final boolean brakeEngaged) {
        LOG.fine(() -> String.format("changeBrakeState(%s, %s)", side, brakeEngaged));
        final Axis axis = side == ShutterSide.PLUSX ? Axis.getPlusXSide() : Axis.getMinusXSide();
        issueCommand("changeBrakeState", axis.getName(), brakeEngaged);
    }

    @Override
    public void calibrate() {
        LOG.fine("calibrate()");
        issueCommand("calibrate");
    }

    @Override
    public void center() {
        LOG.fine("center()");
        issueCommand("gotoCenter");
    }

    @Override
    public void prodMode() {
        LOG.fine("prodMode()");
        issueCommand("switchToNormalMode");
    }

    @Override
    public void resync() {
        LOG.fine("resync()");
        issueCommand("resync");
    }

    @Override
    public void stopMotion() {
        LOG.fine("stopMotion()");
        issueCommand("stopAll");
    }

    private void issueCommand(final String bareCmd, Object... args) {
        try {
            final String cmd = String.format("%s/%s", Constants.WORKER_SUBSYSTEM_NAME, bareCmd);
            final CommandTask task = command.execute(cmd, args);
            task.get();
        }
        catch (CancellationException exc) {
            plugin.getDispatcher().showCommandFailure("Commmand was canceled.");
        }
        catch (IllegalStateException exc) {
            final String msg =
                String.format("Subsystem %s is offline.", Constants.WORKER_SUBSYSTEM_NAME);
            LOG.warning(msg);
            plugin.getDispatcher().showWorkerIsUnreachable(msg);
        }
        catch (InterruptedException exc) {
            LOG.log(Level.WARNING, "Interrupted while awaiting a command response.", exc);
            plugin.getDispatcher().showCommandFailure("Interupted while awaitng a command response.");
        }
        catch (CommandRejectedException exc) {
            plugin.getDispatcher().showCommandFailure(exc.getMessage());
        }
        catch (TimeoutException exc) {
            plugin.getDispatcher().showCommandFailure("Command timed out.");
        }
        catch (ExecutionException exc) {
            plugin.getDispatcher().showCommandFailure("Command failed inside worker.");
        }
    }


}
