package org.lsst.ccs.subsystem.shutter.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.util.Arrays;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import plotter.Axis;
import plotter.Axis.AxisPosition;
import plotter.ContinuousAxisModel;
import plotter.DataArea;
import plotter.DataPointChartOverlay;
import plotter.LineChartOverlay;
import plotter.LinearAxisModel;
import plotter.PlotLayout;

/**
 * Displays plots of the predicted and actual motions of the shutter blade sets. The horizontal axis
 * is elapsed time in seconds. The vertical axis shows various quantities normalized to
 * the interval [-1, 1]. The normalizing divisor for acceleration and velocity is the
 * max absolute value that occurred during the move. For position it's the maximum stroke
 * length of a blade set. Predicted values are shown with line charts while actual position
 * measurements from Hall transitions and encoder samples are shown as individual points.
 * Each blade set uses its own set of plot colors.
 * <p>
 * This class is not thread safe and its methods should be called only from
 * the event dispatch thread.
 * @author tonyj
 * @author tether
 */
public final class TrajectoryDisplay extends JPanel {

    private final static int DISPLAY_WIDTH = 500;  // pixels.
    private final static int DISPLAY_HEIGHT = 300; // pixels.
    private final static double DEFAULT_TIME_PERIOD = 5.0;   // Seconds.
    private final static float LINE_WIDTH = 3.0f; // pixels

    private final static List<Color> plotColors = Arrays.asList(
        // pred acc, pred vel, pred pos, Hall, encoder
        Color.RED, Color.BLUE, Color.GREEN, Color.YELLOW, Color.BLACK,
        Color.ORANGE, Color.CYAN, Color.GRAY, Color.GREEN, Color.RED
    );

    private final static List<Integer> predictedQuantities = Arrays.asList(
        PredictionPlotData.ACCELERATION,
        PredictionPlotData.VELOCITY,
        PredictionPlotData.POSITION
    );

    private final static List<String> predictedKeys = Arrays.asList(
        "Pred acc", "Pred vel", "Pred pos"
    );

    private final LinearAxisModel timeAxisModel;

    TrajectoryDisplay(Map<ShutterSide, PredictionPlotData> predictionData,
                  Map<ShutterSide, HallPlotData> hallData,
                  Map<ShutterSide, EncoderPlotData> encoderData) {

        super(new BorderLayout());

        // We'll use a panel with a PlotLayout, which places an added item according to its type.
        final JPanel plotPanel = new JPanel(new PlotLayout());
        plotPanel.setPreferredSize(new Dimension(DISPLAY_WIDTH, DISPLAY_HEIGHT));

        // The vertical axis will show the various quantities, all normalized
        // into the interval [-1, +1].
        final ContinuousAxisModel valueAxisModel = new LinearAxisModel(-1.1, 1.1);
        final Axis valueAxis = new Axis(valueAxisModel, AxisPosition.LEFT);

        // The horizontal axis will show time, in seconds, elapsed from the start of the movement.
        timeAxisModel = new LinearAxisModel(0.0, DEFAULT_TIME_PERIOD);
        final Axis timeAxis = new Axis(timeAxisModel, AxisPosition.BOTTOM);

        // Define a common canvas for all plots.
        final DataArea da = new DataArea(timeAxis, valueAxis);
        da.setBackground(Color.LIGHT_GRAY);

        // Add each plot as an overlay on the data area.
        final Map<ShutterSide, KeyPanel> legend = new EnumMap<>(ShutterSide.class);
        legend.put(MINUSX, new KeyPanel("-X side"));
        legend.put(PLUSX, new KeyPanel("+X side"));
        final Iterator<Color> colorIt = plotColors.iterator();

        for (final ShutterSide side: ShutterSide.values()) {
            // Make sure that changes to data cause a repaint of this display.
            predictionData.get(side).addPlotDataListener(this::repaint);
            hallData.get(side).addPlotDataListener(this::repaint);
            encoderData.get(side).addPlotDataListener(this::repaint);

            // Add the line plots for the predicted trajectory.
            final Iterator<String> keyIt = predictedKeys.iterator();
            for (final int what: predictedQuantities) {
                LineChartOverlay overlay =
                        new LineChartOverlay(keyIt.next(),
                            predictionData.get(side),
                            PredictionPlotData.TIME,
                            what,
                            timeAxisModel,
                            valueAxisModel);
                overlay.setLineColor(colorIt.next());
                overlay.setLineWidth(LINE_WIDTH);
                da.add(overlay);
                legend.get(side).add(overlay);
            }

            // Add the point plot for the actual trajectory measured by Hall transitions.
            DataPointChartOverlay hallSensors =
                new DataPointChartOverlay("Hall trans",
                    hallData.get(side),
                    HallPlotData.TIME,
                    HallPlotData.POSITION,
                    -1, -1,// No error bars for time axis.
                    HallPlotData.ERROR, HallPlotData.ERROR,  // Error bars for value axis.
                    timeAxisModel,
                    valueAxisModel);
            hallSensors.setLineColor(colorIt.next());
            hallSensors.setLineWidth(LINE_WIDTH);
            da.add(hallSensors);
            legend.get(side).add(hallSensors);

            // Add the point plot for the actual trajectory measured by encoder sampling.
            DataPointChartOverlay encoderPos =
                new DataPointChartOverlay("Enc pos",
                    encoderData.get(side),
                    EncoderPlotData.TIME,
                    EncoderPlotData.POSITION,
                    -1, -1,
                    EncoderPlotData.ERROR, EncoderPlotData.ERROR,
                    timeAxisModel,
                    valueAxisModel);
            encoderPos.setLineColor(colorIt.next());
            encoderPos.setLineWidth(LINE_WIDTH);
            da.add(encoderPos);
            legend.get(side).add(encoderPos);
        }

        // Add the axes and the plot display area to the plot panel.
        plotPanel.add(valueAxis);
        plotPanel.add(timeAxis);
        plotPanel.add(da);

        // Create a combined legend covering both sides of the shutter.
        JPanel bothLegends = new JPanel(new BorderLayout());
        bothLegends.add(BorderLayout.NORTH, legend.get(MINUSX));
        bothLegends.add(BorderLayout.SOUTH, legend.get(PLUSX));
        bothLegends.setBorder(BorderFactory.createTitledBorder("Legend"));

        // In this JPane put the plot+axes above the combined legend.
        add(plotPanel,BorderLayout.CENTER);
        add(bothLegends,BorderLayout.SOUTH);

    }

    /**
     * Changes the span covered by the time axis and repaints this display.
     * @param maxTime The maximum value on the time axis (the minimum is always zero).
     */
    void setTimePeriod(float maxTime) {
        timeAxisModel.setMax(maxTime);
        repaint();
    }

    
    private final static class KeyPanel extends JPanel {

        KeyPanel(String title) {
            super(new FlowLayout());
            add(new JLabel(title));
            setBackground(Color.LIGHT_GRAY);
        }

        private void add(final LineChartOverlay overlay) {
            final JLabel label = new JLabel(overlay.getTitle());
            label.setIcon(overlay.getKeyIcon());
            add(label);
        }

        private void add(final DataPointChartOverlay overlay) {
            final JLabel label = new JLabel(overlay.getTitle());
            label.setIcon(overlay.getKeyIcon());
            add(label);
        }
    }

}


