package org.lsst.ccs.subsystem.shutter.gui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Arc2D;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.JComponent;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;

/**
 * Displays a diagram of the shutter; two movable sets of three rectangular blades each, on top of
 * circular aperture.
 * This class is not thread safe and its methods should only be called from the Swing event
 * dispatch thread.
 * @author tonyj
 * @author tether
 */
public final class BladesDisplay extends JComponent {

    private final static int DISPLAY_WIDTH = 400;  // pixels
    private final static int DISPLAY_HEIGHT = 500; // pixels
    private final static int BLADE_WIDTH = DISPLAY_WIDTH;
    private final static int BLADE_HEIGHT = 100; //pixels
    private final static int APERTURE_DIAMETER = 300; // pixels
    private final static int APERTURE_WIDTH_INSET = (DISPLAY_WIDTH - APERTURE_DIAMETER) / 2;
    private final static int APERTURE_HEIGHT_INSET = (DISPLAY_HEIGHT - APERTURE_DIAMETER) / 2;
    private final static Color BLADE_FILL_COLOR = new Color(
        0.5f, 0.5f, 0.5f, // RGB saturations. Light greyish.
        0.5f              // alpha value. Semi-transparent.
    );
    private final static int AXIS_SIZE = 50; // pixels
    private final static int ARROWHEAD_SIZE = 5; // pixels
    private final static int AXIS_ORIGIN_OFFSET= 5; // pixels
    private final static int AXIS_LABEL_POS = AXIS_SIZE - 5;  // How far along axis to put label.
    private final static int AXIS_LABEL_SEP = 15; // pixels. How far away from axis to put label.

    private final Map<ShutterSide, BladeSet> shutters = new EnumMap<>(ShutterSide.class);

    public BladesDisplay() {
        final Dimension dimension = new Dimension(DISPLAY_WIDTH, DISPLAY_HEIGHT);
        setPreferredSize(dimension);
        setMinimumSize(dimension);
        shutters.put(PLUSX, new BladeSet(BLADE_WIDTH, BLADE_HEIGHT));
        shutters.put(MINUSX, new BladeSet(BLADE_WIDTH, BLADE_HEIGHT));

        // Until such time as position information is sent from the worker
        // subsystem set the shutter as closed with the -X blade set
        // extended and the +X set retracted.
        shutters.get(PLUSX).setPosition(0.0f);
        shutters.get(MINUSX).setPosition(1.0f);
    }

    BladeSet getShutter(final ShutterSide side) {
        return shutters.get(side);
    }

    @Override
    protected void paintComponent(Graphics g) {
        Graphics2D g2 = (Graphics2D) g.create();
        RenderingHints hints =
            new RenderingHints(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2.addRenderingHints(hints);
        // Paint a border representing the shutter enclosure.
        g2.drawRect(0, 0, DISPLAY_WIDTH, DISPLAY_HEIGHT);
        // Paint the aperture.
        Shape circle = new Arc2D.Float(
            APERTURE_WIDTH_INSET, APERTURE_HEIGHT_INSET, // Framing rectangle upper left corner.
            APERTURE_DIAMETER, APERTURE_DIAMETER,        // Framing rectangle width and height.
            0, 360, // Draw all 360 degrees.
            Arc2D.CHORD); // Connection between arc endpoints. Zero-length chord in this case.
        g2.draw(circle);

        shutters.get(PLUSX).paint(g2);    // Retracted position at top.
        g2.translate(0, DISPLAY_HEIGHT);
        g2.scale(1, -1);
        shutters.get(MINUSX).paint(g2);   // Retracted position at bottom.
        
        // On the display draw camera axis directions using DM orientation.
        // The coordinates for drawing are the standard math XY axes with Y inreasing upwards,
        // origin at lower left. Since that reverses the Y axis we have to invert the font
        // as well to get the text right side up.
        final Graphics2D g3 = (Graphics2D)g.create();
        final AffineTransform flip = AffineTransform.getScaleInstance(1, -1);
        g3.translate(0, DISPLAY_HEIGHT);
        g3.transform(flip);
        g3.setFont(g3.getFont().deriveFont(flip));
        g3.addRenderingHints(hints);
        drawAxes(g3, AXIS_SIZE, ARROWHEAD_SIZE, AXIS_ORIGIN_OFFSET, AXIS_ORIGIN_OFFSET);
        g3.drawString("X", AXIS_LABEL_SEP, AXIS_LABEL_POS);
        g3.drawString("Y", AXIS_LABEL_POS, AXIS_LABEL_SEP);
    }

    static class BladeSet {

        private final List<Blade> blades = new ArrayList<>();

        BladeSet(int bladeWidth, int bladeHeight) {
            for (int i = 0; i < 3; i++) {
                blades.add(new Blade(bladeWidth, bladeHeight));
            }
        }

        /**
         * Set the position of the innermost edge of the blade set.
         * @param pos 0=retracted, 1=extended.
         */
        void setPosition(float pos) {
            pos = Math.max(0, Math.min(1, pos));
            blades.get(0).setPosition(1 * BLADE_HEIGHT * pos);
            blades.get(1).setPosition(2 * BLADE_HEIGHT * pos);
            blades.get(2).setPosition(3 * BLADE_HEIGHT * pos);
        }

        float getPosition() {
            return blades.get(0).getPosition() / 100;
        }

        void paint(Graphics2D g2) {
            for (Blade blade : blades) {
                blade.paint(g2);
            }
        }
    }

    private static class Blade extends Rectangle.Float {

        Blade(int width, int height) {
            super(0, 0, width, height);
        }

        void setPosition(float y) {
            super.y = y;
        }

        float getPosition() {
            return super.y;
        }

        private void paint(Graphics2D g2) {
            g2.setColor(BLADE_FILL_COLOR);
            g2.fill(this);
            g2.setColor(Color.BLACK);
            g2.draw(this);
        }
    }
    
    // Object construction routines below will use a standard coordinate system with Y 
    // increasing upward.
    
    // Make a standard arrowhead which will be transformed as needed.
    private GeneralPath makeArrowhead() {
        final GeneralPath head = new GeneralPath();
        head.moveTo(0.0, 0.0);  // apex
        head.lineTo(-1.0, -3.0);
        head.lineTo(+1.0, -3.0);
        head.closePath();
        return head;
    }
    
    // Plain axes to be transformed as needed.
    private GeneralPath makeAxes() {
        final GeneralPath axes = new GeneralPath();
        axes.moveTo(0.0, 1.0);
        axes.lineTo(0.0, 0.0);  // origin
        axes.lineTo(1.0, 0.0);
        return axes;
    }
    
    private void drawAxes(Graphics2D g, double axisScale, double arrowScale, double x, double y) {
        // Create plain axes scaled and translated.
        final GeneralPath axes = makeAxes();
        axes.transform(AffineTransform.getScaleInstance(axisScale, axisScale));
        axes.transform(AffineTransform.getTranslateInstance(x, y));
        g.draw(axes);
        // Create arrowheads scaled, rotated and translated.
        final GeneralPath xhead = makeArrowhead();
        xhead.transform(AffineTransform.getScaleInstance(arrowScale, arrowScale));
        xhead.transform(AffineTransform.getQuadrantRotateInstance(3));
        // Adding 2.0 tprevents the end of the axis from poking out past the tip of the arrowhead.
        xhead.transform(AffineTransform.getTranslateInstance(axisScale + x + 2.0, y));
        g.fill(xhead);
        final GeneralPath yhead = makeArrowhead();
        yhead.transform(AffineTransform.getScaleInstance(arrowScale, arrowScale));
        yhead.transform(AffineTransform.getTranslateInstance(x, axisScale + y + 2.0));
        g.fill(yhead);
    }

}

