package org.lsst.ccs.subsystem.shutter.gui;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Queues GUI actions and dispatches them to a fixed list of objects that
 * implement the {@code PluginActions} interface.
 * @author tether
 */
public class Dispatcher implements PluginActions {
    private static final Logger LOG = Logger.getLogger(Dispatcher.class.getName());

    private final ShutterPlugin plugin;
    private final List<PluginActions> receivers;
    private final BlockingQueue<Runnable> actions;
    
    public Dispatcher(final ShutterPlugin plugin, PluginActions... receivers) {
        this.plugin = plugin;
        this.receivers = Collections.unmodifiableList(Arrays.asList(receivers));
        this.actions = new LinkedBlockingQueue<>();
    }

    void start() {
        // Start a task that reads and executes the queued actions
        // until interrupted. Thread will restart after 10 millisec if it crashes.
        plugin.getScheduler().setLogger(LOG);
        plugin.getScheduler().setDefaultLogLevel(Level.SEVERE);
        LOG.fine("Starting the dispatcher task.");
        plugin.getScheduler().scheduleWithFixedDelay(
            () -> {
                try {
                    while (true) {
                        actions.take().run();
                    }
                }
                catch (InterruptedException exc) {
                    LOG.fine("The dispatcher task is stopped.");
                }
            },
            0, 10, TimeUnit.MILLISECONDS);
    }

    ////////// Implementation of PluginActions //////////

    @Override
    public void takeExposure(double exposureTime) {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.takeExposure(exposureTime);});
    }

    @Override
    public void showTrajectory(MotionDone motion) {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.showTrajectory(motion);});
    }

    @Override
    public void showWorkerIsUnreachable(String message) {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.showWorkerIsUnreachable(message);});
    }

    @Override
    public void showWorkerIsReachable() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.showWorkerIsReachable();});
    }

    @Override
    public void showStatus(ShutterStatus status) {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.showStatus(status);});
    }

    @Override
    public void showStateBundle(PhysicalState phys, SoftwareState soft) {
        actions.add(() -> {for (final PluginActions rcv: receivers) rcv.showStateBundle(phys, soft);});
    }

    @Override
    public void showCommandFailure(String message) {
        actions.add(() -> {for (final PluginActions rcv: receivers) rcv.showCommandFailure(message);});
    }

    @Override
    public void showCommandSuccess() {
        actions.add(() -> {for (final PluginActions rcv: receivers) rcv.showCommandSuccess();});
    }

    @Override
    public void openShutter() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.openShutter();});
    }

    @Override
    public void disconnect() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.disconnect();});
    }

    @Override
    public void connect() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.connect();});
    }

    @Override
    public void closeShutter() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.closeShutter();});
    }

    @Override
    public void changeBrakeState(final ShutterSide side, final boolean brakeEngaged) {
        actions.add(() -> {
            for (final PluginActions rcv: receivers) rcv.changeBrakeState(side, brakeEngaged);});        
    }

    @Override
    public void stopMotion() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.stopMotion();});
    }

    @Override
    public void resync() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.resync();});
    }

    @Override
    public void goToProd() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.goToProd();});
    }

    @Override
    public void center() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.center();});
    }

    @Override
    public void calibrate() {
        actions.add(
            () -> {for (final PluginActions rcv: receivers) rcv.calibrate();});
    }

}
