package org.lsst.ccs.subsystem.shutter.gui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Arc2D;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import javax.swing.JComponent;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;

/**
 * Displays a diagram of the shutter; two movable sets of three rectangular blades each, on top of
 * circular aperture.
 * This class is not thread safe and its methods should only be called from the Swing event
 * dispatch thread.
 * @author tonyj
 * @author tether
 */
public final class BladesDisplay extends JComponent {

    private final static int DISPLAY_WIDTH = 500;  // pixels
    private final static int DISPLAY_HEIGHT = 400; // pixels
    private final static int BLADE_WIDTH = 100;    // pixels
    private final static int BLADE_HEIGHT = DISPLAY_HEIGHT;
    private final static int APERTURE_DIAMETER = 300;
    private final static int APERTURE_WIDTH_INSET = (DISPLAY_WIDTH - APERTURE_DIAMETER) / 2;
    private final static int APERTURE_HEIGHT_INSET = (DISPLAY_HEIGHT - APERTURE_DIAMETER) / 2;
    private final static Color BLADE_FILL_COLOR = new Color(
        0.5f, 0.5f, 0.5f, // RGB saturations. Light greyish.
        0.5f              // alpha value. Semi-transparent.
    );

    private final Map<ShutterSide, BladeSet> shutters = new EnumMap<>(ShutterSide.class);

    public BladesDisplay() {
        final Dimension dimension = new Dimension(DISPLAY_WIDTH, DISPLAY_HEIGHT);
        setPreferredSize(dimension);
        setMinimumSize(dimension);
        shutters.put(PLUSX, new BladeSet(BLADE_WIDTH, BLADE_HEIGHT));
        shutters.put(MINUSX, new BladeSet(BLADE_WIDTH, BLADE_HEIGHT));

        // Until such time as position information is sent from the worker
        // subsystem set the shutter as closed with the -X blade set
        // extended and the +X set retracted.
        shutters.get(PLUSX).setPosition(0.0f);
        shutters.get(MINUSX).setPosition(1.0f);
    }

    BladeSet getShutter(final ShutterSide side) {
        return shutters.get(side);
    }

    @Override
    protected void paintComponent(Graphics g) {
        Graphics2D g2 = (Graphics2D) g;
        RenderingHints hints =
            new RenderingHints(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2.addRenderingHints(hints);
        // Paint the aperture.
        Shape circle = new Arc2D.Float(
            APERTURE_WIDTH_INSET, APERTURE_HEIGHT_INSET, // Framing rectangle upper left corner.
            APERTURE_DIAMETER, APERTURE_DIAMETER,        // Framing rectangle width and height.
            0, 360, // Draw all 360 degrees.
            Arc2D.CHORD); // Connection between arc endpoints. Zero-length chord in this case.
        g2.draw(circle);

        // Paint the -X shutter on the left, extending toward the right.
        shutters.get(MINUSX).paint(g2);
        AffineTransform savedTransform = g2.getTransform();
        try {
            // Paint the +X shutter ...
            g2.translate(DISPLAY_WIDTH, 0);  // ... on the right ...
            g2.scale(-1,1);        // ... and extending toward the left.
            shutters.get(PLUSX).paint(g2);
        }
        finally {
            g2.setTransform(savedTransform);
        }

    }

    static class BladeSet {

        private final List<Blade> blades = new ArrayList<>();

        BladeSet(int bladeWidth, int bladeHeight) {
            for (int i = 0; i < 3; i++) {
                blades.add(new Blade(bladeWidth, bladeHeight));
            }
        }

        /**
         * Set the position of the innermost edge of the blade set.
         * @param pos 0=retracted, 1=extended.
         */
        void setPosition(float pos) {
            pos = Math.max(0, Math.min(1, pos));
            blades.get(0).setPosition(1 * BLADE_WIDTH * pos);
            blades.get(1).setPosition(2 * BLADE_WIDTH * pos);
            blades.get(2).setPosition(3 * BLADE_WIDTH * pos);
        }

        float getPosition() {
            return blades.get(0).getPosition() / 100;
        }

        void paint(Graphics2D g2) {
            for (Blade blade : blades) {
                blade.paint(g2);
            }
        }
    }

    private static class Blade extends Rectangle.Float {

        Blade(int width, int height) {
            super(0, 0, width, height);
        }

        void setPosition(float x) {
            super.x = x;
        }

        float getPosition() {
            return super.x;
        }

        private void paint(Graphics2D g2) {
            g2.setColor(BLADE_FILL_COLOR);
            g2.fill(this);
            g2.setColor(Color.BLACK);
            g2.draw(this);
        }
    }

}

