package org.lsst.ccs.subsystem.shutter.gui;

import data.DiscretePlotData;
import data.MetaData;
import data.MutablePlotData;
import data.PlotDataListener;
import data.SuggestedRange;
import java.util.ArrayList;
import java.util.List;
import org.freehep.util.Value;
import util.ListenerList;

import org.lsst.ccs.subsystem.shutter.common.HallTransition;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Contains Hall transition data gathered from the actual shutter.
 * @author azemoon
 * @author tether
 */
public class HallPlotData implements DiscretePlotData, MutablePlotData {

    public static final int TIME = 0;
    public static final int POSITION = 1;
    public static final int ERROR = 2;

    private final List<HallDatum> data = new ArrayList<HallDatum>();
    private final ListenerList<PlotDataListener> listeners = new ListenerList<PlotDataListener>();
    private final String[] names = {"Time", "Position"};
    private static double DEFAULT_ERROR = 0.02;

    public HallPlotData() {
    }

    /**
     * Clears away all data and causes the plot pane to be repainted.
     */
    public void reset() {
       data.clear();
       firePlotDataChanged();
    }

    /**
     * Adds new Hall transitions and causes the plot pane to be repainted.
     * @param trans the list of Hall transitions
     * @param t0 the start time for the motion
     */
    public void addData(List<HallTransition> trans, CCSTimeStamp t0) {
        for (final HallTransition h: trans) {
            final double t = h.timeDiff(t0).toMillis() * 1e-3;
            final HallDatum d = new HallDatum(t, h.getPosition(), DEFAULT_ERROR);
            data.add(d);
        }
        firePlotDataChanged();
    }

    @Override
    public int getNPoints() {
        return data.size();
    }

    @Override
    public int getNDimensions() {
        return names.length;
    }

    @Override
    public String names(int index) {
        return names[index];
    }

    @Override
    public Class types(int index) {
        return Double.TYPE;
    }

    @Override
    public void getValue(Value value, int dim, int index) {
        HallDatum transition = data.get(index);
        switch (dim) {
            case TIME: value.set(transition.getTime()); break;
            case POSITION: value.set(transition.getPosition()); break;
            case ERROR: value.set(transition.getError()); break;
            default:
                throw new IndexOutOfBoundsException("dim out of bounds.");
        }
    }

    @Override
    public MetaData getMetaData() {
        return null;
    }

    @Override
    public String getTitle() {
        return "Hall transitions";
    }

    @Override
    public SuggestedRange getSuggestedRange() {
        return null;
    }

    @Override
    public void addPlotDataListener(PlotDataListener listener) {
        listeners.addListener(listener);
    }

    @Override
    public void removePlotDataListener(PlotDataListener listener) {
        listeners.removeListener(listener);
    }

    private void firePlotDataChanged() {
        if (!listeners.isEmpty()) {
            for (PlotDataListener l : listeners.getListeners()) {
                l.dataChanged();
            }
        }
    }

    @Override
    public Object lock() {
        return data;
    }

    private static class HallDatum {

        private final double time;
        private final double position;
        private final double error;

        public HallDatum(double time, double position, double error) {
            this.time = time;
            this.position = position;
            this.error = error;
        }

        public double getTime() {
            return time;
        }

        public double getPosition() {
            return position;
        }

        public double getError() {
            return error;
        }
    }
}
