package org.lsst.ccs.subsystem.shutter.gui;

import java.util.Arrays;
import java.util.logging.Logger;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusAggregator;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.shutter.common.Constants;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;

/**
 * Detects the presence or absence of the worker subsystem on the CCS status bus
 * and listens for any status messages from it.
 * @author tether
 */
public class StatusListener implements PluginActions, AgentStatusListener, StatusMessageListener  {
    private static final Logger LOG = Logger.getLogger(StatusListener.class.getName());

    private final ShutterPlugin plugin;

    public StatusListener(final ShutterPlugin plugin) {
        this.plugin = plugin;
    }

    ////////// Implementation of PluginActions //////////

    @Override
    public void disconnect() {
        LOG.fine("disconnect()");
        final AgentStatusAggregator agg = plugin.getConsole().getStatusAggregator();
        agg.removeListener(this);
        final AgentMessagingLayer aml = plugin.getConsole().getMessagingAccess();
        aml.removeStatusMessageListener(this);
    }

    @Override
    public void connect() {
        LOG.fine("connect()");
        final AgentStatusAggregator agg = plugin.getConsole().getStatusAggregator();
        agg.addListener(this, Arrays.asList(plugin.getWorkerName()), null);
        final AgentMessagingLayer aml = plugin.getConsole().getMessagingAccess();
        aml.addStatusMessageListener(
            this,
            msg -> msg.getOriginAgentInfo().getName().equals(plugin.getWorkerName()));
    }

    ////////// Implementation of AgentStatusListener //////////

    @Override
    public void disconnect(AgentStatusEvent event) {
        LOG.fine("Worker is offline.");
        plugin.getDispatcher().showWorkerIsUnreachable("OFFLINE");
    }

    @Override
    public void connect(AgentStatusEvent event) {
        LOG.fine("Worker is online.");
        plugin.getDispatcher().showWorkerIsReachable();
    }

    ////////// Implementation of StatusMessageListener //////////

    @Override
    public void onStatusMessage(final StatusMessage msg) {
        LOG.fine(msg.toString());
        final PhysicalState pstate =
            (PhysicalState)
            msg
            .getState()
            .getComponentState(Constants.PHYSICAL_STATE_PATH, PhysicalState.class);
        plugin.getDispatcher().showPhysicalState(pstate);
        if (msg instanceof StatusSubsystemData) {
            final StatusSubsystemData sd = (StatusSubsystemData)msg;
            final String key = sd.getDataKey();
            if (key.equals(ShutterStatus.class.getSimpleName())) {
                final ShutterStatus status =
                    (ShutterStatus)
                    sd.getSubsystemData()
                    .getValue();
                plugin.getDispatcher().showStatus(status);
            }
            else if (key.equals(MotionDone.class.getSimpleName())) {
                final MotionDone status =
                    (MotionDone)
                    sd.getSubsystemData()
                    .getValue();
                plugin.getDispatcher().showTrajectory(status);
            }
        }

    }

}
