package org.lsst.ccs.subsystem.shutter.gui;

import data.DiscretePlotData;
import data.MetaData;
import data.MutablePlotData;
import data.PlotDataListener;
import data.SuggestedRange;
import java.util.ArrayList;
import java.util.List;
import org.freehep.util.Value;
import org.lsst.ccs.subsystem.shutter.common.EncoderSample;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;
import util.ListenerList;

/**
 * Contains encoder value samples taken during a blade set motion.
 * @author azemoon
 * @author tether
 */
public class EncoderPlotData implements DiscretePlotData, MutablePlotData {

    /** The dimension number used to get the time using
     * {@link #getValue(org.freehep.util.Value, int, int) } */
    public static final int TIME = 0;

    /** The dimension number used to get the position using
     * {@link #getValue(org.freehep.util.Value, int, int) } */
    public static final int POSITION = 1;

    /** The dimension number used to get the position error using 
     * {@link #getValue(org.freehep.util.Value, int, int) } */
    public static final int ERROR = 2;

    private final List<EncoderDatum> data = new ArrayList<EncoderDatum>();
    private final ListenerList<PlotDataListener> listeners = new ListenerList<PlotDataListener>();
    private final String[] names = {"Time", "Position"};
    private static double DEFAULT_ERROR = 0.02;

    public EncoderPlotData() {
    }

    /**
     * Clears all data and causes the plot pane to be repainted.
     */
    public void reset() {
       data.clear();
       firePlotDataChanged();
    }

    /**
     * Adds a new set of positions and causes the plot pane to be redrawn.
     * @param pos the list of blade positions
     * @param t0 the start time for the motion.
     */
    public void addData(List<EncoderSample> pos, CCSTimeStamp t0) {
       for (final EncoderSample p : pos){
           final double t = p.timeDiff(t0).toMillis() * 1e-3;
           data.add(new EncoderDatum(t, p.getPosition(), DEFAULT_ERROR));
      }
      firePlotDataChanged();
    }

    @Override
    public int getNPoints() {
        return data.size();
    }

    @Override
    public int getNDimensions() {
        return names.length;
    }

    @Override
    public String names(int index) {
        return names[index];
    }

    @Override
    public Class types(int index) {
        return Double.TYPE;
    }

    @Override
    public void getValue(Value value, int dim, int index) {
        EncoderDatum mp = data.get(index);
        switch (dim) {
            case 0: value.set(mp.getTime()); break;
            case 1: value.set(mp.getPosition()); break;
            case 2: value.set(mp.getError()); break;
        }
    }

    @Override
    public MetaData getMetaData() {
        return null;
    }

    @Override
    public String getTitle() {
        return "Motor Readout";
    }

    @Override
    public SuggestedRange getSuggestedRange() {
        return null;
    }

    @Override
    public void addPlotDataListener(PlotDataListener listener) {
        listeners.addListener(listener);
    }

    @Override
    public void removePlotDataListener(PlotDataListener listener) {
        listeners.removeListener(listener);
    }

    private void firePlotDataChanged() {
        if (!listeners.isEmpty()) {
            for (PlotDataListener l : listeners.getListeners()) {
                l.dataChanged();
            }
        }
    }

    @Override
    public Object lock() {
        return data;
    }

    private static class EncoderDatum {

        private final double time;
        private final double position;
        private final double error;

        public EncoderDatum(double time, double position, double error) {
            this.time = time;
            this.position = position;
            this.error = error;
        }

        public double getTime() {
            return time;
        }

        public double getPosition() {
            return position;
        }

        public double getError() {
            return error;
        }
    }

}
