package org.lsst.ccs.subsystem.shutter.gui;

import java.util.EnumMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.JTextField;
import static javax.swing.SwingUtilities.invokeLater;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;

/**
 * Provides various controls for the shutter subsystem. This class is not thread safe
 * and its methods should be called from the Swing event dispatch thread, via invokeLater() for example.
 * @author tonyj
 * @author tether
 */
public class ControlPanel extends javax.swing.JPanel implements PluginActions {
    private static final Logger LOG = Logger.getLogger(ControlPanel.class.getName());

    final ShutterPlugin plugin;
    final Map<ShutterSide, Boolean> brakeIsOn;
    final Map<ShutterSide, JTextField> brakeField;

    /**
     * Saves the plugin reference for later use and initializes components of this panel.
     * @param plugin The console plugin being used to run the GUI.
     */
    public ControlPanel(final ShutterPlugin plugin) {
        this.plugin = plugin;
        initComponents();
        this.brakeIsOn = new EnumMap<>(ShutterSide.class);
        this.brakeIsOn.put(ShutterSide.PLUSX, null);
        this.brakeIsOn.put(ShutterSide.MINUSX, null);
        this.brakeField = new EnumMap<>(ShutterSide.class);
        this.brakeField.put(ShutterSide.PLUSX, plusXBrakeField);
        this.brakeField.put(ShutterSide.MINUSX, minusXBrakeField);
    }

    ////////// Implementation of PluginActions //////////
    @Override
    public void showCommandFailure(String message) {
        LOG.log(Level.FINE, "showCommandFailure(): {0}", message);
    }

    @Override
    public void showCommandSuccess() {
        LOG.fine("showCommandSuccess()");
    }

    @Override
    public void showPhysicalState(final PhysicalState phys) {
        LOG.log(Level.FINE, "Physical state is {0}", phys);
        invokeLater( () -> shutterStateText.setText(phys.toString()) );
    }

    @Override
    public void showStatus(final ShutterStatus status) {
        invokeLater( () -> {
            for (final ShutterSide side: ShutterSide.values()) {
                brakeIsOn.put(side, status.getAxisStatus(side).isBrakeSet());
                brakeField.get(side).setText(brakeIsOn.get(side) ? "ON" : "OFF");
            }
        });
    }

    @Override
    public void showWorkerIsReachable() {
        invokeLater( () -> setEnabled(true));
    }

    @Override
    public void showWorkerIsUnreachable(final String message) {
        LOG.fine(message);
        invokeLater(() -> setEnabled(false));
    }
    
    ////////// End of PluginActions implementation //////////

    
    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        takeExposureButton = new javax.swing.JButton();
        exposureTimeLabel = new javax.swing.JLabel();
        exposureTimeSpinner = new javax.swing.JSpinner();
        secondsLabel = new javax.swing.JLabel();
        closeButton = new javax.swing.JButton();
        openButton = new javax.swing.JButton();
        jLabel1 = new javax.swing.JLabel();
        shutterStateText = new javax.swing.JTextField();
        stopButton = new javax.swing.JButton();
        calibrateButton = new javax.swing.JButton();
        resyncButton = new javax.swing.JButton();
        centerButton = new javax.swing.JButton();
        prodModeButton = new javax.swing.JButton();
        jSeparator2 = new javax.swing.JSeparator();
        jSeparator3 = new javax.swing.JSeparator();
        plusXBrakeButton = new javax.swing.JButton();
        plusXBrakeField = new javax.swing.JTextField();
        minusXBrakeButton = new javax.swing.JButton();
        minusXBrakeField = new javax.swing.JTextField();

        takeExposureButton.setText("Take exposure");
        takeExposureButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleTakeExposure(evt);
            }
        });

        exposureTimeLabel.setText("Exposure time:");

        exposureTimeSpinner.setModel(new javax.swing.SpinnerNumberModel(Float.valueOf(5.0f), Float.valueOf(1.0f), Float.valueOf(20.0f), Float.valueOf(0.1f)));

        secondsLabel.setText("seconds");

        closeButton.setText("Close");
        closeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleCloseShutter(evt);
            }
        });

        openButton.setText("Open");
        openButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleOpenShutter(evt);
            }
        });

        jLabel1.setText("Shutter state:");

        shutterStateText.setEditable(false);
        shutterStateText.setHorizontalAlignment(javax.swing.JTextField.LEFT);

        stopButton.setBackground(new java.awt.Color(255, 0, 51));
        stopButton.setForeground(new java.awt.Color(0, 0, 0));
        stopButton.setText("Stop/reset");
        stopButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleStop(evt);
            }
        });

        calibrateButton.setText("Calibrate");
        calibrateButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleCalibrate(evt);
            }
        });

        resyncButton.setText("Resync");
        resyncButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleResync(evt);
            }
        });

        centerButton.setText("Center");
        centerButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleCenter(evt);
            }
        });

        prodModeButton.setText("Prod mode");
        prodModeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleProdMode(evt);
            }
        });

        plusXBrakeButton.setText("+X brake");
        plusXBrakeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handlePlusXBrake(evt);
            }
        });

        plusXBrakeField.setEditable(false);

        minusXBrakeButton.setText("-X brake");
        minusXBrakeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleMinusXBrake(evt);
            }
        });

        minusXBrakeField.setEditable(false);

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addComponent(jSeparator3)
                .addContainerGap())
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createSequentialGroup()
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(jLabel1)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(shutterStateText, javax.swing.GroupLayout.PREFERRED_SIZE, 136, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addGap(0, 0, Short.MAX_VALUE)
                                .addComponent(stopButton))
                            .addComponent(jSeparator2))
                        .addContainerGap())
                    .addGroup(layout.createSequentialGroup()
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(takeExposureButton)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(exposureTimeLabel)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(exposureTimeSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(secondsLabel)
                                .addGap(18, 18, 18)
                                .addComponent(closeButton)
                                .addGap(18, 18, 18)
                                .addComponent(openButton))
                            .addGroup(layout.createSequentialGroup()
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(calibrateButton)
                                        .addGap(18, 18, 18)
                                        .addComponent(prodModeButton)
                                        .addGap(18, 18, 18)
                                        .addComponent(resyncButton))
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(plusXBrakeButton)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(plusXBrakeField, javax.swing.GroupLayout.PREFERRED_SIZE, 41, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addGap(18, 18, 18)
                                        .addComponent(minusXBrakeButton)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(minusXBrakeField, javax.swing.GroupLayout.PREFERRED_SIZE, 44, javax.swing.GroupLayout.PREFERRED_SIZE)))
                                .addGap(26, 26, 26)
                                .addComponent(centerButton)))
                        .addGap(0, 51, Short.MAX_VALUE))))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel1)
                    .addComponent(shutterStateText, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(stopButton, javax.swing.GroupLayout.PREFERRED_SIZE, 43, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(18, 18, 18)
                .addComponent(jSeparator3, javax.swing.GroupLayout.PREFERRED_SIZE, 15, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(18, 18, 18)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(takeExposureButton)
                    .addComponent(exposureTimeLabel)
                    .addComponent(exposureTimeSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(secondsLabel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(closeButton)
                    .addComponent(openButton))
                .addGap(18, 18, 18)
                .addComponent(jSeparator2, javax.swing.GroupLayout.PREFERRED_SIZE, 19, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(18, 18, 18)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(calibrateButton)
                    .addComponent(prodModeButton)
                    .addComponent(resyncButton)
                    .addComponent(centerButton))
                .addGap(18, 18, 18)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(plusXBrakeButton)
                    .addComponent(plusXBrakeField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(minusXBrakeButton)
                    .addComponent(minusXBrakeField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap())
        );
    }// </editor-fold>//GEN-END:initComponents

    private void handleTakeExposure(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleTakeExposure
        plugin.getDispatcher().takeExposure(getExposureTime());
    }//GEN-LAST:event_handleTakeExposure

    private void handleCloseShutter(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleCloseShutter
        plugin.getDispatcher().closeShutter();
    }//GEN-LAST:event_handleCloseShutter

    private void handleOpenShutter(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleOpenShutter
        plugin.getDispatcher().openShutter();
    }//GEN-LAST:event_handleOpenShutter

    private void handleStop(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleStop
        plugin.getDispatcher().stopMotion();
    }//GEN-LAST:event_handleStop

    private void handleCalibrate(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleCalibrate
        plugin.getDispatcher().calibrate();
    }//GEN-LAST:event_handleCalibrate

    private void handleProdMode(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleProdMode
        plugin.getDispatcher().goToProd();
    }//GEN-LAST:event_handleProdMode

    private void handleResync(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleResync
        plugin.getDispatcher().resync();
    }//GEN-LAST:event_handleResync

    private void handleCenter(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleCenter
        plugin.getDispatcher().center();
    }//GEN-LAST:event_handleCenter

    private void handlePlusXBrake(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handlePlusXBrake
        final Boolean state = brakeIsOn.get(ShutterSide.PLUSX);
        if (state != null) {
            plugin.getDispatcher().changeBrakeState(ShutterSide.PLUSX, !state);
        }
    }//GEN-LAST:event_handlePlusXBrake

    private void handleMinusXBrake(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleMinusXBrake
        final Boolean state = brakeIsOn.get(ShutterSide.MINUSX);
        if (state != null) {
            plugin.getDispatcher().changeBrakeState(ShutterSide.MINUSX, !state);
        }
    }//GEN-LAST:event_handleMinusXBrake

    public double getExposureTime() {
        return ((Number) exposureTimeSpinner.getValue()).doubleValue();
    }

    @Override
    public void setEnabled(boolean enabled) {
        takeExposureButton.setEnabled(enabled);
        exposureTimeSpinner.setEnabled(enabled);
        closeButton.setEnabled(enabled);
        openButton.setEnabled(enabled);
        calibrateButton.setEnabled(enabled);
        prodModeButton.setEnabled(enabled);
        resyncButton.setEnabled(enabled);
        centerButton.setEnabled(enabled);
        stopButton.setEnabled(enabled);
        plusXBrakeButton.setEnabled(enabled);
        minusXBrakeButton.setEnabled(enabled);
        if (! enabled) {
            shutterStateText.setText("");
            for (final ShutterSide side: ShutterSide.values()) {brakeField.get(side).setText("");}
        }
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton calibrateButton;
    private javax.swing.JButton centerButton;
    private javax.swing.JButton closeButton;
    private javax.swing.JLabel exposureTimeLabel;
    private javax.swing.JSpinner exposureTimeSpinner;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JSeparator jSeparator2;
    private javax.swing.JSeparator jSeparator3;
    private javax.swing.JButton minusXBrakeButton;
    private javax.swing.JTextField minusXBrakeField;
    private javax.swing.JButton openButton;
    private javax.swing.JButton plusXBrakeButton;
    private javax.swing.JTextField plusXBrakeField;
    private javax.swing.JButton prodModeButton;
    private javax.swing.JButton resyncButton;
    private javax.swing.JLabel secondsLabel;
    private javax.swing.JTextField shutterStateText;
    private javax.swing.JButton stopButton;
    private javax.swing.JButton takeExposureButton;
    // End of variables declaration//GEN-END:variables

}
