package org.lsst.ccs.subsystem.shutter.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Font;
import static java.awt.Font.BOLD;
import static java.awt.Font.ITALIC;
import java.awt.GridLayout;
import java.util.EnumMap;
import java.util.Map;
import java.util.function.Function;
import static javax.swing.BorderFactory.createCompoundBorder;
import static javax.swing.BorderFactory.createEmptyBorder;
import static javax.swing.BorderFactory.createLineBorder;
import static javax.swing.BorderFactory.createTitledBorder;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import static javax.swing.SwingUtilities.invokeLater;
import javax.swing.border.TitledBorder;
import static org.lsst.ccs.subsystem.shutter.common.Constants.BLADE_SET_HOME;
import static org.lsst.ccs.subsystem.shutter.common.Constants.MAX_STROKE_LENGTH;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import static org.lsst.ccs.subsystem.shutter.gui.GUIUtil.exampleShutterStatus;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;

/**
 * Displays the latest status information for the two blade sets. Not thread safe, use from the
 * event dispatch thread.
 * @author tether
 */
public class BladeSetStatusPanel extends JPanel implements PageActions {
    
    public BladeSetStatusPanel() {
        this.setLayout(new GridLayout(1 + Row.values().length, 3));
        
        final Function<String, JLabel> newHeading = (final String text) -> {
            final JLabel label = new JLabel(text);
            label.setFont(label.getFont().deriveFont(BOLD | ITALIC));
            label.setHorizontalAlignment(SwingConstants.CENTER);
            return label;
            
        };
        
        this.add(newHeading.apply("Parameter"));
        this.add(newHeading.apply("+X side"));
        this.add(newHeading.apply("-X side"));
        
        for (final Row r: Row.values()) {
            this.add(r.rowName);
            this.add(r.values.get(PLUSX));
            this.add(r.values.get(MINUSX));
        }
        
        final TitledBorder border = createTitledBorder(createLineBorder(Color.black, 1), "Blade set status");
        border.setTitleFont(border.getTitleFont().deriveFont(BOLD));
        border.setTitleJustification(TitledBorder.CENTER);
        this.setBorder(
            createCompoundBorder(
                border,
                createEmptyBorder(5, 5, 5, 5))  // Inset the interior elements slightly.
        );
    }

    private static final Font PLAIN_TEXTFIELD_FONT = (new JTextField()).getFont();
    
    private static final Font ITALIC_TEXTFIELD_FONT = PLAIN_TEXTFIELD_FONT.deriveFont(ITALIC);

    @Override
    public void showStatus(final ShutterStatus status) {
            for (final ShutterSide side: ShutterSide.values()) {
                final ShutterStatus.AxisStatus axstatus = status.getAxisStatus(side);

                Row.BrakeStatus.setValue(side, axstatus.isBrakeEngaged() ? "ENGAGED" : "RELEASED");

                final double pos = axstatus.getActPos();
                final double vel = axstatus.getActVel();
                final double home = BLADE_SET_HOME.get(side);
                final double percent = 100.0 * Math.abs(pos - home) / MAX_STROKE_LENGTH;
                Row.Position.setValue(side, pos);
                Row.Extension.setValue(side, percent);
                Row.Velocity.setValue(side, vel);


                Row.ChipTemp.setValue(side, axstatus.getMotorTemp());
                final JTextField value = Row.ChipTemp.values.get(side);
                if (axstatus.hasSafeTemp()) {
                    value.setForeground(Color.GREEN);
                    value.setFont(PLAIN_TEXTFIELD_FONT);
                }
                else {
                    value.setForeground(Color.RED);
                    value.setFont(ITALIC_TEXTFIELD_FONT);
                }
            }
    }

    @Override
    public void showWorkerIsUnreachable(final String message) {
        for (final Row r: Row.values()) {r.clearValues();}
    }
            

    private enum Row {
        Position("Position", "% 5.1f mm"),
        Extension("How far extended", "% 5.1f%%"),
        Velocity("Velocity", "% 7.1f mm/s"),
        ChipTemp("IC chip temperature", "% 5.1f \u2103"),  // u2103 is the "degrees Celsius" sign.
        BrakeStatus("Axis brake", "%s");
        
        public final JLabel rowName;
        public final String format;
        public final Map<ShutterSide, JTextField> values;
        
        Row(final String rowName, final String format) {
            this.rowName = new JLabel(rowName);
            this.format = format;
            this.values = new EnumMap<>(ShutterSide.class);
            values.put(PLUSX, new JTextField());
            values.put(MINUSX, new JTextField());
            values.get(PLUSX).setEditable(false);
            values.get(MINUSX).setEditable(false);
        }
        
        public void setValue(final ShutterSide side, final Object v) {
            values.get(side).setText(String.format(format, v));
        }
        
        public void clearValues() {
            values.get(PLUSX).setText("");
            values.get(MINUSX).setText("");
        }
    }
    
    public static void main(final String[] args) {
        invokeLater( () -> {
            final JFrame frame = new JFrame();
            final BladeSetStatusPanel panel = new BladeSetStatusPanel();
            frame.getContentPane().add(panel, BorderLayout.CENTER);
            panel.showStatus(exampleShutterStatus());
            frame.pack();
            frame.setVisible(true);

        });
    }
}
