package org.lsst.ccs.subsystem.shutter.gui;

import javax.swing.JComponent;
import javax.swing.JScrollPane;
import static org.lsst.ccs.bus.data.AgentCategory.SHUTTER;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.services.optpage.OptionalPage;
import org.lsst.ccs.messaging.AgentPresenceListener;
import static org.lsst.ccs.subsystem.motorplatform.bus.MotorplatformType.MOTORPLATFORM_TYPE_AGENT_PROPERTY;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * A CCS graphical console plugin that defines a menu item for the shutter in the subsystems menu
 * and which displays the shutter GUI in a panel provided by the console.
 * <p>
 * Thread-safe.
 * @author tether
 */
@Plugin(name="LSST Shutter plugin", description="Camera shutter status and controls.")
public class ShutterPlugin
    extends org.lsst.ccs.gconsole.base.ConsolePlugin
    implements AgentPresenceListener
{
    private static final Logger LOG = Logger.getLogger(ShutterPlugin.class.getPackage().getName());

    private final Scheduler scheduler;
    
    public ShutterPlugin() {
        this.scheduler = new Scheduler("Shutter plugin dispatcher", 1);
    }

    ////////// Overrides for ShutterPlugin //////////

    /**
      * Registers all optional pages.
     */
    @Override
    public void initialize() {
        LOG.fine("Initializing plugin.");
        OptionalPage.getService().add(new ShutterPage(getConsole(), scheduler));
    }

    @Override
    public void shutdown() {
        LOG.fine("Shutting down plugin.");
        scheduler.shutdownNow();
    }

    ////////// Shutter page containing the GUI //////////
    
    /** Describes the optional page for the camera shutter subsystem GUI. */
    public static class ShutterPage implements OptionalPage {

        private static final String PAGE_NAME = "Shutter GUI";
        private static final String MENU_PATH_FORMAT = "CCS Subsystems/%s/" + PAGE_NAME; // Insert agent name.

        private final Scheduler scheduler;
        private final Console console;
        private final DisplayPanel dispPanel;
        private final BladeSetStatusPanel bladeStatusTable;
        private final GeneralStatusPanel  generalStatusTable;
        private final CommandPanel commandControls;
        private final RootPanel rootPanel;
        private final JScrollPane scrollPane;
        private final Commander commander;
        private final StatusListener statusListener;
        private final Dispatcher dispat;
        
        public ShutterPage(final Console console, final Scheduler scheduler) {
            this.console = console;
            this.scheduler = scheduler;
            // Passing "this" to other constructors is OK so long as those
            // constructors don't expose it to threads other than the one
            // running this constructor. Ideally they should just
            // store the value in a private final field.
            this.dispat = new Dispatcher();
            this.dispPanel = new DisplayPanel();
            this.bladeStatusTable = new BladeSetStatusPanel();
            this.generalStatusTable = new GeneralStatusPanel();
            this.commandControls = new CommandPanel(dispat);
            this.rootPanel = RootPanel.create(dispPanel, generalStatusTable, bladeStatusTable, commandControls);
            rootPanel.setName(PAGE_NAME);
            this.scrollPane = new JScrollPane(rootPanel);
            this.commander = new Commander(this);
            this.statusListener = new StatusListener(this);

            dispat.addReceivers(
                dispPanel, bladeStatusTable, generalStatusTable, commandControls, commander, statusListener
            );
            dispat.start(this.scheduler);
        }
 
        /**
         * {@inheritDoc}
         * @return {@code false}, so that the page isn't displayed until the user asks for it.
         */
        @Override
        public boolean isAutoOpen() {return false;}
        
        /**
         * {@inheritDoc} Generates a non-null menu path only for worker subsystems
         * that have the motorplatform type of the camera shutter.
         * @param agent Information about an agent on the buses.
         * @return The CCS console menu path for a camera shutter subsystem if the agent is that subsystem,
         * otherwise null.
         * <p>
         * The UI framework will call this method multiple times, once with a null argument
         * and once for every agent seen on the CCS buses. The null argument is to permit
         * a page to be registered no matter what subsystems are on the buses.
         */
        @Override
        public String getPage(final AgentInfo agent) {
            String menuPath = null;
            if (agent != null && agent.getType() == AgentInfo.AgentType.WORKER) {
                final String motorType = agent.getAgentProperty(MOTORPLATFORM_TYPE_AGENT_PROPERTY);
                if (motorType.equals(SHUTTER.name())) {
                    menuPath = String.format(MENU_PATH_FORMAT, agent.getName());
                    dispat.setWorkerName(agent.getName());
                }
            }
            return menuPath;
        }
        
        /**
         * {@inheritDoc}
         * Provides the top-level GUI component for the camera shutter GUI.
         * @param agent the bus agent that passed the test in {@link #getPage(org.lsst.ccs.bus.data.AgentInfo)}.
         * @param existingComponent is ignored.
         * @return A new instance of {@link RootPanel}.
         */
        @Override
        public JComponent open(final AgentInfo agent, final JComponent existingComponent) {
            return scrollPane;
        }
        
        /**
         * {@inheritDoc} Informs the GUI that the camera shutter subsystem is on-line, 
         * @param pageName ignored.
         */
        @Override
        public void opened(final String pageName) {
            getDispatcher().showWorkerIsReachable();
            getDispatcher().connect();
        }
        
        /**
         * {@inheritDoc} Informs the GUI that the camera shutter subsystem is off-line.
         * @param pageName ignored.
         */
        @Override
        public void closed(final String pageName) {
            getDispatcher().disconnect();
            getDispatcher().showWorkerIsUnreachable("");
        }
        
        public Console getConsole() {return console;}
        
        public Dispatcher getDispatcher() {return dispat;}
    }
        
}
