package org.lsst.ccs.subsystem.shutter.gui;

import static java.util.Arrays.asList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import static javax.swing.SwingUtilities.invokeLater;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * Queues GUI actions and dispatches them to an adjustable list of objects that
 * implement the {@code PageActions} interface. Thread-safe.
 * @author tether
 */
public class Dispatcher implements PageActions {
    private static final Logger LOG = Logger.getLogger(Dispatcher.class.getName());

    private final List<PageActions> receivers;
    private final BlockingQueue<Runnable> actions;

    /**
     * Creates a new, inactive instance with an empty list of receivers.
     */
    public Dispatcher() {
        this.receivers = new CopyOnWriteArrayList<>();
        this.actions = new LinkedBlockingQueue<>();
    }
    
    /**
     * Safely adds receivers to the list.
     * @param recv the objects which will receive dispatched plugin actions.
     */
    public void addReceivers(final PageActions... recv) {
        receivers.addAll(asList(recv));
    }

    void start(final Scheduler sched) {
        // Start a task that reads and executes the queued actions
        // until interrupted. Thread will restart after 10 millisec if it crashes.
        sched.setLogger(LOG);
        sched.setDefaultLogLevel(Level.SEVERE);
        LOG.fine("Starting the dispatcher task.");
        sched.scheduleWithFixedDelay(
            () -> {
                try {
                    while (true) {
                        final Runnable action = actions.take();
                        invokeLater( () -> action.run() );
                    }
                }
                catch (InterruptedException exc) {
                    LOG.fine("The dispatcher task is stopped.");
                }
            },
            0, 10, TimeUnit.MILLISECONDS);
    }

    ////////// Implementation of PageActions //////////

    @Override
    public void takeExposure(double exposureTime) {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.takeExposure(exposureTime);});
    }

    @Override
    public void showTrajectory(MotionDone motion) {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.showTrajectory(motion);});
    }

    @Override
    public void showWorkerIsUnreachable(String message) {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.showWorkerIsUnreachable(message);});
    }

    @Override
    public void showWorkerIsReachable() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.showWorkerIsReachable();});
    }

    @Override
    public void showStatus(ShutterStatus status) {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.showStatus(status);});
    }

    @Override
    public void showStateBundle(PhysicalState phys, SoftwareState soft) {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.showStateBundle(phys, soft);});
    }

    @Override
    public void openShutter() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.openShutter();});
    }

    @Override
    public void disconnect() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.disconnect();});
    }

    @Override
    public void connect() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.connect();});
    }

    @Override
    public void closeShutter() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.closeShutter();});
    }

    @Override
    public void changeBrakeState(final ShutterSide side, final boolean brakeEngaged) {
        actions.add(() -> {
            for (final PageActions rcv: receivers) rcv.changeBrakeState(side, brakeEngaged);});
    }

    @Override
    public void stopMotion() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.stopMotion();});
    }

    @Override
    public void resync() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.resync();});
    }

    @Override
    public void prodMode() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.prodMode();});
    }

    @Override
    public void center() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.center();});
    }

    @Override
    public void calibrate() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.calibrate();});
    }

    @Override
    public void toggleSafetyCheck() {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.toggleSafetyCheck();});
    }

    @Override
    public void setWorkerName(final String workerName) {
        actions.add(() -> {for (final PageActions rcv: receivers) rcv.setWorkerName(workerName);});
    }

}
