package org.lsst.ccs.subsystem.shutter.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import static java.awt.Font.BOLD;
import static java.awt.Font.ITALIC;
import java.awt.GridLayout;
import java.util.function.Function;
import static java.util.stream.Collectors.joining;
import static javax.swing.BorderFactory.createCompoundBorder;
import static javax.swing.BorderFactory.createEmptyBorder;
import static javax.swing.BorderFactory.createLineBorder;
import static javax.swing.BorderFactory.createTitledBorder;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import static javax.swing.SwingUtilities.invokeLater;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import static org.lsst.ccs.subsystem.shutter.gui.GUIUtil.exampleShutterStatus;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;

/**
 * Displays the latest status information not related to blade sets. Not thread safe, use from the
 * event dispatch thread.
 * @author tether
 */
public class GeneralStatusPanel extends JPanel implements PluginActions {
    
    public GeneralStatusPanel() {
        this.setLayout(new GridLayout(1 + Row.values().length, 2));
        
        final Function<String, JLabel> newHeading = (final String text) -> {
            final JLabel label = new JLabel(text);
            label.setFont(label.getFont().deriveFont(BOLD | ITALIC));
            label.setHorizontalAlignment(SwingConstants.CENTER);
            return label;
            
        };
        
        this.add(newHeading.apply("Parameter"));
        this.add(newHeading.apply("Value"));
        
        for (final Row r: Row.values()) {
            this.add(r.rowName);
            this.add(r.value);
        }
        
        final TitledBorder border = createTitledBorder(createLineBorder(Color.black, 1), "General status");
        border.setTitleFont(border.getTitleFont().deriveFont(BOLD));
        border.setTitleJustification(TitledBorder.CENTER);
        this.setBorder(
            createCompoundBorder(
                border,
                createEmptyBorder(5, 5, 5, 5))  // Inset the interior elements slightly.
        );
    }

    @Override
    public void showStatus(final ShutterStatus status) {
        Row.Calibrated.setValue(status.isCalibrated() ? "YES" : "NO");
        Row.SafetyChecks.setValue(status.isSafetyOn() ? "ON" : "OFF");
        Row.RTDList.setValue(
            status.getTemperature().stream()
            .map((Integer i) -> i.toString())
            .collect(joining(", "))
        );
    }

    @Override
    public void showWorkerIsUnreachable(final String message) {
        for (final Row r: Row.values()) {r.clearValue();}
        Row.SubsysVisible.setValue("NO");
    }

    @Override
    public void showStateBundle(PhysicalState phys, SoftwareState soft) {
        Row.PhysState.setValue(phys);
        Row.SoftState.setValue(soft.getDescription());
    }

    @Override
    public void showWorkerIsReachable() {
        Row.SubsysVisible.setValue("YES");
    }

    private enum Row {
        SubsysVisible("Subsystem visible?", "%s"),
        PhysState("Physical state", "%s"),
        SoftState("Subsystem state", "%s"),
        Calibrated("Calibrated?", "%s"),
        SafetyChecks("Safety checks", "%s"),
        RTDList("RTD readings", "%s");
        
        public final JLabel rowName;
        public final String format;
        public final JTextField value;
        
        Row(final String rowName, final String format) {
            this.rowName = new JLabel(rowName);
            this.format = format;
            this.value = new JTextField();
            value.setEditable(false);
        }
        
        public void setValue(final Object v) {
            value.setText(String.format(format, v));
        }
        
        public void clearValue() {
            value.setText("");
        }
    }
    
    public static void main(final String[] args) {
        invokeLater( () -> {
            final JFrame frame = new JFrame();
            final GeneralStatusPanel panel = new GeneralStatusPanel();
            frame.getContentPane().add(panel, BorderLayout.CENTER);            
            panel.showStatus(exampleShutterStatus());
            panel.showStateBundle(PhysicalState.CLOSED, SoftwareState.CLOSED);
            panel.showWorkerIsReachable();
            frame.pack();
            frame.setVisible(true);

        });
    }
}
