package org.lsst.ccs.subsystem.shutter.gui;

import java.awt.BorderLayout;
import java.time.Duration;
import java.util.EnumMap;
import java.util.Map;
import java.util.logging.Logger;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import static javax.swing.SwingUtilities.invokeLater;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import org.lsst.ccs.subsystem.shutter.gui.BladesDisplay.BladeSet;
import org.lsst.ccs.subsystem.shutter.sim.CubicSCurve;
import org.lsst.ccs.subsystem.shutter.sim.PredictedTrajectory;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus.AxisStatus;

/**
 * Owns and updates the plot data, the plot pane and the shutter animation.
 * This class is not thread safe and its
 * methods should be called from the Swing dispatch thread.
 *
 * @author azemoon
 * @author tether
 */
public class DisplayPanel extends JPanel implements PluginActions {
    private static final Logger LOG = Logger.getLogger(DisplayPanel.class.getName());

    private final BladesDisplay assembly;
    private final TrajectoryDisplay plotPanel;
    private final Map<ShutterSide, PredictionPlotData> prediction = new EnumMap<>(ShutterSide.class);
    private final Map<ShutterSide, HallPlotData> hallData = new EnumMap<>(ShutterSide.class);
    private final Map<ShutterSide, EncoderPlotData> encoderData = new EnumMap<>(ShutterSide.class);

    public DisplayPanel() {
        super(new BorderLayout());
        assembly = new BladesDisplay();
        prediction.put(PLUSX, new PredictionPlotData());
        prediction.put(MINUSX, new PredictionPlotData());
        hallData.put(PLUSX, new HallPlotData());
        hallData.put(MINUSX, new HallPlotData());
        encoderData.put(PLUSX, new EncoderPlotData());
        encoderData.put(MINUSX, new EncoderPlotData());
        plotPanel = new TrajectoryDisplay(prediction, hallData, encoderData);

        JSplitPane split = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        // The JSplitPane will give any extra space to the rightmost element in a horizontal split unless
        // the user overrides it by moving the divider. We want that element to be the graph so we
        // add it last. Sort of a poor man's GridBag.
        split.add(assembly);
        split.add(plotPanel);
        add(split, BorderLayout.CENTER);
    }

    /**
     * Sets the blade set positions in the shutter assembly drawing, makes sure that
     * it's visible and repaints it.
     *
     * @param plus the normalized position of blade set PLUSX
     * @param minus the normalized position of blade set MINUSX
     */
    public void setBladeSetPositions(final double plus, final double minus) {
        assembly.getShutter(PLUSX).setPosition((float) plus);
        assembly.getShutter(MINUSX).setPosition((float) minus);
        assembly.setVisible(true);
        assembly.repaint();
    }

    /**
     * Hides the shutter assembly picture. This is done at startup before we know the
     * shutter positions.
     */
    public void hideAssembly() {
        assembly.setVisible(false);
        assembly.repaint();
    }

    Map<ShutterSide, PredictionPlotData> getHistories() {
        return prediction;
    }

    private final static int PREDICTION_SAMPLES = 201;

    /**
     * Plots the prediction for a single blade set movement.
     * @param motion the motion history.
     */
    public void showPredictedTrajectory(final MotionDone motion) {
        // Run the simulation.
        final double moveTime = motion.targetDuration().toNanos() * 1e-9;
        final Duration dt = motion.targetDuration().dividedBy(PREDICTION_SAMPLES - 1);
        final double startPos = motion.side().normalizePosition(motion.startPosition());
        final double targetPos = motion.side().normalizePosition(motion.targetPosition());
        final PredictedTrajectory motor = new PredictedTrajectory(
            startPos,
            motion.startTime(),
            dt,
            PREDICTION_SAMPLES,
            new CubicSCurve(
                targetPos - startPos,
                moveTime)
        );
        final PredictionPlotData plotData = prediction.get(motion.side());
        plotData.reset();
        prediction.get(motion.side().opposite()).reset();
        plotData.addData(motor.getPositions(), motion.startTime());
        final int n = plotData.getNPoints();
        plotPanel.setTimePeriod((float)moveTime);
    }

    /**
     * Updates the plot pane and the shutter assembly drawing based on a movement in the
     * actual shutter. Plot data includes both blade positions and Hall transitions.
     *
     * @param motion the movement history collected at the shutter
     */
    void showActualTrajectory(final MotionDone motion) {
        showEncoderSamples(motion);
        showActualHallTransitions(motion);
    }

    /**
     * Adds actual blade positions to the plot and moves the corresponding blade set picture.
     * @param motion the history of the motion
     */
    private void showEncoderSamples(final MotionDone motion) {
        final BladeSet blades = assembly.getShutter(motion.side());
        final float newPos = (float)motion.side().normalizePosition(motion.endPosition());
        blades.setPosition(newPos);
        assembly.repaint();
        encoderData.get(motion.side()).reset();
        encoderData.get(motion.side().opposite()).reset();
        encoderData.get(motion.side()).addData(motion.side(), motion.encoderSamples(), motion.startTime());
    }

    /**
     * Adds Hall transitions to the plot pane.
     *
     * @param motion the history of the motion.
     */
    void showActualHallTransitions(final MotionDone motion) {
        hallData.get(motion.side()).reset();
        hallData.get(motion.side().opposite()).reset();
        hallData.get(motion.side()).addData(motion.side(), motion.hallTransitions(), motion.startTime());
    }

    ////////// Implementation of PluginActions //////////

    @Override
    public void showTrajectory(MotionDone motion) {
        LOG.fine("showTrajectory()");
        if (!motion.isDummy()) {
            invokeLater( () -> showPredictedTrajectory(motion) );
            invokeLater( () -> showActualTrajectory(motion) );
        }
    }

    @Override
    public void showStatus(final ShutterStatus status) {
        LOG.fine("showStatus()");
        for (final ShutterSide side: ShutterSide.values()) {
           final AxisStatus axis = status.getAxisStatus(side);
           invokeLater( () ->
               assembly
               .getShutter(side)
               .setPosition((float)side.normalizePosition(axis.getActPos())));
        }
        invokeLater( () -> assembly.repaint() );
    }

}
