package org.lsst.ccs.subsystem.shutter.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import static java.awt.Font.BOLD;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.util.EnumMap;
import java.util.Map;
import java.util.function.Consumer;
import java.util.function.Supplier;
import static javax.swing.BorderFactory.createCompoundBorder;
import static javax.swing.BorderFactory.createEmptyBorder;
import static javax.swing.BorderFactory.createLineBorder;
import static javax.swing.BorderFactory.createTitledBorder;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import static javax.swing.SwingUtilities.invokeLater;
import javax.swing.border.TitledBorder;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.MINUSX;
import static org.lsst.ccs.subsystem.shutter.common.ShutterSide.PLUSX;
import static org.lsst.ccs.subsystem.shutter.gui.GUIUtil.fakeDispatcher;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;

/**
 * Contains all the command buttons and any parameter selection controls they may have. Not thread-safe, use
 * only from the event dispatch thread.
 * @author tether
 */
public class CommandPanel extends JPanel implements PluginActions {
    private final PluginActions dispatcher;
    private final Map<ShutterSide, Boolean> brakeState;
    private final JSpinner exposureTimeSpinner;

    public CommandPanel(final PluginActions dispatcher) {
        this.dispatcher = dispatcher;
        this.brakeState = new EnumMap<>(ShutterSide.class);
        this.exposureTimeSpinner = new JSpinner();
        exposureTimeSpinner.setModel(
            // Exposure times in seconds.
            new SpinnerNumberModel(1.0 /*current*/, 0.1 /*min*/, 3600.0 /*max*/, 0.1 /*step*/)
        );
        
        this.setLayout(new GridLayout(9, 2));
        
        final Supplier<Box> filler = () -> {
            final Box result = new Box(BoxLayout.LINE_AXIS);
            result.add(Box.createGlue());
            return result;
        };
        
        // Row 0.
        this.add(Cmd.Stop.button);
        this.add(filler.get());
        
        // Row 1 is blank.
        this.add(filler.get());
        this.add(filler.get());
        
        // Row 2.
        this.add(Cmd.Expose.button);
        final Box timeBox = new Box(BoxLayout.LINE_AXIS);
        timeBox.add(new JLabel("for"));
        timeBox.add(exposureTimeSpinner);
        timeBox.add(new JLabel("seconds"));
        timeBox.add(Box.createGlue());
        this.add(timeBox);
        
        // Row 3.
        this.add(Cmd.Open.button);
        this.add(Cmd.Close.button);
        
        // Row 4 is blank.
        this.add(filler.get());
        this.add(filler.get());
        
        // Row 5.
        this.add(Cmd.Calibrate.button);
        this.add(Cmd.GotoProd.button);
        
        // Row 6.
        this.add(Cmd.Resync.button);
        this.add(Cmd.Center.button);
        
        // Row 7.
        this.add(Cmd.TogglePlusXBrake.button);
        this.add(Cmd.ToggleMinusXBrake.button);
        
        // Row 8.
        this.add(Cmd.ToggleSafety.button);
        this.add(filler.get());
        
            
        final TitledBorder border = createTitledBorder(createLineBorder(Color.black, 1), "Commands");
        border.setTitleFont(border.getTitleFont().deriveFont(BOLD));
        border.setTitleJustification(TitledBorder.CENTER);
        this.setBorder(
            createCompoundBorder(
                border,
                createEmptyBorder(5, 5, 5, 5))  // Inset the interior elements slightly.
        );
    
        
    }
    
    /**
     * Gets the dispatcher for plugin actions.
     * @return The dispatcher.
     */
    public PluginActions getDispatcher() {return dispatcher;}
    
    /**
     * Gets the exposure time from the spinner.
     * @return The exposure time in seconds.
     */
    public double getExposureTime() {return ( (Number)exposureTimeSpinner.getValue() ).doubleValue();}
    
    /**
     * Reverses the current state of the brake for the given side but only if we know what that state is.
     * @param side the side whose brake state is to be changed.
     */
    public void toggleBrakeState(final ShutterSide side) {
        final Boolean state = brakeState.get(side);
        if (state != null) {
            // We've had a report on the current brake state.
            getDispatcher().changeBrakeState(side, !state);
        }
    }

    @Override
    public void showStatus(final ShutterStatus status) {
        for (final ShutterSide side: ShutterSide.values()) {
            brakeState.put(side, status.getAxisStatus(side).isBrakeEngaged());
        }
    }

    @Override
    public void showWorkerIsReachable() {
        exposureTimeSpinner.setEnabled(true);
        for (final Cmd command: Cmd.values()) {command.button.setEnabled(true);}
    }

    @Override
    public void showWorkerIsUnreachable(String message) {
        brakeState.clear();
        exposureTimeSpinner.setEnabled(false);
        for (final Cmd command: Cmd.values()) {command.button.setEnabled(false);}
    }

    private enum Cmd {
        Stop("STOP",              panel -> panel.getDispatcher().stopMotion()),
        Expose("Take Exposure",   panel -> panel.getDispatcher().takeExposure(panel.getExposureTime())),
        Open("Open",              panel -> panel.getDispatcher().openShutter()),
        Close("Close",            panel -> panel.getDispatcher().closeShutter()),
        Calibrate("Calibrate",    panel -> panel.getDispatcher().calibrate()),
        GotoProd("Prod mode",     panel -> panel.getDispatcher().prodMode()),
        Resync("Resync",          panel -> panel.getDispatcher().resync()),
        Center("Center",          panel -> panel.getDispatcher().center()),
        ToggleSafety
              ("Toggle safety",   panel -> panel.getDispatcher().toggleSafetyCheck()),
        TogglePlusXBrake
              ("Toggle +X brake", panel ->panel.toggleBrakeState(PLUSX)),
        ToggleMinusXBrake
              ("Toggle -X brake", panel -> panel.toggleBrakeState(MINUSX));
        
        static {
            Stop.button.setBackground(Color.red);
            Stop.button.setOpaque(true);          // Needed with some look-and-feels, MacOS for one,  ...
            Stop.button.setBorderPainted(false);  // ... to change the color inside the button.
            Stop.button.setForeground(Color.white);
            Stop.button.setFont(Stop.button.getFont().deriveFont(BOLD));
        }
        
        public final JButton button;
        public final Consumer<CommandPanel> handler;

        private Cmd(final String cmdName, final Consumer<CommandPanel> handler) {
            this.button = new JButton(cmdName);
            this.handler = handler;
            button.addActionListener(this::dispatch);
        }
        
        private void dispatch(final ActionEvent evt) {
            final CommandPanel panel = (CommandPanel)button.getParent();
            handler.accept(panel);
        }
    }
    
    public static void main(final String[] args) {
        invokeLater( () -> {
            final JFrame frame = new JFrame();
            final CommandPanel panel = new CommandPanel(fakeDispatcher());
            frame.getContentPane().add(panel, BorderLayout.CENTER);
            panel.showStatus(GUIUtil.exampleShutterStatus());
            panel.showWorkerIsReachable();
            frame.pack();
            frame.setVisible(true);
        });
    }
}
