package org.lsst.ccs.subsystem.shutter.gui;

import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.status.MotionDone;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;

/**
 * All the actions that the may be performed by the shutter GUI on itself, the CCS status
 * bus, the shutter worker, etc.
 * @author tether
 */
public interface PluginActions {

    /**
     * Command the shutter worker to start a calibration.
     */
    default void calibrate() {}

    /**
     * Command the shutter to toggle the setting of the safety check flag.
     */
    default void toggleSafetyCheck() {}

    /**
     * Command the shutter worker to start centering the blade sets.
     */
    default void center() {}

    /**
     * Command the shutter to engage or release one of the blade set brakes.
     * @param side The blade set to be affected.
     * @param brakeEngaged True if the brake is to be engaged, else false.
     */
    default void changeBrakeState(ShutterSide side, boolean brakeEngaged) {}

    /**
     * Command the shutter worker to start closing the shutter.
     */
    default void closeShutter() {}

    /**
     * Connect to the CCS buses and check that the worker is online..
     */
    default void connect() {}

    /**
     * Disconnect from the CCS buses.
     */
    default void disconnect() {}

    /**
     * Command the shutter worker to go into production mode.
     */
    default void prodMode() {}

    /**
     * Command the shutter worker to start opening the shutter.
     */
    default void openShutter() {}

    /**
     * Command the shutter worker to resynchronize its state with that
     * of the shutter controller.
     */
    default void resync() {}

    /**
       Show that the shutter worker accepted the last command.
     */
    default void showCommandSuccess() {}

    /**
     * Show that the last attempt to command the shutter worker failed.
     * @param message A short explanation of the failure..
     */
    default void showCommandFailure(String message) {}

    /**
     * Update the display of worker state bundle elements.
     * @param phys The PhysicalState value to display.
     * @param soft The SoftwareState value to display.
     */
    default void showStateBundle(PhysicalState phys, SoftwareState soft) {}

    /**
     * Update displays according to the latest {@code ShutterStatus} message received
     * @param status The {@code ShutterStatus} message.
     */
    default void showStatus(ShutterStatus status) {}

    /**
     * Show that we're connected to the buses and that the worker is online.
     */
    default void showWorkerIsReachable() {}

    /**
     * Show that the worker subsystem can't be communicated with. It could be
     * that we can't connect to the buses, that the worker isn't online or that
     * the worker is unresponsive..
     * @param message A short message describing the problem.
     */
    default void showWorkerIsUnreachable(String message) {}

    /**
     * Update displays using the latest {@code MotionDone} message received.
     * @param motion The {@code MotionDone} message.
     */
    default void showTrajectory(MotionDone motion) {}

    /**
     * Command the shutter worker to stop all blade set motion and reset the
     * controller.
     */
    default void stopMotion() {}

    /**
     * Command the shutter worker to start taking an exposure.
     * @param exposureTime The exposure time in seconds.
     */
    default void takeExposure(double exposureTime){}

}
