package org.lsst.ccs.subsystem.shutter.gui;

import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JScrollPane;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelEvent;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.subsystem.shutter.common.Constants;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.scheduler.Scheduler;

/**
 * A CCS graphical console plugin that defines a menu item for the shutter in the subsystems menu
 * and which displays the shutter GUI in a panel provided by the console.
 * @author tether
 */
@Plugin(name="LSST Shutter plugin", description="Camera shutter status and controls.")
public class ShutterPlugin
    extends org.lsst.ccs.gconsole.base.ConsolePlugin
    implements AgentPresenceListener
{
    private static final Logger LOG = Logger.getLogger(ShutterPlugin.class.getPackage().getName());

    private static final String CCS_SUBSYSTEM_MENU_NAME = "CCS Subsystems";
    private static final String CCS_SHUTTER_MENU_ITEM_NAME = "Camera shutter";

    private final DisplayPanel dispPanel;
    private final ControlPanel ctrlPanel;
    private final RootPanel rootPanel;
    private final JScrollPane scrollPane;
    private final MenuItemAction menuItem;
    private final Commander commander;
    private final StatusListener statusListener;
    private final Scheduler scheduler;
    private final Dispatcher dispat;
    
    public ShutterPlugin() {
        // Passing "this" to other constructors is OK so long as those
        // constructors don't expose it to threads other than the one
        // running this constructor. Ideally they should just
        // store the value in a private final field.
        this.dispPanel = new DisplayPanel();
        this.ctrlPanel = new ControlPanel(this);
        this.rootPanel = RootPanel.create(dispPanel, ctrlPanel);
        this.scrollPane = new JScrollPane(rootPanel);
        this.menuItem = new MenuItemAction();
        this.commander = new Commander(this);
        this.statusListener = new StatusListener(this);
        this.scheduler = new Scheduler("Shutter plugin dispatcher", 1);
        this.dispat = new Dispatcher(
            this,
            dispPanel, ctrlPanel, commander, statusListener);
    }

    public Scheduler getScheduler() {return scheduler;}

    public Dispatcher getDispatcher() {return dispat;}

    public String getMenuName() {return CCS_SUBSYSTEM_MENU_NAME;}

    public String getMenuItemName() {return CCS_SHUTTER_MENU_ITEM_NAME;}

    public String getWorkerName() {return Constants.WORKER_SUBSYSTEM_NAME;}

    ////////// Overrides for ShutterPlugin //////////

    /**
      * Enables controls to activate or deactivate the GUI. One control is an item named
      * "Shutter" in the standard subsystems menu. The item has a check box next to the
      * item name which tells whether the subsystem GUI is active. The other control
      * is the close box on the tab for the panel inside which the GUI is displayed.
     */
    @Override
    public void initialize() {
        LOG.fine("Initializing plugin.");
        getServices().addMenu(menuItem, getMenuName());
    }

    @Override
    public void start() {
        LOG.fine("Starting plugin.");
        getDispatcher().start();
        getDispatcher().showWorkerIsUnreachable("");
    }

    @Override
    public void stop() {
        LOG.fine("Stopping plugin.");
        if (menuItem.isSelected()) { // Fake the deselection of our menu item.
            menuItem.deselect();
            menuItem.actionPerformed(null);
        }
    }

    @Override
    public void shutdown() {
        LOG.fine("Shutting down plugin.");
        scheduler.shutdownNow();
    }

    // Event handlers. We get two basic kinds of events. The first kind is Action events generated
    // by the use our item in the CCS Subsystems menu. When these events occur the "selected"
    // flag in our action object will have been set properly for us before the event
    // is issued. The second kind is panel manager (PM) events that occur when our display is opened,
    // closed, becomes the front tab, another tab becomes the front, etc. The menu item checkbox
    // isn't set for us by these events. We need both kinds of handlers because the console
    // doesn't know that the menu item and the display are related and so won't use the menu
    // action when the tab is closed by some other means. There's also some
    // redundancy as calling open() or close() for our display will generate the PM events. So
    // we have to prevent double-closing for example (or at least make it harmless).
    //
    // Closing events are forwarded to the menu item action object for consistency.
    private void handlePanelEvent(final PanelEvent e) {
        if (
            e.hasKey(Panel.OPEN) &&
            (Boolean) e.getNewValue() == false)
        {
            LOG.fine("Panel Manager close event.");
            if (menuItem.isSelected()) {
                // The display should be deactivated even though the menu item wasn't used.
                // Fake a menu action event.
                menuItem.deselect();
                menuItem.actionPerformed(null);
            }
        }
    }

    /**
     * Activates or deactivates our display when our menu item selection state changes.
     */
    private class MenuItemAction extends AbstractAction {

        /**
         * Sets the name of the menu item and makes sure the menu item selection
         * state is false.
         */
        public MenuItemAction() {
            super(getMenuItemName());
            putValue(Action.SELECTED_KEY, false);
        }

        /**
         * Returns the current selection state of our menu item.
         * @return true if selected (check box is checked), else false.
         */
        public boolean isSelected() {
            return (Boolean)getValue(Action.SELECTED_KEY);
        }

        /**
         * Used by the PM event handler to change the menu item selection state.
         */
        public void deselect() {
            putValue(Action.SELECTED_KEY, false);
        }

        /**
         * Activate our display if our menu item is selected, else deactivate the display.
         * @param e Unused action event.
         */
        @Override
        public void actionPerformed(final ActionEvent e) {
            if (e == null) {
                LOG.fine("Fake menu item action: deselect");
            }
            else {
                LOG.fine("Menu item action: " + (isSelected() ? "select" : "deselect"));
            }
            // We're should become active if our check box is selected in the subsystems menu.
            if (isSelected()) {
                activate();
            }
            else {
                deactivate();
            }
        }

        /** Display the GUI via the console's panel manager, set up the
         *  controls and start monitoring the worker subsystem.
         */
        private void activate() {
            LOG.fine("Activating.");
            final PanelManager pm = getConsole().getPanelManager();
            // Display a tab with our GUI in it.
            pm.open(scrollPane, "Camera Shutter Control Panel");
            // Get closing events for our display. Can only use this component-specific
            // for of addListener() after opening our display.
            pm.addListener(e -> handlePanelEvent(e), scrollPane);
            // Start monitoring.
            getDispatcher().connect();
        }

        /**
         * Remove our GUI from the console's display
         */
        private void deactivate() {
            LOG.fine("Deactivating.");
            final PanelManager pm = getConsole().getPanelManager();
            // Make our GUI disappear. Generates PM closing events.
            pm.close(scrollPane);
            // Stop monitoring.
            getDispatcher().disconnect();
        }
    }
    
}
