package org.lsst.ccs.subsystem.shutter.gui;

import java.util.EnumMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.swing.JTextField;

import static javax.swing.SwingUtilities.invokeLater;
import static org.lsst.ccs.subsystem.shutter.common.Constants.BLADE_SET_HOME;
import static org.lsst.ccs.subsystem.shutter.common.Constants.MAX_STROKE_LENGTH;

import org.lsst.ccs.subsystem.shutter.common.PhysicalState;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.subsystem.shutter.common.SoftwareState;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus;
import org.lsst.ccs.subsystem.shutter.status.ShutterStatus.AxisStatus;

/**
 * Provides various controls for the shutter subsystem. This class is not thread safe
 * and its methods should be called from the Swing event dispatch thread, via invokeLater() for example.
 * @author tonyj
 * @author tether
 */
public class ControlPanel extends javax.swing.JPanel implements PluginActions {
    private static final Logger LOG = Logger.getLogger(ControlPanel.class.getName());

    final ShutterPlugin plugin;
    final Map<ShutterSide, Boolean> brakeIsEngaged;
    final Map<ShutterSide, JTextField> brakeField;
    final Map<ShutterSide, JTextField> positionField;
    final Map<ShutterSide, JTextField> temperatureField;

    /**
     * Saves the plugin reference for later use and initializes components of this panel.
     * @param plugin The console plugin being used to run the GUI.
     */
    public ControlPanel(final ShutterPlugin plugin) {
        this.plugin = plugin;
        initComponents();
        this.brakeIsEngaged = new EnumMap<>(ShutterSide.class);
        this.brakeIsEngaged.put(ShutterSide.PLUSX, null);
        this.brakeIsEngaged.put(ShutterSide.MINUSX, null);
        this.brakeField = new EnumMap<>(ShutterSide.class);
        this.brakeField.put(ShutterSide.PLUSX, plusXBrakeField);
        this.brakeField.put(ShutterSide.MINUSX, minusXBrakeField);
        this.positionField = new EnumMap<>(ShutterSide.class);
        this.positionField.put(ShutterSide.PLUSX, plusXPositionField);
        this.positionField.put(ShutterSide.MINUSX, minusXPositionField);
        this.temperatureField = new EnumMap<>(ShutterSide.class);
        this.temperatureField.put(ShutterSide.PLUSX, plusXTemperatureField);
        this.temperatureField.put(ShutterSide.MINUSX, minusXTemperatureField);
    }

    ////////// Implementation of PluginActions //////////
    @Override
    public void showCommandFailure(String message) {
        LOG.log(Level.FINE, "showCommandFailure(): {0}", message);
    }

    @Override
    public void showCommandSuccess() {
        LOG.fine("showCommandSuccess()");
    }

    @Override
    public void showStateBundle(final PhysicalState phys, SoftwareState soft) {
        LOG.log(Level.FINE, "Physical state is {0}, software state is {1}.", new Object[]{phys, soft});
        invokeLater( () -> {
            shutterStateField.setText(phys.toString());
            ccsStateField.setText(soft.getDescription());
        });
    }

    @Override
    public void showStatus(final ShutterStatus status) {
        invokeLater(() -> {
            calibrationStateField.setText(status.isCalibrated() ? "YES" : "NO");
            safetyChecksField.setText(status.isSafetyOn() ? "ON" : "OFF");
            rtdField.setText(
                    status.getTemperature().stream()
                    .map((Integer i) -> i.toString())
                    .collect(Collectors.joining(", "))
            );
            for (final ShutterSide side: ShutterSide.values()) {
                final AxisStatus axstatus = status.getAxisStatus(side);
                brakeIsEngaged.put(side, axstatus.isBrakeEngaged());
                brakeField.get(side).setText(brakeIsEngaged.get(side) ? "ENGAGED" : "RELEASED");
                final double pos = axstatus.getActPos();
                final double home = BLADE_SET_HOME.get(side);
                final double percent = 100.0 * Math.abs(pos - home) / MAX_STROKE_LENGTH;
                final String strpos = String.format("% 5.1f mm = % 5.1f%%", pos, percent);
                positionField.get(side).setText(strpos);
                final String strtemp = String.format("% 5.1f \u2103", axstatus.getMotorTemp()); // u2103 is degrees-C sign.
                temperatureField.get(side).setText(strtemp);
            }
        });
    }

    @Override
    public void showWorkerIsReachable() {
        invokeLater( () -> setEnabled(true));
    }

    @Override
    public void showWorkerIsUnreachable(final String message) {
        LOG.fine(message);
        invokeLater(() -> setEnabled(false));
    }

    ////////// End of PluginActions implementation //////////


    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        takeExposureButton = new javax.swing.JButton();
        exposureTimeLabel = new javax.swing.JLabel();
        exposureTimeSpinner = new javax.swing.JSpinner();
        secondsLabel = new javax.swing.JLabel();
        closeButton = new javax.swing.JButton();
        openButton = new javax.swing.JButton();
        jLabel1 = new javax.swing.JLabel();
        shutterStateField = new javax.swing.JTextField();
        stopButton = new javax.swing.JButton();
        calibrateButton = new javax.swing.JButton();
        resyncButton = new javax.swing.JButton();
        centerButton = new javax.swing.JButton();
        prodModeButton = new javax.swing.JButton();
        jSeparator2 = new javax.swing.JSeparator();
        jSeparator3 = new javax.swing.JSeparator();
        plusXBrakeButton = new javax.swing.JButton();
        plusXBrakeField = new javax.swing.JTextField();
        minusXBrakeButton = new javax.swing.JButton();
        minusXBrakeField = new javax.swing.JTextField();
        plusXPositionLabel = new javax.swing.JLabel();
        plusXPositionField = new javax.swing.JTextField();
        minusXPositionLabel = new javax.swing.JLabel();
        minusXPositionField = new javax.swing.JTextField();
        jLabel2 = new javax.swing.JLabel();
        ccsStateField = new javax.swing.JTextField();
        jLabel3 = new javax.swing.JLabel();
        minusXTemperatureField = new javax.swing.JTextField();
        jLabel4 = new javax.swing.JLabel();
        plusXTemperatureField = new javax.swing.JTextField();
        jLabel5 = new javax.swing.JLabel();
        calibrationStateField = new javax.swing.JTextField();
        jLabel6 = new javax.swing.JLabel();
        safetyChecksField = new javax.swing.JTextField();
        toggleSafetyButton = new javax.swing.JButton();
        jLabel7 = new javax.swing.JLabel();
        rtdField = new javax.swing.JTextField();

        takeExposureButton.setText("Take exposure");
        takeExposureButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleTakeExposure(evt);
            }
        });

        exposureTimeLabel.setText("Exposure time:");

        exposureTimeSpinner.setModel(new javax.swing.SpinnerNumberModel(Float.valueOf(1.0f), Float.valueOf(0.1f), Float.valueOf(3600.0f), Float.valueOf(0.1f)));

        secondsLabel.setText("seconds");

        closeButton.setText("Close");
        closeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleCloseShutter(evt);
            }
        });

        openButton.setText("Open");
        openButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleOpenShutter(evt);
            }
        });

        jLabel1.setText("Shutter state:");

        shutterStateField.setEditable(false);
        shutterStateField.setHorizontalAlignment(javax.swing.JTextField.LEFT);

        stopButton.setBackground(new java.awt.Color(255, 0, 51));
        stopButton.setText("Stop/reset");
        stopButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleStop(evt);
            }
        });

        calibrateButton.setText("Calibrate");
        calibrateButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleCalibrate(evt);
            }
        });

        resyncButton.setText("Resync");
        resyncButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleResync(evt);
            }
        });

        centerButton.setText("Center");
        centerButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleCenter(evt);
            }
        });

        prodModeButton.setText("Prod mode");
        prodModeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleProdMode(evt);
            }
        });

        plusXBrakeButton.setText("+X brake");
        plusXBrakeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handlePlusXBrake(evt);
            }
        });

        plusXBrakeField.setEditable(false);

        minusXBrakeButton.setText("-X brake");
        minusXBrakeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                handleMinusXBrake(evt);
            }
        });

        minusXBrakeField.setEditable(false);

        plusXPositionLabel.setText("+X posn");

        plusXPositionField.setEditable(false);

        minusXPositionLabel.setText("-X posn");

        minusXPositionField.setEditable(false);

        jLabel2.setText("CCS state:");

        ccsStateField.setEditable(false);

        jLabel3.setText("-X temp");

        minusXTemperatureField.setEditable(false);

        jLabel4.setText("+X temp");

        plusXTemperatureField.setEditable(false);

        jLabel5.setText("Shutter calibrated?");

        calibrationStateField.setEditable(false);
        calibrationStateField.setText("      ");

        jLabel6.setText("Safety checks:");

        safetyChecksField.setEditable(false);
        safetyChecksField.setText("     ");

        toggleSafetyButton.setText("Toggle Safety Check");
        toggleSafetyButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                toggleSafetyCheck(evt);
            }
        });

        jLabel7.setText("RTD values:");

        rtdField.setEditable(false);
        rtdField.setText(" ");

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addComponent(jSeparator3)
                .addContainerGap())
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createSequentialGroup()
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addComponent(jSeparator2)
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(takeExposureButton)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(exposureTimeLabel)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(exposureTimeSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(secondsLabel)
                                .addGap(18, 18, 18)
                                .addComponent(closeButton)
                                .addGap(18, 18, 18)
                                .addComponent(openButton)
                                .addGap(18, 18, 18)
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(jLabel3)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(minusXTemperatureField))
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(minusXPositionLabel)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(minusXPositionField, javax.swing.GroupLayout.PREFERRED_SIZE, 151, javax.swing.GroupLayout.PREFERRED_SIZE)))
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(plusXPositionLabel)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(plusXPositionField, javax.swing.GroupLayout.PREFERRED_SIZE, 150, javax.swing.GroupLayout.PREFERRED_SIZE))
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(jLabel4)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(plusXTemperatureField)))
                                .addGap(0, 59, Short.MAX_VALUE))
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(jLabel1)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(shutterStateField, javax.swing.GroupLayout.PREFERRED_SIZE, 136, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(jLabel7)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(rtdField))
                                    .addGroup(layout.createSequentialGroup()
                                        .addComponent(jLabel2)
                                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                        .addComponent(ccsStateField, javax.swing.GroupLayout.PREFERRED_SIZE, 276, javax.swing.GroupLayout.PREFERRED_SIZE)))
                                .addGap(18, 18, 18)
                                .addComponent(jLabel5)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(calibrationStateField, javax.swing.GroupLayout.PREFERRED_SIZE, 69, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                .addComponent(stopButton)))
                        .addContainerGap())
                    .addGroup(layout.createSequentialGroup()
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(calibrateButton)
                                .addGap(18, 18, 18)
                                .addComponent(prodModeButton)
                                .addGap(18, 18, 18)
                                .addComponent(resyncButton)
                                .addGap(26, 26, 26)
                                .addComponent(centerButton)
                                .addGap(18, 18, 18)
                                .addComponent(toggleSafetyButton))
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(plusXBrakeButton)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(plusXBrakeField, javax.swing.GroupLayout.PREFERRED_SIZE, 112, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addGap(18, 18, 18)
                                .addComponent(minusXBrakeButton)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(minusXBrakeField, javax.swing.GroupLayout.PREFERRED_SIZE, 112, javax.swing.GroupLayout.PREFERRED_SIZE))
                            .addGroup(layout.createSequentialGroup()
                                .addComponent(jLabel6)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(safetyChecksField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)))
                        .addGap(0, 453, Short.MAX_VALUE))))
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel1)
                    .addComponent(shutterStateField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(stopButton, javax.swing.GroupLayout.PREFERRED_SIZE, 43, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel2)
                    .addComponent(ccsStateField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel5)
                    .addComponent(calibrationStateField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(jLabel6)
                        .addComponent(safetyChecksField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                    .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                        .addComponent(jLabel7)
                        .addComponent(rtdField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)))
                .addGap(18, 18, 18)
                .addComponent(jSeparator3, javax.swing.GroupLayout.PREFERRED_SIZE, 15, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(18, 18, 18)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(takeExposureButton)
                    .addComponent(exposureTimeLabel)
                    .addComponent(exposureTimeSpinner, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(secondsLabel, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(closeButton)
                    .addComponent(openButton)
                    .addComponent(minusXPositionLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 43, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(minusXPositionField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(plusXPositionLabel, javax.swing.GroupLayout.PREFERRED_SIZE, 18, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(plusXPositionField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(jLabel3)
                    .addComponent(minusXTemperatureField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(jLabel4)
                    .addComponent(plusXTemperatureField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addGap(4, 4, 4)
                .addComponent(jSeparator2, javax.swing.GroupLayout.PREFERRED_SIZE, 19, javax.swing.GroupLayout.PREFERRED_SIZE)
                .addGap(18, 18, 18)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(calibrateButton)
                    .addComponent(prodModeButton)
                    .addComponent(resyncButton)
                    .addComponent(centerButton)
                    .addComponent(toggleSafetyButton))
                .addGap(18, 18, 18)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                    .addComponent(plusXBrakeButton)
                    .addComponent(plusXBrakeField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                    .addComponent(minusXBrakeButton)
                    .addComponent(minusXBrakeField, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                .addContainerGap())
        );
    }// </editor-fold>//GEN-END:initComponents

    private void handleTakeExposure(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleTakeExposure
        plugin.getDispatcher().takeExposure(getExposureTime());
    }//GEN-LAST:event_handleTakeExposure

    private void handleCloseShutter(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleCloseShutter
        plugin.getDispatcher().closeShutter();
    }//GEN-LAST:event_handleCloseShutter

    private void handleOpenShutter(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleOpenShutter
        plugin.getDispatcher().openShutter();
    }//GEN-LAST:event_handleOpenShutter

    private void handleStop(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleStop
        plugin.getDispatcher().stopMotion();
    }//GEN-LAST:event_handleStop

    private void handleCalibrate(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleCalibrate
        plugin.getDispatcher().calibrate();
    }//GEN-LAST:event_handleCalibrate

    private void handleProdMode(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleProdMode
        plugin.getDispatcher().prodMode();
    }//GEN-LAST:event_handleProdMode

    private void handleResync(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleResync
        plugin.getDispatcher().resync();
    }//GEN-LAST:event_handleResync

    private void handleCenter(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleCenter
        plugin.getDispatcher().center();
    }//GEN-LAST:event_handleCenter

    private void handlePlusXBrake(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handlePlusXBrake
        final Boolean state = brakeIsEngaged.get(ShutterSide.PLUSX);
        if (state != null) {
            plugin.getDispatcher().changeBrakeState(ShutterSide.PLUSX, !state);
        }
    }//GEN-LAST:event_handlePlusXBrake

    private void handleMinusXBrake(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_handleMinusXBrake
        final Boolean state = brakeIsEngaged.get(ShutterSide.MINUSX);
        if (state != null) {
            plugin.getDispatcher().changeBrakeState(ShutterSide.MINUSX, !state);
        }
    }//GEN-LAST:event_handleMinusXBrake

    private void toggleSafetyCheck(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_toggleSafetyCheck
        plugin.getDispatcher().toggleSafetyCheck();
    }//GEN-LAST:event_toggleSafetyCheck

    public double getExposureTime() {
        return ((Number) exposureTimeSpinner.getValue()).doubleValue();
    }

    @Override
    public void setEnabled(boolean enabled) {
        takeExposureButton.setEnabled(enabled);
        exposureTimeSpinner.setEnabled(enabled);
        closeButton.setEnabled(enabled);
        openButton.setEnabled(enabled);
        calibrateButton.setEnabled(enabled);
        prodModeButton.setEnabled(enabled);
        resyncButton.setEnabled(enabled);
        centerButton.setEnabled(enabled);
        stopButton.setEnabled(enabled);
        plusXBrakeButton.setEnabled(enabled);
        minusXBrakeButton.setEnabled(enabled);
        toggleSafetyButton.setEnabled(enabled);
        if (! enabled) {
            shutterStateField.setText("");
            ccsStateField.setText("");
            calibrationStateField.setText("");
            rtdField.setText("");
            for (final ShutterSide side: ShutterSide.values()) {
                brakeField.get(side).setText("");
                positionField.get(side).setText("");
                temperatureField.get(side).setText("");
            }
        }
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton calibrateButton;
    private javax.swing.JTextField calibrationStateField;
    private javax.swing.JTextField ccsStateField;
    private javax.swing.JButton centerButton;
    private javax.swing.JButton closeButton;
    private javax.swing.JLabel exposureTimeLabel;
    private javax.swing.JSpinner exposureTimeSpinner;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JLabel jLabel3;
    private javax.swing.JLabel jLabel4;
    private javax.swing.JLabel jLabel5;
    private javax.swing.JLabel jLabel6;
    private javax.swing.JLabel jLabel7;
    private javax.swing.JSeparator jSeparator2;
    private javax.swing.JSeparator jSeparator3;
    private javax.swing.JButton minusXBrakeButton;
    private javax.swing.JTextField minusXBrakeField;
    private javax.swing.JTextField minusXPositionField;
    private javax.swing.JLabel minusXPositionLabel;
    private javax.swing.JTextField minusXTemperatureField;
    private javax.swing.JButton openButton;
    private javax.swing.JButton plusXBrakeButton;
    private javax.swing.JTextField plusXBrakeField;
    private javax.swing.JTextField plusXPositionField;
    private javax.swing.JLabel plusXPositionLabel;
    private javax.swing.JTextField plusXTemperatureField;
    private javax.swing.JButton prodModeButton;
    private javax.swing.JButton resyncButton;
    private javax.swing.JTextField rtdField;
    private javax.swing.JTextField safetyChecksField;
    private javax.swing.JLabel secondsLabel;
    private javax.swing.JTextField shutterStateField;
    private javax.swing.JButton stopButton;
    private javax.swing.JButton takeExposureButton;
    private javax.swing.JButton toggleSafetyButton;
    // End of variables declaration//GEN-END:variables

}
