package org.lsst.ccs.subsystems.shutter.gui.jas;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.subsystems.shutter.gui.ShutterGUISubsystem;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Defines a CCS console plugin that displays the GUI for the camera shutter.
 * @author tether
 */
@Plugin(name="ShutterGUI", description="Camera shutter status display.")
public class ShutterPlugin extends ConsolePlugin implements AgentPresenceListener {

    private static final Logger LOG = Logger.getLogger(ShutterPlugin.class.getPackage().getName());

    private volatile ShutterGUISubsystem subsys;
    private volatile JPanel shutterPanel;
    private volatile JScrollPane scrollPane;
    private volatile PanelManager pm;

    ////////// Overrides for ConsolePlugin //////////

    /**
      * Display the GUI and make it ready to process status bus messages.
     */
    @Override
    public void initialize() {
        pm = getConsole().getPanelManager();
        Action openShutterGui = new AbstractAction("Shutter") {
            @Override
            public void actionPerformed(ActionEvent e) {
                if ( subsys == null ) {
                    subsys = new ShutterGUISubsystem();
                    shutterPanel = new JPanel();
                    shutterPanel.setLayout(new BorderLayout());
                    shutterPanel.add(subsys.getDisplayComponent(), BorderLayout.CENTER);
                    shutterPanel.add(subsys.getControlsComponent(), BorderLayout.SOUTH);
                    scrollPane = new JScrollPane(shutterPanel);
                    pm.open(scrollPane, "Camera Shutter Control Panel");
                    initTask();
                } else {
                    closeMonitorGUI();
                }
            }
        };

        pm.addListener(e -> {
            if (e.hasKey(Panel.OPEN) && (Boolean) e.getNewValue() == false) {
                if ( e.getSource().equals(scrollPane) ) {
                    closeMonitorGUI();
                    openShutterGui.putValue(Action.SELECTED_KEY, false);
                }
            }
        });
        openShutterGui.putValue(Action.SELECTED_KEY, false);
        getServices().addMenu(openShutterGui, "CCS Subsystems");

    }

    private void closeMonitorGUI() {
        pm.close(scrollPane);
        scrollPane = null;
        closeTask();
        subsys = null;
    }

    private void initTask() {
        subsys.initGui(Agent.getEnvironmentMessagingAccess());
        this
            .getConsole()
            .getMessagingAccess()
            .getAgentPresenceManager()
            .addAgentPresenceListener(this);
    }

    private void closeTask() {
        getConsole()
            .getMessagingAccess()
            .getAgentPresenceManager()
            .removeAgentPresenceListener(this);
    }

    /**
     * Does nothing.
     */
    @Override
    public void start() {
    }

    /**
     * Does nothing.
     */
    @Override
    public void stop() {
    }

    /**
     * Stop listening for bus membership changes.
     */
    @Override
    public void shutdown() {
        if ( subsys != null ) {
            closeTask();
        }
    }
    
    ////////// Implementation of AgentPresenceListener //////////

    /**
     * Enables the control panel when
     * the shutter subsystem has connected to the CCS bus.
     * @param agentArray Contains the information on the subsystems that connected.
     */
    @Override
    public void connected(AgentInfo... agentArray) {
        for (final AgentInfo agent: agentArray) {
            if (agent.hasAgentProperty("Foo")) {
                // TODO Enable the GUI.
            }
        }
    }

    /**
     * Disables the control panel and stops listening for command replies
     * the worker subsystem has disconnected from the CCS bus.
     * @param agent Contains the information on the subsystem that disconnected.
     */
    @Override
    public void disconnecting(AgentInfo agent) {
        if (agent.hasAgentProperty("foo")) {
            // TODO Disable the GUI.
        }
    }

    
}
