package org.lsst.ccs.subsystems.shutter.gui;

import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import static javax.swing.SwingUtilities.invokeLater;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.lang.reflect.InvocationTargetException;

import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;

import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;

import org.lsst.ccs.Agent;

import org.lsst.ccs.messaging.AgentMessagingLayer;
import org.lsst.ccs.messaging.CommandOriginator;
import org.lsst.ccs.messaging.StatusMessageListener;

import org.lsst.ccs.subsystems.shutter.status.*;

import org.lsst.ccs.subsystems.shutter.common.ShutterSide;


/**
 *
 * @author azemoon
 * @author tether
 */
public class ShutterGUISubsystem implements CommandOriginator, StatusMessageListener {

    static final String shutterWorkerName = "ccs-shutter";

    private AgentMessagingLayer msglayer;
    private final ShutterDisplay display;
    private final ShutterControlPanel control;
    
    private final ScheduledExecutorService pingerExec;
    private ScheduledFuture<?> pinger;

    public ShutterGUISubsystem() {
        display = new ShutterDisplay();
        control = new ShutterControlPanel();    
        pingerExec = Executors.newScheduledThreadPool(1);
}
    
    public static void main(String[] args) throws InterruptedException, InvocationTargetException {
        final Agent guiAgent = new Agent("Shutter GUI", AgentInfo.AgentType.LISTENER);
        final ShutterGUISubsystem shutterGUI = new ShutterGUISubsystem();
        // TODO guiAgent.start();
        shutterGUI.initGui(guiAgent.getMessagingAccess());
        final JFrame frame = new JFrame("Shutter GUI demo");
        invokeLater(() -> {
            frame.getRootPane().setDoubleBuffered(true);
            frame.getContentPane().add(shutterGUI.getDisplayComponent(),BorderLayout.CENTER);
            frame.getContentPane().add(shutterGUI.getControlsComponent(),BorderLayout.SOUTH);
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setResizable(true);
            frame.pack();
            frame.setVisible(true);
        });
    }

    /**
     * Sets up GUI action listeners, status bus listeners and starts attempting to get
     * the blade set positions from the worker subsystem.
     * @param aml the AgentMessagingLayer for this GUI subsystem
     * @throws InterruptedException
     * @throws java.lang.reflect.InvocationTargetException
     */
    public void initGui(final AgentMessagingLayer aml) {
        msglayer = aml;
        
        invokeLater(()->{
            // Until we get the current blade set positions, disable the controls
            // and don't show the shutter assembly drawing.
            control.setEnabled(false);
            //display.hideAssembly();
            // Connect actions to controls.
            setUpActionListeners(control);
        });
        
        // Ignore all status bus messages save those containing subsystem data from the shutter worker.
        // TODO final BusMessageFilter isFromShutter = BusMessageFilter.messageOrigin(shutterWorkerName);
        // TODO final BusMessageFilter isSubsystemData = BusMessageFilter.messageClass(StatusSubsystemData.class);
        //TODO msglayer
        //    .addStatusMessageListener(
        //        this,
        //        isFromShutter.and(isSubsystemData)
        //    );
        
        // Spawn a repeating task to ask the shutter worker for the current blade set positions. 
        pinger = pingerExec.scheduleWithFixedDelay(
            () -> {
               CommandRequest cmd = new CommandRequest(shutterWorkerName, "getBladeSetPositions");
               try {
                   aml.sendCommandRequest(cmd, this);
               } catch (RuntimeException exc) {
                   // Can't communicate with the shutter worker.
                   // Using System.out.printXXX() here causes a JAS hang.
               }
            },
            0, 5, TimeUnit.SECONDS);
    }
    
    public JComponent getDisplayComponent() {
        return display;
    }
    public JComponent getControlsComponent() {
        return control;
    }

    public void sendMoveToPosition(final ShutterSide side, final double targetPosition) {
        final Double arg2 = targetPosition;
        String command = "moveToPosition";

        CommandRequest cmd = new CommandRequest(shutterWorkerName, command, side, arg2);
        msglayer.sendCommandRequest(cmd, this);
    }

    public void sendTakeExposure(double exposureTime) {
        Double arg1 = exposureTime;
        String command = "takeExposure";

        CommandRequest cmd = new CommandRequest(shutterWorkerName, command, arg1);
        msglayer.sendCommandRequest(cmd, this);
    }

    public void sendCloseShutter() {
        CommandRequest cmd = new CommandRequest(shutterWorkerName, "closeShutter");
        msglayer.sendCommandRequest(cmd, this);
    }

    public void sendOpenShutter() {
        CommandRequest cmd = new CommandRequest(shutterWorkerName, "openShutter");
        msglayer.sendCommandRequest(cmd, this);
    }

    /**
     * Updates the GUI in response to status bus messages from the shutter worker.
     * @param s the status bus message
     */
    @Override
    public void onStatusMessage(StatusMessage s) {
        final StatusSubsystemData data = (StatusSubsystemData)s;
        if(data.getDataKey().equals(StatusKey.MOVEMENT.getKey())) {
            System.out.println("~_~_~_~_~_~MovementStatus_~_~_~_~_~_~_~_");
            invokeLater(()->display.showActualMovement((MovementHistoryStatus)extract(data)));
        } else if(data.getDataKey().equals(StatusKey.MOVE_BLADE_SET.getKey())) {
            System.out.println("~_~_~_~_~_~_~MoveBladeSetStatus_~_~_~_~_~_~_");
            invokeLater(() -> {
                control.setEnabled(false);
                display.showPredictedMoveTo((MoveToPositionStatus)extract(data));
            });
        } else if (data.getDataKey().equals(StatusKey.UNSAFE_MOVE.getKey())) {
            invokeLater(() ->
                JOptionPane.showMessageDialog(
                    display,
                    "Unsafe move",
                    "Operator error",
                    JOptionPane.ERROR_MESSAGE)
            );
        } else if (data.getDataKey().equals(StatusKey.TAKE_EXPOSURE.getKey())) {
            System.out.println("~_~_~_~_~_~_TakeExposureStatus~_~_~_~_~_~_~_");
            invokeLater(() -> {
                control.setEnabled(false);
                display.showPredictedExposure((TakeExposureStatus)extract(data));
            });
        } else if (data.getDataKey().equals(StatusKey.READY.getKey())) {
            System.out.println("~_~_~_~_~_~_ReadyForActionStatus~_~_~_~_~_~_~_");
            final ReadyForActionStatus status = (ReadyForActionStatus)extract(data);
            invokeLater(()->control.setEnabled(status.isReadyForAction()));
        } else if (data.getDataKey().equals(StatusKey.CLOSE_SHUTTER.getKey())) {
            System.out.println("~_~_~_~_~_~_CloseShutterStatus~_~_~_~_~_~_~_");
            invokeLater(()->{
                control.setEnabled(false);
                display.showPredictedClose((CloseShutterStatus)extract(data));
            });
        } else if (data.getDataKey().equals(StatusKey.OPEN_SHUTTER.getKey())) {
            System.out.println("~_~_~_~_~_~_OpenShutterStatus~_~_~_~_~_~_~_");
            invokeLater(()->{
                control.setEnabled(false);
                display.showPredictedOpen((OpenShutterStatus)extract(data));
            });
        }
        
    }
    
    private java.io.Serializable extract(StatusSubsystemData data) {
        final KeyValueData kval = (KeyValueData)data.getSubsystemData();
        return kval.getValue();
    }
    
    /**
     * Does nothing.
     * @param ack the CommandAck
     */
    @Override
    public void processAck(CommandAck ack) {}
    
    /**
     * Does nothing.
     * @param nack the CommandNack
     */
    @Override
    public void processNack(CommandNack nack) {}
    
    /**
     * Updates the GUI when we receive the current positions of the shutter blade sets.
     * @param res the CommandResult
     * 
     * When the positions arrive we give them to the shutter assembly drawing and make it visible.
     * We also cancel the ping task and shut down the executor service that runs it.
     */
    @Override
    public void processResult(CommandResult res) {
        final boolean ok = res.wasSuccessful();
        final Object obj = res.getResult();
        if (ok && (obj instanceof BladePositionResult)) {
            pinger.cancel(true);
            pingerExec.shutdown(); // Don't need its daemon thread any more.
            final BladePositionResult pos = (BladePositionResult)obj;
            invokeLater(()->{
                display.setBladeSetPositions(pos.getPlusx(), pos.getMinusx());
                display.repaint();
                control.setEnabled(true);
            });
        }
    }
    

    /**Binds actions to the shutter Swing controls.
     * @param control the panel of controls
     */
        public void setUpActionListeners(final ShutterControlPanel control) {
        control.addTakeExposureActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
               //Publish command: showPredictedExposure control.getMovementTime(), control.getExposureTime()
                System.out.println("ShutterGUI: sending TakeImage command");
                sendTakeExposure(control.getExposureTime());
                System.out.println("ShutterGUI: sent TakeImage command");
            }
        });
        
        control.addMoveToActionListener(new ActionListener() {

            @Override
            public void actionPerformed(ActionEvent e) {
               //Publish command: showPredictedMoveTo control.getBladeSetIndex(), control.getTargetPosition(), control.getMovementTime()
                System.out.println("ShutterGUI: sending MoveToPosition command");
                sendMoveToPosition(control.getSide(), control.getTargetPosition());
                System.out.println("ShutterGUI: sent MoveToPosition command");
            }
        });

        control.addCloseShutterActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
               //Publish command: showPredictedClose
                System.out.println("ShutterGUI: sending CloseShutter command");
                sendCloseShutter();
                System.out.println("ShutterGUI: sent CloseShutter command");
            }
        });

        control.addOpenShutterActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
               //Publish command: showPredictedOpen
                System.out.println("ShutterGUI: sending OpenShutter command");
                sendOpenShutter();
                System.out.println("ShutterGUI: sent OpenShutter command");
            }
        });

    }
}
