package org.lsst.ccs.subsystem.shutter.common;

import java.util.EnumSet;
import java.util.Set;

/**
 * Enumerates all the error conditions that may be signaled by an Error message from the PLC.
 * The Error message has a 32-bit mask with one bit for each possible error.
 * @author tether
 */
public enum PLCError {
    /** Hardware protection logic veto is in effect, can't move. */
    MOTION_VETO(0, "Hardware protection logic veto is in effect, can't move."),

    /** Axis brake set, can't move. */
    BRAKE_SET(1, "Axis brake is set, can't move."),

    /** Hall transition position and encoder position differ too much. */
    HALL_DIFF(2, "Hall transition position too far from the expected."),

    /** Exposure time was too short. */
    SHORT_EXPOSURE(3, "Exposure time too short."),

    /** Axis must be homed first. */
    NOT_HOMED(4, "Axis is not homed."),

    /** Hit motion limit near home position. */
    HOME_LIMIT(5, "Hit motion limit near axis home position."),

    /** Hit motion limit near the deployed (fully extended) position. */
    DEPLOYED_LIMIT(6, "Hit motion limit near axis deployed position."),

    /* Carrying out the motion would have caused the blade sets to collide. */
    UNSAFE_MOVE(7, "Command rejected, collision danger."),

    /* Requested a motion speed above the limit. */
    TOO_FAST(8, "Can't move that fast."),

    /** Can't move a disabled axis. */
    AXIS_DISABLED(9, "Axis is disabled, can't move.")
    ;

    private final int bitnum;
    private final String description;

    PLCError(final int bitnum, final String description) {
        this.bitnum = bitnum;
        this.description = description;
    }

    /**
     * Gets the bit position of this error in an int error mask.
     * @return The bit position (0-31).
     */
    public int getBitnum() {return bitnum;}

    /**
     * Get the description of the error.
     * @return The short description string.
     */
    public String getDescription() {return description;}

    /**
     * Create a set of {@code PLCError} containing all values whose bit is set in the error mask given.
     * @param mask Has a one bit for each error present.
     * @return The set of {@code PLCError}.
     */
    public static Set<PLCError> decodeMask(final int mask) {
        final Set<PLCError> errors = EnumSet.noneOf(PLCError.class);
        for (final PLCError err: PLCError.values()) {
            if ((mask & (1 << err.getBitnum())) != 0) {errors.add(err);}
        }
        return errors;
    }

    /**
     * Creates an int with one bit set for each element present in the given set of errors.
     * @param errors The set of errors.
     * @return The error bit-mask.
     */
    public static int encodeMask(final Set<PLCError> errors) {
        int mask = 0;
        for (final PLCError err: errors) {
            mask |= 1 << err.getBitnum();
        }
        return mask;
    }

    /**
     * Copy a set of {@code PLCError}. Gets around a shortcoming of {@code Enum.copyOf()} when the argument
     * isn't an {@code EnumSet}.
     * @param errors The set of errors to copy.
     * @return
     */
    public static Set<PLCError> copyOf(final Set<PLCError> errors) {
        return errors.isEmpty() ? EnumSet.noneOf(PLCError.class) : EnumSet.copyOf(errors);
    }
}
