package org.lsst.ccs.subsystem.shutter.status;

import java.io.Serializable;
import java.util.EnumMap;
import java.util.Map;
import java.util.TreeMap;
import org.lsst.ccs.bus.annotations.DataAttributes;
import org.lsst.ccs.subsystem.shutter.common.RTD;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Holds a ShutterStatus message suitable for trending. Immutable.
 * <p>
 * Note that booleans and enumerations are stored as ints since those data types were not
 * trendable at the time this code was written. For the same reason CCSTimeStamps are stored
 * as the double values gotten from {@link CCSTimeStamp#getTAIDouble()}.
 
 * @author tether
 */
public final class ShutterStatus implements Serializable {
    private static final long serialVersionUID = 7L;

    @DataAttributes(description = "Type of motion profile. 0 = S-curve.", units="unitless")
    private final int motionProfile;

    @DataAttributes(description = "Hall switch positions calibrated? 0 = no, 1 = yes.", units = "unitless")
    private final int isCalibrated;

    @DataAttributes(description = "The state of the shutter controller's state machine.", units = "unitless")
    private final int smState;

    private final Map<ShutterSide, ShutterAxisStatus> axstatus;

    @DataAttributes(description = "0 for shutter safety checks disable, 1 for enabled.", units = "unitless")
    private final int isSafetyOn;

    @DataAttributes(description = "RTD temperatures.", units = "Celsius")
    private final Map<RTD, Double> temperature;
    
    @DataAttributes(description = "RTD temperature in safe range? 0 = no, 1 = yes.", units = "unitless")
    private final Map<RTD, Integer> tempIsSafe; // Integer for trending.
    
    @DataAttributes(description = "Is brake power on? 0 = no, 1 = yes.", units = "unitless")
    private final int brakePowerIsOn;
    
    @DataAttributes(
        description = "The time at which this message was created or last updated."
        + " The output of CCSTimeStamp.getTAIDouble().",
        units = "s"
    )
    private final double creationTime;
    
    @DataAttributes(description = "The state of the PTP module.", units = "unitless")
    private final int ptpState;
    
    @DataAttributes(description = "The current number of leap seconds from the PTP module.", units = "count")
    private final int leapSeconds;
    
    @DataAttributes(description = "Is the PTP leap-second count valid? 0 = no, 1 = yes.", units = "unitless")
    private final int leapIsValid;

    /**
     * Constructs from scratch.
     * @param motionProfile The index number of the motion profile in effect.
     * @param isCalibrated Is the shutter properly calibrated?
     * @param smState The current state of the PLC state machine.
     * @param axes The source of the {@code ShutterAxisStatus} values to be stored in this message.
     * @param isSafetyOn Are shutter hardware safety checks enabled in the PLC?
     * @param temperature The RTD temperatures in degrees Celsius.
     * @param tempIsSafe The RTD temperature-in-safe-range flags.
     * @param brakePowerIsOn True if and only if 24V power is on as far as this subsystem knows.
     * @param creationTime The time at which this message was created.This will be the time
     * recorded in the original message from the PLC or the current system time if the subsystem
     * is updating it with information that didn't come from the PLC, such as brake power info.
     * @param ptpState The state of the EL6688 PTP device.
     * @param leapSeconds The current number of leap seconds.
     * @param leapIsValid True if and only if the leap second count is valid.
     * @throws IllegalArgumentException if either the {@code temperature} list or the 
     * {@code tempIsSafe} list is null or is not of size 3.
    */
    public ShutterStatus(
        final int motionProfile,
        final boolean isCalibrated,
        final int smState,
        final Map<ShutterSide, ShutterAxisStatus> axes,
        final boolean isSafetyOn,
        final Map<RTD, Double> temperature,
        final Map<RTD, Boolean> tempIsSafe,
        final boolean brakePowerIsOn,
        final CCSTimeStamp creationTime,
        final PtpDeviceState ptpState,
        final int leapSeconds,
        final boolean leapIsValid
    )
    {
        this.motionProfile = motionProfile;
       // Booleans are not representable in CCS trending so use ints.JIRA LSSTCCSSHUTTER-79.
        this.isCalibrated = isCalibrated ? 1 : 0;
        this.smState = smState;
        this.axstatus = new EnumMap<>(axes);
        this.isSafetyOn = isSafetyOn ? 1 : 0;
        if (temperature == null || temperature.size() != 3) {
            throw new IllegalArgumentException("Temperature map is null or has size != 3");
        }
        if (tempIsSafe == null || tempIsSafe.size() != 3) {
            throw new IllegalArgumentException("Temp-is-safe map is null or has size != 3");
        }
        this.temperature = new TreeMap<>(temperature);
        this.tempIsSafe = new TreeMap<>();
        for (final RTD rtd: RTD.values()) {
            this.tempIsSafe.put(rtd, tempIsSafe.get(rtd) ? 1 : 0);
        }
        this.brakePowerIsOn = brakePowerIsOn ? 1 : 0;
        this.creationTime = creationTime.getTAIDouble();
        this.ptpState = PtpDeviceState.toStateNumber(ptpState);
        this.leapSeconds = leapSeconds;
        this.leapIsValid = leapIsValid ? 1 : 0;
    }

    /** Gets the motion profile in use
     * @return The integer index of the motion profile.
     */
    public int getMotionProfile() {
        return motionProfile;
    }

    /**
     * Is a good calibration in effect?
     * @return true if it is, else false.
     */
    public boolean isCalibrated() {
        return isCalibrated == 1;
    }

    /**
     * Gets the current state for the shutter controller's state machine.
     * @return The integer state index.
     */
    public int getSmState() {
        return smState;
    }

    /**
     * Gets the status info for one of the blade set axes.
     * @param side The side whose status info is wanted.
     * @return The {@code ShutterAxisStatus} object for the given blade set.
     */
    public ShutterAxisStatus getAxisStatus(final ShutterSide side) {
        return axstatus.get(side);
    }

    /**
     * Are safety checks in effect?
     * @return true if they are, else false.
    */
    public boolean isSafetyOn() {
        return isSafetyOn == 1;
    }

    /**
     * Gets the map of three RTD temperatures in degrees C.
     * @return A new map indexed by RTD enumerator.
     */
    public Map<RTD, Double> getTemperature() {
        return new TreeMap<>(temperature);
    }

    /**
     * Gets the map of temperature-is-safe flags for the RTDs.
     * @return A new map indexed by RTD enumerator.
     */
    public Map<RTD, Boolean> getTempIsSafe() {
        final Map<RTD, Boolean> result = new TreeMap<>();
        for (final RTD rtd: RTD.values()) {result.put(rtd, tempIsSafe.get(rtd) == 1);}
        return result;
    }

    /**
     * Gets the subsystems current knowledge of the state of brake power.
     * @return The brake-power-is-on flag.
     */
    public boolean brakePowerIsOn() {
        return this.brakePowerIsOn == 1;
    }
    
    /**
     * Gets the creation time stamp.
     * @return The TAI time as a double.
     */
    public double getCreationTime() {
        return creationTime;
    }
    
    /**
     * Gets the state of the EL6688 PTP device.
     * @return The state (as an enumeration).
     */
    public PtpDeviceState getPtpState() {
        return PtpDeviceState.fromStateNumber(ptpState);
    }
    
    /**
     * Gets the state of the EL6688 PTP device.
     * @return The state (as an integer).
     */
    public int getPtpStateAsInt() {
        return ptpState;
    }
    
    public int getLeapSeconds() {
        return leapSeconds;
    }
    
    public boolean isLeapValid() {
        return leapIsValid != 0 ? true : false;
    }
    

    /**
     * Creates a string of the format 'ShutterStatus{fieldName=value, ..., axstatus[PLUSX]=
 ShutterAxisStatus{...}, axstatus[MINUSX]=ShutterAxisStatus{...}}'.
     * @return The created string.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("ShutterStatus{motionProfile=").append(motionProfile);
        sb.append(", isCalibrated=").append(isCalibrated);
        sb.append(", smState=").append(smState);
        sb.append(", axstatus=").append(axstatus);
        sb.append(", isSafetyOn=").append(isSafetyOn);
        sb.append(", temperature=").append(temperature);
        sb.append(", tempIsSafe=").append(tempIsSafe);
        sb.append(", brakePowerIsOn=").append(brakePowerIsOn);
        sb.append('}');
        return sb.toString();
    }

    /**
     * Holds the status report for a single axis in a {@code ShutterStatus} message. Immutable.
     */
    public final static class ShutterAxisStatus implements Serializable {
        private static final long serialVersionUID = 3L;

        @DataAttributes(description = "The actual blade set axis encoder reading.", units="mm")
        private final double actPos;

        @DataAttributes(description = "The actual blade set velocity from the encoder.", units="mm/s")
        private final double actVel;

        @DataAttributes(description = "The requested blade set acceleration.", units="mm/s2")
        private final double setAcc;

        @DataAttributes(description = "Is the motor enabled? 0 = no, 1 = yes.", units="unitless")
        private final int enabled;

        @DataAttributes(description = "Is the motor brake engaged? 0 = no, 1 = yes.", units="unitless")
        private final int brakeEngaged;

        @DataAttributes(
            description = "Is the blade set at its low motion limit? 0 = no, 1 = yes. ",
            units="unitless"
        )
        private final int lowLimit;

        @DataAttributes(
            description = "Is the blade set at its high motion limit? 0 = no, 1 = yes. ",
            units="unitless"
        )
         private final int highLimit;

        @DataAttributes(description = "Has the axis been calibrated? 0 = no, 1 = yes.", units="unitless")
        private final int isHomed;

        @DataAttributes(description = "Beckhoff error code for last operation. 0 = OK.", units="unitless")
        private final int errorID;

        @DataAttributes(description = "Temperature of the motor controller chip. ", units="Celsius")
        private final double ctrlTemp;

        @DataAttributes(
            description = "Is the temperature of the motor controller in the safe range? 0 = no, 1 = yes.",
            units="unitless"
        )
        private final int hasSafeTemp;

        /** Constructs from scratch.
         *
         * @param actPos See {@link #getActPos() }
         * @param actVel See {@link #getActVel() }
         * @param setAcc See {@link #getSetAcc() }
         * @param enabled See {@link #isEnabled() }
         * @param brakeEngaged See {@link #isBrakeEngaged() }
         * @param lowLimit See {@link #atLowLimit() }
         * @param highLimit See {@link #atHighLimit() }
         * @param isHomed See {@link #isHomed() }
         * @param errorID See {@link #getErrorID() }
         * @param ctrlTemp See {@link #getCtrlTemp() }
         * @param hasSafeTemp See {@link #hasSafeTemp() }
         */
        public ShutterAxisStatus(
            final double actPos,
            final double actVel,
            final double setAcc,
            final boolean enabled,
            final boolean brakeEngaged,
            final boolean lowLimit,
            final boolean highLimit,
            final boolean isHomed,
            final int errorID,
            final double ctrlTemp,
            final boolean hasSafeTemp
        )
        {
            this.actPos = actPos;
            this.actVel = actVel;
            this.setAcc = setAcc;
            this.enabled = enabled ? 1 : 0;
            // Booleans are not representable in CCS trending so use ints.
            this.brakeEngaged = brakeEngaged ? 1 : 0;
            this.lowLimit = lowLimit ? 1 : 0;
            this.highLimit = highLimit ? 1 : 0;
            this.isHomed = isHomed ? 1 : 0;
            this.errorID = errorID;
            this.ctrlTemp = ctrlTemp;
            this.hasSafeTemp = hasSafeTemp ? 1 : 0;
        }

        /**
         * Constructs a string of the form 'ShutterAxisStatus{fieldname=value, ...}'
         * @return The string.
         */
        @Override
        public String toString() {
            return "AxisStatus{" + "actPos=" + actPos + ", actVel=" + actVel + ", setAcc="
                    + setAcc + ", enabled=" + enabled + ", brakeEngaged=" + brakeEngaged
                    + ", lowLimit=" + lowLimit + ", highLimit=" + highLimit + ", isHomed="
                    + isHomed + ", errorID=" + errorID + ", ctrlTemp=" + ctrlTemp +
                    ", hasSafeTemp=" + hasSafeTemp + '}';
        }

        /**
         * Gets the actual position of the axis.
         * @return The position in mm.
         */
        public double getActPos() {
            return actPos;
        }

        /**
         * Gets the actual velocity of the axis.
         * @return The velocity in mm/sec.
         */
        public double getActVel() {
            return actVel;
        }

        /**
         * Gets the "set", or commanded, acceleration of the axis.
         * @return The acceleration in mm/sec/sec.
         */
        public double getSetAcc() {
            return setAcc;
        }

        /**
         * Is the axis enabled for motion?
         * @return true if it is, else false.
         */
        public boolean isEnabled() {
            return enabled == 1;
        }

        /**
         * Is the brake set on this axis?
         * @return true if it is, else false.
         */
        public boolean isBrakeEngaged() {
            return brakeEngaged == 1;
        }

        /**
         * Is the axis at the the low limit switch?
         * @return true if it is, else false.
         */
        public boolean atLowLimit() {
            return lowLimit == 1;
        }

        /**
         * Is the axis at the high limit switch?
         * @return true if it is, else false.
         */
        public boolean atHighLimit() {
            return highLimit == 1;
        }

        /**
         * Has the homing operation been performed on this axis since the last controller reset?
         * @return true if it has, else false.
         */
        public boolean isHomed() {
            return isHomed == 1;
        }

        /**
         * Gets the error code for the axis.
         * @return The error code set by the Beckhoff motion control task.
         */
        public int getErrorID() {
            return errorID;
        }

        /**
         * Gets the internal temperature of the axis motor controller.
         * @return The temperature in degrees Celsius.
         */
        public double getCtrlTemp() {
            return ctrlTemp;
        }
        
        /**
         * Is the controller temperature in the safe operating range?
         * @return true if it is, else false.
         */
        public boolean hasSafeTemp() {
            return hasSafeTemp == 1;
        }
    }
}
