package org.lsst.ccs.subsystem.shutter.sim;

import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * Samples a motion profile function to produce a stream of PredictedPosition objects.
 * <p>
 * The physical motion profile is sampled at regular intervals, each sampling being used
 * to make a PredictedPosition object. The starting time is added to the time-since-move-start
 * used by the profile. The sampling is done once but can be retrieved as many times as needed.
 *
 * <p> Instances are immutable.
 * 
 * @author tether
 *
 */

public final class PredictedTrajectory {
    
    /**
     * Samples the given physical motion profile.
     * @param startPosition the  normalized start position (dimensionless in [0, 1])
     * @param startTime The absolute time of motion start.
     * @param dt the time between samples in seconds
     * @param numSamples the number of positions to take
     * @param profile the motion profile to be sampled
     */
    public PredictedTrajectory(
            final double startPosition,
            final CCSTimeStamp startTime,
            final Duration dt,
            final int numSamples,
            final MotionProfile profile)
    {
        final Instant t0 = startTime.getUTCInstant();
        final double totalTime = (numSamples - 1) * dt.toNanos() * 1e-9;
        positions =
                Stream
                .iterate(t0, t -> t.plus(dt))
                .limit(numSamples)
                .map(t ->  {
                    // TODO Time stamp construction using nanosecond-precision Instant or equiv.
                    // Have to be careful to record the exact time passed to the motion profile,
                    // including any rounding or truncation done by currentTimeFromMillis().
                    final CCSTimeStamp stamp = CCSTimeStamp.currentTimeFromMillis(t.toEpochMilli());
                    final double s = Duration.between(t0, stamp.getUTCInstant()).toNanos() * 1e-9;
                    return new PredictedPosition(
                        stamp,
                        profile.acceleration(s),
                        profile.velocity(s),
                        startPosition + profile.distance(s));
                })
                .collect(Collectors.toList());
        }
    
    /**
     * Returns the time-ordered list of predicted positions (and accelerations, etc.).
     * @return the position list (unmodifiable).
     */
    public List<PredictedPosition> getPositions() {return positions;}

    /** The samples taken from the motion profile. */
    final private List<PredictedPosition> positions;
}
