package org.lsst.ccs.subsystem.shutter.status;

import java.io.Serializable;
import java.util.EnumMap;
import java.util.Map;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;

/**
 * Holds a ShutterStatus message sent from the PLC. Immutable.
 * @author tether
 */
public final class ShutterStatus implements Serializable {
    private static final long serialVersionUID = 1L;

    private final int motionProfile;

    private final boolean isCalibrated;

    private final int smState;

    private final Map<ShutterSide, AxisStatus> axstatus;

    /**
     * Constructs from scratch.
     * @param motionProfile The index number of the motion profile in effect.
     * @param isCalibrated Is the shutter properly calibrated?
     * @param smState The current state of the PLC state machine.
     * @param axes The source of the {@code AxisStatus} values to be stored in this message.
     */
    public ShutterStatus(
        final int motionProfile,
        final boolean isCalibrated,
        final int smState,
        final Map<ShutterSide, AxisStatus> axes
    )
    {
        this.motionProfile = motionProfile;
        this.isCalibrated = isCalibrated;
        this.smState = smState;
        this.axstatus = new EnumMap<>(axes);
    }

    /** Gets the motion profile in use
     * @return The integer index of the motion profile.
     */
    public int getMotionProfile() {
        return motionProfile;
    }

    /**
     * Is a good calibration in effect?
     * @return true if it is, else false.
     */
    public boolean isCalibrated() {
        return isCalibrated;
    }

    /**
     * Gets the current state for the shutter controller's state machine.
     * @return The integer state index.
     */
    public int getSmState() {
        return smState;
    }

    /**
     * Gets the status info for one of the blade set axes.
     * @param side The side whose status info is wanted.
     * @return The {@code AxisStatus} object for the given blade set.
     */
    public AxisStatus getAxisStatus(final ShutterSide side) {
        return axstatus.get(side);
    }

    /**
     * Creates a string of the format 'ShutterStatus{fieldName=value, ..., axstatus[PLUSX]=
     * AxisStatus{...}, axstatus[MINUSX]=AxisStatus{...}}'.
     * @return The created string.
     */
    @Override
    public String toString() {
        final StringBuilder buff = new StringBuilder(
            "ShutterStatus{" +
            "motionProfile=" +
            motionProfile +
            ", isCalibrated=" +
            isCalibrated +
            ", smState=" +
            smState);
        for (ShutterSide side: ShutterSide.values()) {
            buff.append(", axstatus[");
            buff.append(side);
            buff.append("]=");
            buff.append(getAxisStatus(side));
        }
        buff.append("}");
        return buff.toString();
    }

    /**
     * Holds the status report for a single axis in a {@code ShutterStatus} message. Immutable.
     */
    public final static class AxisStatus implements Serializable {
        private static final long serialVersionUID = 1L;
        private final double actPos;
        private final double actVel;
        private final double setAcc;
        private final boolean enabled;
        private final boolean brakeEngaged;
        private final boolean lowLimit;
        private final boolean highLimit;
        private final boolean isHomed;
        private final int errorID;
        private final double motorTemp;

        /** Constructs from scratch.
         * 
         * @param actPos See {@link #getActPos() }
         * @param actVel See {@link #getActVel() }
         * @param setAcc See {@link #getSetAcc() }
         * @param enabled See {@link #isEnabled() }
         * @param brakeEngaged See {@link #isBrakeEngaged() }
         * @param lowLimit See {@link #atLowLimit() }
         * @param highLimit See {@link #atHighLimit() }
         * @param isHomed See {@link #isHomed() }
         * @param errorID See {@link #getErrorID() }
         * @param motorTemp See {@link #getMotorTemp() }
         */
        public AxisStatus(
            final double actPos,
            final double actVel,
            final double setAcc,
            final boolean enabled,
            final boolean brakeEngaged,
            final boolean lowLimit,
            final boolean highLimit,
            final boolean isHomed,
            final int errorID,
            final double motorTemp
        )
        {
            this.actPos = actPos;
            this.actVel = actVel;
            this.setAcc = setAcc;
            this.enabled = enabled;
            this.brakeEngaged = brakeEngaged;
            this.lowLimit = lowLimit;
            this.highLimit = highLimit;
            this.isHomed = isHomed;
            this.errorID = errorID;
            this.motorTemp = motorTemp;
        }

        /**
         * Constructs a string of the form 'AxisStatus{fieldname=value, ...}'
         * @return The string.
         */
        @Override
        public String toString() {
            return "AxisStatus{" + "actPos=" + actPos + ", actVel=" + actVel + ", setAcc=" + setAcc +
                   ", enabled=" + enabled + ", brakeEngaged=" + brakeEngaged + ", lowLimit=" + lowLimit +
                   ", highLimit=" + highLimit + ", isHomed=" + isHomed + ", errorID=" + errorID +
                   ", motorTemp=" + motorTemp + '}';
        }

        /**
         * Gets the actual position of the axis.
         * @return The position in mm.
         */
        public double getActPos() {
            return actPos;
        }

        /**
         * Gets the actual velocity of the axis.
         * @return The velocity in mm/sec.
         */
        public double getActVel() {
            return actVel;
        }

        /**
         * Gets the "set", or commanded, acceleration of the axis.
         * @return The acceleration in mm/sec/sec.
         */
        public double getSetAcc() {
            return setAcc;
        }

        /**
         * Is the axis enabled for motion?
         * @return true if it is, else false.
         */
        public boolean isEnabled() {
            return enabled;
        }

        /**
         * Is the brake set on this axis?
         * @return true if it is, else false.
         */
        public boolean isBrakeEngaged() {
            return brakeEngaged;
        }

        /**
         * Is the axis at the the low limit switch?
         * @return true if it is, else false.
         */
        public boolean atLowLimit() {
            return lowLimit;
        }

        /**
         * Is the axis at the high limit switch?
         * @return true if it is, else false.
         */
        public boolean atHighLimit() {
            return highLimit;
        }

        /**
         * Has the homing operation been performed on this axis since the last controller reset?
         * @return true if it has, else false.
         */
        public boolean isHomed() {
            return isHomed;
        }

        /**
         * Gets the error code for the axis.
         * @return The error code set by the Beckhoff motion control task.
         */
        public int getErrorID() {
            return errorID;
        }

        /**
         * Gets the axis motor temperature reading.
         * @return The temperature in degrees Celsius.
         */
        public double getMotorTemp() {
            return motorTemp;
        }
    }
}
