package org.lsst.ccs.subsystem.shutter.status;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.subsystem.shutter.common.RTDName;
import org.lsst.ccs.subsystem.shutter.common.ShutterSide;

/**
 * Holds a ShutterStatus message sent from the PLC. Immutable.
 * @author tether
 */
public final class ShutterStatus implements Serializable {
    private static final long serialVersionUID = 3L;

    private final int motionProfile;

    private final int isCalibrated;

    private final int smState;

    private final Map<ShutterSide, AxisStatus> axstatus;

    private final int isSafetyOn;

    private final Map<RTDName, Integer> temperature;

    /**
     * Constructs from scratch.
     * @param motionProfile The index number of the motion profile in effect.
     * @param isCalibrated Is the shutter properly calibrated?
     * @param smState The current state of the PLC state machine.
     * @param axes The source of the {@code AxisStatus} values to be stored in this message.
     * @param isSafetyOn Are shutter hardware safety checks enabled in the PLC?
     * @param temperature The list of three raw RTD values.
     * @throws IllegalArgumentException if the {@code temperature} list is null or is not of size 3.
     */
    public ShutterStatus(
        final int motionProfile,
        final boolean isCalibrated,
        final int smState,
        final Map<ShutterSide, AxisStatus> axes,
        final boolean isSafetyOn,
        final List<Integer> temperature
    )
    {
        this.motionProfile = motionProfile;
       // Booleans are not representable in CCS trending so use ints.JIRA LSSTCCSSHUTTER-79.
        this.isCalibrated = isCalibrated ? 1 : 0;
        this.smState = smState;
        this.axstatus = new EnumMap<>(axes);
        this.isSafetyOn = isSafetyOn ? 1 : 0;
        if (temperature == null || temperature.size() != 3) {
            throw new IllegalArgumentException("temperature list is null or has size != 3");
        }
        this.temperature = new HashMap<>();
        for (final RTDName rtd: RTDName.values()) { // JIRA LSSTCCSSHUTTER-125.
            this.temperature.put(rtd, temperature.get(rtd.ordinal()));
        }
    }

    /** Gets the motion profile in use
     * @return The integer index of the motion profile.
     */
    public int getMotionProfile() {
        return motionProfile;
    }

    /**
     * Is a good calibration in effect?
     * @return true if it is, else false.
     */
    public boolean isCalibrated() {
        return isCalibrated == 1;
    }

    /**
     * Gets the current state for the shutter controller's state machine.
     * @return The integer state index.
     */
    public int getSmState() {
        return smState;
    }

    /**
     * Gets the status info for one of the blade set axes.
     * @param side The side whose status info is wanted.
     * @return The {@code AxisStatus} object for the given blade set.
     */
    public AxisStatus getAxisStatus(final ShutterSide side) {
        return axstatus.get(side);
    }

    /**
     * Are safety checks in effect?
     * @return true if they are, else false.
    */
    public boolean isSafetyOn() {
        return isSafetyOn == 1;
    }

    /**
     * Gets the three RTD raw values.
     * @return A new list containing the integer raw values.
     */
    public List<Integer> getTemperature() {
        final List<Integer> temps = new ArrayList<>();
        for (final RTDName rtd: RTDName.values()) {
            temps.add(temperature.get(rtd));
        }
        return temps;
    }

    /**
     * Creates a string of the format 'ShutterStatus{fieldName=value, ..., axstatus[PLUSX]=
     * AxisStatus{...}, axstatus[MINUSX]=AxisStatus{...}}'.
     * @return The created string.
     */
    @Override
    public String toString() {
        final StringBuilder buff = new StringBuilder(
            "ShutterStatus{" +
            "motionProfile=" +
            motionProfile +
            ", isCalibrated=" +
            isCalibrated +
            ", smState=" +
            smState +
            ", isSafetyOn=" +
            isSafetyOn +
            ", temperature=[" +
            temperature.get(0) +
            ", " +
            temperature.get(1) +
            ", " +
            temperature.get(2) + "]");
        for (ShutterSide side: ShutterSide.values()) {
            buff.append(", axstatus[");
            buff.append(side);
            buff.append("]=");
            buff.append(getAxisStatus(side));
        }
        buff.append("}");
        return buff.toString();
    }

    /**
     * Holds the status report for a single axis in a {@code ShutterStatus} message. Immutable.
     */
    public final static class AxisStatus implements Serializable {
        private static final long serialVersionUID = 1L;
        private final double actPos;
        private final double actVel;
        private final double setAcc;
        private final int enabled;
        private final int brakeEngaged;
        private final int lowLimit;
        private final int highLimit;
        private final int isHomed;
        private final int errorID;
        private final double motorTemp;
        private final int hasSafeTemp;

        /** Constructs from scratch.
         *
         * @param actPos See {@link #getActPos() }
         * @param actVel See {@link #getActVel() }
         * @param setAcc See {@link #getSetAcc() }
         * @param enabled See {@link #isEnabled() }
         * @param brakeEngaged See {@link #isBrakeEngaged() }
         * @param lowLimit See {@link #atLowLimit() }
         * @param highLimit See {@link #atHighLimit() }
         * @param isHomed See {@link #isHomed() }
         * @param errorID See {@link #getErrorID() }
         * @param motorTemp See {@link #getMotorTemp() }
         * @param hasSafeTemp See {@link #hasSafeTemp() }
         */
        public AxisStatus(
            final double actPos,
            final double actVel,
            final double setAcc,
            final boolean enabled,
            final boolean brakeEngaged,
            final boolean lowLimit,
            final boolean highLimit,
            final boolean isHomed,
            final int errorID,
            final double motorTemp,
            final boolean hasSafeTemp
        )
        {
            this.actPos = actPos;
            this.actVel = actVel;
            this.setAcc = setAcc;
            this.enabled = enabled ? 1 : 0;
            // Booleans are not representable in CCS trending so use ints.
            this.brakeEngaged = brakeEngaged ? 1 : 0;
            this.lowLimit = lowLimit ? 1 : 0;
            this.highLimit = highLimit ? 1 : 0;
            this.isHomed = isHomed ? 1 : 0;
            this.errorID = errorID;
            this.motorTemp = motorTemp;
            this.hasSafeTemp = hasSafeTemp ? 1 : 0;
        }

        /**
         * Constructs a string of the form 'AxisStatus{fieldname=value, ...}'
         * @return The string.
         */
        @Override
        public String toString() {
            return "AxisStatus{" + "actPos=" + actPos + ", actVel=" + actVel + ", setAcc="
                    + setAcc + ", enabled=" + enabled + ", brakeEngaged=" + brakeEngaged
                    + ", lowLimit=" + lowLimit + ", highLimit=" + highLimit + ", isHomed="
                    + isHomed + ", errorID=" + errorID + ", motorTemp=" + motorTemp +
                    ", hasSafeTemp=" + hasSafeTemp + '}';
        }

        /**
         * Gets the actual position of the axis.
         * @return The position in mm.
         */
        public double getActPos() {
            return actPos;
        }

        /**
         * Gets the actual velocity of the axis.
         * @return The velocity in mm/sec.
         */
        public double getActVel() {
            return actVel;
        }

        /**
         * Gets the "set", or commanded, acceleration of the axis.
         * @return The acceleration in mm/sec/sec.
         */
        public double getSetAcc() {
            return setAcc;
        }

        /**
         * Is the axis enabled for motion?
         * @return true if it is, else false.
         */
        public boolean isEnabled() {
            return enabled == 1;
        }

        /**
         * Is the brake set on this axis?
         * @return true if it is, else false.
         */
        public boolean isBrakeEngaged() {
            return brakeEngaged == 1;
        }

        /**
         * Is the axis at the the low limit switch?
         * @return true if it is, else false.
         */
        public boolean atLowLimit() {
            return lowLimit == 1;
        }

        /**
         * Is the axis at the high limit switch?
         * @return true if it is, else false.
         */
        public boolean atHighLimit() {
            return highLimit == 1;
        }

        /**
         * Has the homing operation been performed on this axis since the last controller reset?
         * @return true if it has, else false.
         */
        public boolean isHomed() {
            return isHomed == 1;
        }

        /**
         * Gets the error code for the axis.
         * @return The error code set by the Beckhoff motion control task.
         */
        public int getErrorID() {
            return errorID;
        }

        /**
         * Gets the axis motor temperature reading.
         * @return The temperature in degrees Celsius.
         */
        public double getMotorTemp() {
            return motorTemp;
        }
        
        /**
         * Is the motor temperature in the safe operating range?
         * @return true if it is, else false.
         */
        public boolean hasSafeTemp() {
            return hasSafeTemp == 1;
        }
    }
}
