package org.lsst.ccs.subsystems.shutter.common;

import java.util.Collections;
import java.util.List;

/**
 ***************************************************************************
 **
 **  Class to hold blade calibration data. Immutable.
 **
 **  @author Owen Saxton
 **  @author tether
 **
 ***************************************************************************
 */
public final class BladeSetCalibration {

    private final int         status;    // Calibration status
    private final List<Item>  data;      // Calibration data
    
    /**
     * Constructs from a status value and a list of Items.
     * @param status the status of the calibration.
     * See {@link org.lsst.ccs.subsystems.shutter.driver.MovementStatus}
     * @param data the list of calibration items
     */
    public BladeSetCalibration(int status, List<Item> data) {
        this.status = status;
        this.data = data;
    }

   /**
    ***************************************************************************
    **
    **  Inner class to hold a single blade calibration data item, corresponding
    **  to a change of state, or transition, of a Hall sensor. Immutable.
    **
    ***************************************************************************
    */
    public final static class Item {

        final int      sensor;     // Sensor ID
        final int      index;      // Instance number of sensor open-close cycle
        final double   position;   // BladeSet position
        final boolean  open;       // True if switch opened; false if closed
        final boolean  reverse;    // True if reverse move; false if forward


        /**
         * Constructs a single calibration item from the field values.
         * @param sensor the Hall sensor ID number (0, 1, 2, ...)
         * @param index  used to number consecutive transition pairs (0, 1, 2, ...) for
         * the same sensor. A new pair starts when the state changes to open.
         * @param position the absolute position of the sensor involved (mm)
         * @param open the state of the sensor after the transition. true when a magnet is not present
         * @param reverse true if the blade set was being retracted when the transition occurred
         */
        public Item(int sensor, int index, double position, boolean open,
                    boolean reverse)
        {
            this.sensor   = sensor;
            this.index    = index;
            this.position = position;
            this.open     = open;
            this.reverse  = reverse;
        }

        /**
         * Gets the Hall sensor ID number.
         * @return the ID number
         */
        public int getSensor()
        {
            return sensor;
        }

        /**
         * Gets the index used to distinguish members of a set of consecutive transitions.
         * @return the index
         */
        public int getIndex()
        {
            return index;
        }

        /**
         * Gets the absolute position of the transition.
         * @return the position
         */
        public double getPosition()
        {
            return position;
        }

        /**
         * Gets the state of the Hall sensor after the transition.
         * @return the state
         */
        public boolean isOpen()
        {
            return open;
        }

        /**
         * Gets the direction-of-motion flag for the transition.
         * @return the direction flag
         */
        public boolean isReverse()
        {
            return reverse;
        }

    }

    /**
     * Gets the status value for the calibration.
     * @return the status value
     */
    public int getStatus()
    {
        return status;
    }

    /**
     * Gets the list of transition items.
     * @return the list (unmodifiable)
     */
    public List<BladeSetCalibration.Item> getData()
    {
        return Collections.unmodifiableList(data);
    }

}
